/*======================================================================
 * Dungeon Level Map Generation Library.
 * Generates small generic dungeon-like level maps.
 *
 * Released into the Public Domain by Damian Gareth Walker, 2020.
 * Created: 30-Jul-2020.
 *
 * This is the demonstration module.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "levelmap.h"

/*----------------------------------------------------------------------
 * Level 1 Functions.
 */

/**
 * Display the main menu and input an option.
 * @returns the option chosen.
 */
static int getmenuoption (void)
{
    int option;
    printf ("Menu:\n");
    printf ("0: Generate.\n");
    printf ("1: Set width.\n");
    printf ("2: Set height.\n");
    printf ("3: Set corridor chance.\n");
    printf ("4: Set combined chamber chance.\n");
    printf ("5: Set minimum cell size.\n");
    while (! scanf ("%d", &option));
    return option;
}

/**
 * Input a width for the map
 * @param map is the map.
 */
static void setmapwidth (LevelMap *map)
{
    int width;
    printf ("Width? ");
    if (! scanf ("%d", &width)) {
	printf ("Not a valid value.\n");
	return;
    }
    if (map->setwidth (map, width))
	printf ("Value accepted.\n");
    else
	printf ("Value rejected.\n");
}

/**
 * Input a height for the map
 * @param map is the map.
 */
static void setmapheight (LevelMap *map)
{
    int height;
    printf ("Height? ");
    if (! scanf ("%d", &height)) {
	printf ("Not a valid value.\n");
	return;
    }
    if (map->setheight (map, height))
	printf ("Value accepted.\n");
    else
	printf ("Value rejected.\n");
}

/**
 * Input a chance for corridors to form.
 * @param map is the map.
 */
static void setmapcorridorchance (LevelMap *map)
{
    int chance;
    printf ("Chance for corridors (0-100)? ");
    if (! scanf ("%d", &chance)) {
	printf ("Not a valid value.\n");
	return;
    }
    if (map->setcorridor (chance))
	printf ("Value accepted.\n");
    else
	printf ("Value rejected.\n");
}

/**
 * Input a chance for corridors to form.
 * @param map is the map.
 */
static void setmapcombinedchamberchance (LevelMap *map)
{
    int chance;
    printf ("Chance for combined chambers (0-100)? ");
    if (! scanf ("%d", &chance)) {
	printf ("Not a valid value.\n");
	return;
    }
    if (map->setcombined (chance))
	printf ("Value accepted.\n");
    else
	printf ("Value rejected.\n");
}

/**
 * Input a chance for corridors to form.
 * @param map is the map.
 */
static void setmapcellsize (LevelMap *map)
{
    int minimum;
    printf ("Minimum cell size (1+)? ");
    if (! scanf ("%d", &minimum)) {
	printf ("Not a valid value.\n");
	return;
    }
    if (map->setcellsize (minimum))
	printf ("Value accepted.\n");
    else
	printf ("Value rejected.\n");
}

/**
 * Generate and display the map.
 * @param map is the map.
 */
static void generatelevelmap (LevelMap *map)
{
    int x,
	y;
    map->generate (map);
    map->openvoids (map);
    for (y = 0; y < map->height; ++y) {
	for (x = 0; x < map->width; ++x)
	    switch (map->getblock (map, x, y) & 0xf0) {
	    case LEVELMAP_VOID:
		printf ("  ");
		break;
	    case LEVELMAP_OPEN:
		printf (" .");
		break;
	    case LEVELMAP_WALL:
		printf (" #");
		break;
	    case LEVELMAP_DOOR:
		printf (" /");
		break;
	    default:
		printf ("%02x", map->getblock (map, x, y));
	    }
	putchar ('\n');
    }
}

/*----------------------------------------------------------------------
 * Top Level Function.
 */

/**
 * Main control function.
 * @param argc is the command line argument count.
 * @param argv is the array of command line arguments.
 * @returns 0 on success.
 */
int main (int argc, char **argv)
{
    /* local variables */
    LevelMap *map;
    int option;

    /* initialise */
    srand (time (NULL));
    map = new_LevelMap ();

    /* repeat menu till user is ready to generate */
    do {
	switch ((option = getmenuoption ())) {
	case 1:
	    setmapwidth (map);
	    break;
	case 2:
	    setmapheight (map);
	    break;
	case 3:
	    setmapcorridorchance (map);
	    break;
	case 4:
	    setmapcombinedchamberchance (map);
	    break;
	case 5:
	    setmapcellsize (map);
	    break;
	}
    } while (option);

    /* generate map, destroy and return */
    generatelevelmap (map);
    map->destroy (map);
    return 1;
}
