/*======================================================================
 * Dungeon Level Map Generation Library.
 * Generates small generic dungeon-like level maps.
 *
 * Released into the Public Domain by Damian Gareth Walker, 2020.
 * Created: 29-Jul-2020.
 *
 * This is the main module.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "levelmap.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/**
 * @struct cell_data is a representation of cell divisions.
 * When generating a level, it is first divided into cells.
 * These are approximately x/4 * y/4 blocks each, varying randomly.
 * This data is temporary and is discarded when the map is generated.
 * It is never presented to the outside world.
 */
typedef struct cell_data CellData;
struct cell_data {

    /** @var columns is the width of the grid in cells. */
    int columns;
    
    /** @var rows is the height of the grid in cells. */
    int rows;

    /** @var cellwidths are the widths of cell columns. */
    int *widths;

    /** @var cellheights are the heights of the cell rows. */
    int *heights;

    /**
     * @var celltypes are the types of the cells.
     * 0 is a corridor.
     * 1 is a room.
     */
    int *types;

    /**
     * @var celllinks are the links between cells (bitfield).
     * 1 means a link to the east.
     * 2 means a link to the south.
     * 4 means a link to the west.
     * 8 means a link to the north.
     */
    int *links;
};

/**
 * @struct cell_geometry holds various geometric data of a cell.
 * It is used for generating chambers and corridors.
 */
typedef struct cell_geometry CellGeometry;
struct cell_geometry {

    /** @var col is the column number on the cell grid */
    int col;

    /** @var row is the row number on the cell grid */
    int row;

    /** @var tx is the x position of the top left inside corner. */
    int tx;

    /** @var ty is the y position of the top left inside corner. */
    int ty;

    /** @var mx is the x position at the middle. */
    int mx;

    /** @var my is the y position at the middle. */
    int my;

    /** @var bx is the x position of the bottom right inside corner. */
    int bx;

    /** @var by is the y position of the bottom right inside corner. */
    int by;
};

/*----------------------------------------------------------------------
 * File Level Variables.
 */

/** @var corridorchance is the % chance of a cell being a corridor */
static int corridorchance = 50;

/** @var combinedchamberchance is the % chance of chambers being combined */
static int combinedchamberchance = 50;

/** @var mincellsize is the minimum cell width/height, excluding wall */
static int mincellsize = 2;

/*----------------------------------------------------------------------
 * Level 3 Functions.
 */

/**
 * Create a cell order array and fill it with consecutive values.
 * @param cells is the cell data, for dimensions.
 * @returns a new int array with the cell order.
 */
static int *initcellorder (CellData *cells)
{
    /* local variables */
    int numberofcells, /* number of cells in the grid */
	*order, /* order of cells */
	c; /* cell counter */

    /* allocate memory */
    numberofcells = cells->columns * cells->rows;
    if (! (order = malloc (sizeof (int) * numberofcells)))
	return NULL;

    /* fill the buffer with values 0..n */
    for (c = 0; c < numberofcells; ++c)
	order[c] = c;

    /* return the new buffer */
    return order;
}

/**
 * Scramble the order of the cells in the grid.
 * @param cells is the cell data from which we get dimensions.
 * @param cellorder is the cell order array to scramble.
 */
static void scramblecells (CellData *cells, int *cellorder)
{
    /* local variables */
    int numberofcells, /* number of cells in the grid */
	c, /* cell counter */
	c2; /* cell to swap with */

    /* calculate number of cells */
    numberofcells = cells->columns * cells->rows;

    /* randomise their order */
    for (c = 0; c < numberofcells; ++c) {
	c2 = rand () % numberofcells;
	if (c != c2) {
	    cellorder[c] ^= cellorder[c2];
	    cellorder[c2] ^= cellorder[c];
	    cellorder[c] ^= cellorder[c2];
	}
    }
}

/**
 * Link each cell to the next one in the random cell order.
 * @param cells is the cell data, including these links.
 * @param cellorder is the randomised cell order.
 */
static void generatecellpaths (CellData *cells, int *cellorder)
{
    /* local variables */
    int numberofcells, /* number of cells in the grid */
	c, /* cell counter */
	cc, /* column of current cell */
	cr, /* row of current cell */
	dc, /* column of destination cell */
	dr, /* row of destination cell */
	h, /* horizontal direction */
	v, /* vertical direction */
	ccell, /* index of current cell in cells->links */
	ncell; /* index of next cell in cells->links */

    /* outer loop */
    numberofcells = cells->columns * cells->rows;
    for (c = 1; c < numberofcells; ++c) {

	/* initialise current and end positions, and directions */
	cc = cellorder[c - 1] % cells->columns;
	cr = cellorder[c - 1] / cells->columns;
	dc = cellorder[c] % cells->columns;
	dr = cellorder[c] / cells->columns;
	h = (cc < dc) - (cc > dc);
	v = (cr < dr) - (cr > dr);

	/* join cells along the horizontal path */
	while (cc != dc) {
	    ccell = cc + cells->columns * cr;
	    ncell = ccell + h;
	    cells->links[ccell] |= 4 * (h < 0) + (h > 0);
	    cells->links[ncell] |= (h < 0) + 4 * (h > 0);
	    cc += h;
	}

	/* join cells along the vertical path */
	while (cr != dr) {
	    ccell = cc + cells->columns * cr;
	    ncell = ccell + v * cells->columns;
	    cells->links[ccell] |= 8 * (v < 0) + 2 * (v > 0);
	    cells->links[ncell] |= 2 * (v < 0) + 8 * (v > 0);
	    cr += v;
	}
    }
}

/**
 * Attempt to combine a chamber with another one above or to the left.
 * @param cells is the cell data.
 * @param cells is the cell number in question.
 */
static void combinechambers (CellData *cells, int cell)
{
    /* local variables */
    int dir, /* direction to combine 0=west 1=north */
	acell; /* adjacent cell index */

    /* reject cells that cannot be combined */
    if (cell < cells->columns && cell % cells->columns == 0)
	return;
    else if (cells->links[cell] && 0xc == 0)
	return;

    /* decide on direction of combine */
    if (cell < cells->columns)
	dir = 0;
    else if (cell % cells->columns == 0)
	dir = 1;
    else if (cells->types[cell - 1] == 0)
	dir = 1;
    else if (cells->types[cell - cells->columns] == 0)
	dir = 0;
    else
	dir = rand () % 2;

    /* return if we didn't find a room to combine with */
    if (dir == 0 && cell % cells->columns == 0)
	return;
    else if (dir == 1 && cell < cells->columns)
	return;

    /* return if adjacent cell is not a chamber */
    acell = cell - (dir ? cells->columns : 1);
    if (! cells->types[acell])
	return;

    /* turn both cells into combined chambers */
    cells->types[cell] = cells->types[acell] = 2;
}

/**
 * Calculate the geometry of a cell.
 * @param cells is the cell data for the level.
 * @param cell is the cell number we're interested in.
 * @returns the cell geometry of that cell.
 */
static CellGeometry calculatecellgeometry (CellData *cells, int cell)
{
    /* local variables */
    CellGeometry g; /* the geometry to return */
    int c; /* general counter variable */

    /* calculate cell position */
    g.col = cell % cells->columns;
    g.row = cell / cells->columns;
    
    /* calculate the X positions */
    g.tx = 1;
    for (c = 0; c < g.col; ++c)
	g.tx += cells->widths[c];
    g.mx = g.tx + (cells->widths[g.col] - 1) / 2;
    g.bx = g.tx + cells->widths[g.col] - 2;

    /* calculate the Y positions */
    g.ty = 1;
    for (c = 0; c < g.row; ++c)
	g.ty += cells->heights[c];
    g.my = g.ty + (cells->heights[g.row] - 1) / 2;
    g.by = g.ty + cells->heights[g.row] - 2;

    /* return the geometry */
    return g;
}

/*----------------------------------------------------------------------
 * Level 2 Functions.
 */

/**
 * Generate the dimensions of the cell grid, its rows and its columns.
 * Allocate memory for those cells at the same time.
 * @param map is the map to calculate dimensions for.
 * @param cells is the cell data to store.
 */
static void setcelldimensions (LevelMap *map, CellData *cells)
{
    /* local variables */
    int r, /* row counter */
	c; /* column counter */

    /* set the overall grid dimensions */
    cells->columns = (map->width - 1) / 4;
    cells->rows = (map->height - 1) / 4;

    /* allocate memory now */
    cells->widths = malloc (cells->columns * sizeof (int));
    cells->heights = malloc (cells->rows * sizeof (int));
    cells->types = malloc (cells->columns * cells->rows * sizeof (int));
    cells->links = malloc (cells->columns * cells->rows * sizeof (int));

    /* calculate the row and column sizes */
    for (c = 0; c < cells->columns; ++c)
	cells->widths[c] = 4;
    cells->widths[rand () % cells->columns]
	= 4 + (map->width - 1) % 4;
    for (r = 0; r < cells->rows; ++r)
	cells->heights[r] = 4;
    cells->heights[rand () % cells->rows]
	= 4 + (map->height - 1) % 4;

    /* randomise the rows and columns a little */
    while (rand () % 3) {
	c = rand () % (cells->columns - 1);
	if (cells->widths[c] > mincellsize + 1) {
	    --cells->widths[c];
	    ++cells->widths[c + 1];
	}
    }
    while (rand () % 3) {
	r = rand () % (cells->rows - 1);
	if (cells->heights[r] > mincellsize + 1) {
	    --cells->heights[r];
	    ++cells->heights[r + 1];
	}
    }

    /* initialise the cell types and links while we're at it */
    for (c = 0; c < cells->columns; ++c)
	for (r = 0; r < cells->rows; ++r) {
	    cells->types[c + cells->columns * r] = 0;
	    cells->links[c + cells->columns * r] = 0;
	}
}

/**
 * Randomly connect the cells together.
 * @param cells is the complete data on cells.
 */
static int connectcells (CellData *cells)
{
    int *cellorder; /* temporary cell order array */
    if (! (cellorder = initcellorder (cells)))
	return 0;
    scramblecells (cells, cellorder);
    generatecellpaths (cells, cellorder);
    free (cellorder);
    return 1;
}

/**
 * Open up some of the cells as chambers.
 * @param cells is the complete data on cells.
 */
static void openchambers (CellData *cells)
{
    int c; /* cell counter */
    for (c = 0; c < cells->columns * cells->rows; ++c)
	if ((cells->types[c] =
	    (cells->links[c] == 1 ||
	     cells->links[c] == 2 ||
	     cells->links[c] == 4 ||
	     cells->links[c] == 8 ||
	     rand () % 100 >= corridorchance)) &&
	    rand () % 100 < combinedchamberchance)
	    combinechambers (cells, c);
}

/**
 * Generate a corridor on the map from a cell.
 * @param map is the map to modify.
 * @param cells is the cell network.
 * @param cell is the cell number.
 */
static void generatecorridor (LevelMap *map, CellData *cells, int cell)
{
    /* local variables */
    CellGeometry g; /* geometry of the cell */
    int c, /* general counter */
	block; /* block location */

    /* get the geometry */
    g = calculatecellgeometry (cells, cell);

    /* open up a horizontal path */
    for (c = g.tx - 1; c <= g.bx + 1; ++c) {
	block = c + map->width * g.my;
	if (c <= g.mx && (cells->links[cell] & 4) &&
	    map->blocks[block] != LEVELMAP_DOOR)
	    map->blocks[block] = LEVELMAP_OPEN;
	else if (c >= g.mx && (cells->links[cell] & 1) &&
	    map->blocks[block] != LEVELMAP_DOOR)
	    map->blocks[block] = LEVELMAP_OPEN;
    }

    /* open up a vertical path */
    for (c = g.ty - 1; c <= g.by + 1; ++c) {
	block = g.mx + map->width * c;
	if (c <= g.my && (cells->links[cell] & 8) &&
	    map->blocks[block] != LEVELMAP_DOOR)
	    map->blocks[block] = LEVELMAP_OPEN;
	else if (c >= g.my && (cells->links[cell] & 2) &&
	    map->blocks[block] != LEVELMAP_DOOR)
	    map->blocks[block] = LEVELMAP_OPEN;
    }
}

/**
 * Generate an individual chamber on the map from a cell.
 * @param map is the map to modify.
 * @param cells is the cell network.
 * @param cell is the cell number.
 */
static void generatechamber (LevelMap *map, CellData *cells, int cell)
{
    /* local variables */
    CellGeometry g; /* geometry of the cell */
    int x, /* horizontal counter */
	y, /* horizontal counter */
	block; /* block index */

    /* get the geometry */
    g = calculatecellgeometry (cells, cell);

    /* open up the room */
    for (x = g.tx; x <= g.bx; ++x)
	for (y = g.ty; y <= g.by; ++y) {
	    block = x + map->width * y;
	    map->blocks[block] = LEVELMAP_OPEN;
	}

    /* open up the exits */
    if (cells->links[cell] & 1)
	map->blocks[(g.bx + 1) + map->width * g.my] =
	    LEVELMAP_DOOR;
    if (cells->links[cell] & 2)
	map->blocks[g.mx + map->width * (g.by + 1)] =
	    LEVELMAP_DOOR;
    if (cells->links[cell] & 4)
	map->blocks[(g.tx - 1) + map->width * g.my] =
	    LEVELMAP_DOOR;
    if (cells->links[cell] & 8)
	map->blocks[g.mx + map->width * (g.ty - 1)] =
	    LEVELMAP_DOOR;
}

/**
 * Generate a combined chamber on the map from a cell.
 * @param map is the map to modify.
 * @param cells is the cell network.
 * @param cell is the cell number.
 */
static void generateopenchamber (LevelMap *map, CellData *cells, int cell)
{
    /* local variables */
    CellGeometry g; /* geometry of the cell */
    int x, /* horizontal counter */
	y, /* horizontal counter */
	t, /* top row of chamber section */
	l, /* left column of chamber section */
	b, /* bottom row of chamber section */
	r, /* right column of chamber section */
	block; /* block index */

    /* get the geometry */
    g = calculatecellgeometry (cells, cell);

    /* work out exactly where chamber section is */
    r = g.bx + ((cells->links[cell] & 1) &&
		g.col < cells->columns - 1 &&
		cells->types[cell + 1] == 2);
    b = g.by + ((cells->links[cell] & 2) &&
		g.row < cells->rows - 1 &&
		cells->types[cell + cells->columns] == 2);
    l = g.tx - ((cells->links[cell] & 4) &&
		g.col > 0 &&
		cells->types[cell - 1] == 2);
    t = g.ty - ((cells->links[cell] & 8) &&
		g.row > 0 &&
		cells->types[cell - cells->columns] == 2);

    /* open up the room, excluding the outer corners */
    for (x = l; x <= r; ++x)
	for (y = t; y <= b; ++y)
	    if ((x != g.tx - 1 && x != g.bx + 1) ||
		(y != g.ty - 1 && y != g.by + 1)) {
		block = x + map->width * y;
		map->blocks[block] = LEVELMAP_OPEN;
	    }

    /* add the doors out */
    if ((cells->links[cell] & 1) && g.bx == r)
	map->blocks[(g.bx + 1) + map->width * g.my] =
	    LEVELMAP_DOOR;
    if ((cells->links[cell] & 2) && g.by == b)
	map->blocks[g.mx + map->width * (g.by + 1)] =
	    LEVELMAP_DOOR;
    if ((cells->links[cell] & 4) && g.tx == l)
	map->blocks[(g.tx - 1) + map->width * g.my] =
	    LEVELMAP_DOOR;
    if ((cells->links[cell] & 8) && g.ty == t)
	map->blocks[g.mx + map->width * (g.ty - 1)] =
	    LEVELMAP_DOOR;
}

/*----------------------------------------------------------------------
 * Level 1 Functions.
 */

/**
 * Write an integer as a byte to an already open output file.
 * @param  value  A pointer to the integer variable to write.
 * @param  output The output file handle.
 * @return        1 if successful, 0 if not.
 */
static int writebyte (int *value, FILE *output)
{
    unsigned char c; /* character to write */
    c = (char) (*value & 0xff);
    return fwrite (&c, 1, 1, output);
}

/**
 * Read a byte from an already open input file
 * and store it in an integer variable.
 * @param  value A pointer to the integer variable to store into.
 * @param  input The input file handle.
 * @return       1 if successful, 0 if not.
 */
static int readbyte (int *value, FILE *input)
{
    unsigned char c; /* character to read */
    if (! (fread (&c, 1, 1, input)))
	return 0;
    *value = (int) c;
    return 1;
}

/**
 * Generate a size for the map.
 * If the dimensions have already been set, then this does nothing.
 * If one dimension has been set, the other is set at its maximum.
 * @param map is the map to affect.
 */
static void generatesize (LevelMap *map)
{
    /* both dimensions have already been set: leave them alone */
    if (map->width && map->height)
	return;

    /* one dimension has already been set: calculate the other */
    if (map->width || map->height) {
	if (map->width)
	    map->height = 256 / map->width;
	else
	    map->width = 256 / map->height;
	return;
    }

    /* neither dimension has been set: make them random */
    map->width = 9 + (rand () % 8);
    if (rand () % 2)
	map->width = 256 / map->width;
    map->height = 256 / map->width;
}

/**
 * Generate the cell grid.
 * @param map is the map to create the grid for.
 * @returns a newly allocated cell grid.
 */
static CellData *generatecells (LevelMap *map)
{
    CellData *cells; /* cell data to return */
    if (! (cells = malloc (sizeof (CellData))))
	return NULL;
    setcelldimensions (map, cells);
    connectcells (cells);
    openchambers (cells);
    return cells;
}

/**
 * Generate a whole map from a cell grid.
 * @param map is the map to generate.
 * @param cells is the cell network to work from.
 */
static int generatemapfromcells (LevelMap *map, CellData *cells)
{
    /* local variables */
    int c; /* cell count */
    
    /* fill the map with wall */
    if (map->blocks)
	free (map->blocks);
    if (! (map->blocks = malloc (map->width *
				       map->height)))
	return 0;
    memset (map->blocks, LEVELMAP_WALL,
	    map->width * map->height);

    /* main loop */
    for (c = 0; c < cells->columns * cells->rows; ++c)
	switch (cells->types[c]) {
	case 0:
	    generatecorridor (map, cells, c);
	    break;
	case 1:
	    generatechamber (map, cells, c);
	    break;
	case 2:
	    generateopenchamber (map, cells, c);
	    break;
	}

    /* operation was successful */
    return 1;
}

/**
 * Destroy the cell data when it is no longer needed.
 * @param cells is the cell data to destroy.
 */
static void destroycelldata (CellData *cells)
{
    free (cells->widths);
    free (cells->heights);
    free (cells->types);
    free (cells->links);
    free (cells);
}

/**
 * Connect a wall in one direction.
 * @param map is the map.
 * @param b is the block index.
 * @param d is the direction: 1=east, 2=south, 4=west, 8=north.
 */
static void connectwall (LevelMap *map, int b, int d)
{
    /* local variables */
    int a; /* adjacent block index */

    /* work out where adjacent block is, return if off edge */
    switch (d) {
	
    case 1: /* look to the east */
	a = b + 1;
	if (a % map->width == 0)
	    return;
	break;
	
    case 2: /* look to the south */
	a = b + map->width;
	if (a >= map->width * map->height)
	    return;
	break;
	
    case 4: /* look to the west */
	if (b % map->width == 0)
	    return;
	a = b - 1;
	break;
	
    case 8: /* look to the north */
	if (b < map->width)
	    return;
	a = b - map->width;
	break;
	
    default: /* invalid direction */
	return;
    }

    /* join to adjacent terrain if this is a wall or a door */
    if ((map->blocks[a] & 0xf0) == LEVELMAP_WALL ||
	(map->blocks[a] & 0xf0) == LEVELMAP_DOOR)
	map->blocks[b] |= d;

}

/**
 * Connect a non-wall block in one direction.
 * @param map is the map.
 * @param b is the block index.
 * @param d is the direction: 1=east, 2=south, 4=west, 8=north.
 */
static void connectblock (LevelMap *map, int b, int d)
{
    /* local variables */
    int a; /* adjacent block index */

    /* work out where adjacent block is, return if off edge */
    switch (d) {
	
    case 1: /* look to the east */
	a = b + 1;
	if (a % map->width == 0) {
	    map->blocks[b] |= d;
	    return;
	}
	break;
	
    case 2: /* look to the south */
	a = b + map->width;
	if (a >= map->width * map->height) {
	    map->blocks[b] |= d;
	    return;
	}
	break;
	
    case 4: /* look to the west */
	if (b % map->width == 0) {
	    map->blocks[b] |= d;
	    return;
	}
	a = b - 1;
	break;
	
    case 8: /* look to the north */
	if (b < map->width) {
	    map->blocks[b] |= d;
	    return;
	}
	a = b - map->width;
	break;
	
    default: /* invalid direction */
	return;
    }

    /* join to adjacent terrain if this is a wall or a door */
    if ((map->blocks[a] & 0xf0) == (map->blocks[b] & 0xf0))
	map->blocks[b] |= d;

}

/*----------------------------------------------------------------------
 * Public Level Functions.
 */

/**
 * Destroy the level map when it is no longer needed.
 * @param map is the map to destroy.
 */
static void destroy (LevelMap *map)
{
    if (map) {
	if (map->blocks)
	    free (map->blocks);
	free (map);
    }
}

/**
 * Write the level to an open file.
 * @param map    The level map to write.
 * @param output The output file handle.
 * @return       1 if successful, 0 if not.
 */
int write (LevelMap *map, FILE *output)
{
    /* write the map dimensions */
    if (! writebyte (&map->width, output))
	return 0;
    if (! writebyte (&map->height, output))
	return 0;

    /* write the blocks */
    if (! fwrite (map->blocks, map->width * map->height, 1, output))
	return 0;

    /* return successful */
    return 1;
}

/**
 * Read the level from an open file.
 * @param map   The level map to read.
 * @param input The output file handle.
 * @return      1 if successful, 0 if not.
 */
int read (LevelMap *map, FILE *input)
{
    /* read height and width */
    if (! readbyte (&map->width, input))
	return 0;
    if (! readbyte (&map->height, input))
	return 0;

    /* read the blocks */
    if (map->blocks)
	free (map->blocks);
    if (! (map->blocks = malloc (map->width * map->height)))
	return 0;
    if (! fread (map->blocks, map->width * map->height, 1, input))
	return 0;

    /* return success */
    return 1;
}

/**
 * Set the width of the level map prior to generation.
 * If not used, the width of the map will be randomised, or will
 * depend on an explicitly-set height.
 * Usage after generation has no effect.
 * @param map is the map to affect.
 * @param width is the width to set.
 * @returns 1 if successful, 0 if not.
 */
static int setwidth (LevelMap *map, int width)
{
    /* validation */
    if (map->blocks)
	return 0;
    if (width < 9 || width > 28)
	return 0;
    if (map->height * width > 256)
	return 0;

    /* set the width, and also the height if it is unset */
    map->width = width;
    if (! map->height)
	map->height = width / 256;

    /* return the code for success */
    return 1;
}

/**
 * Set the height of the level map prior to generation.
 * If not used, the height of the map will be randomised, or will
 * depend on an explicitly-set width.
 * Usage after generation has no effect.
 * @param map is the map to affect.
 * @param height is the height to set.
 * @returns 1 if successful, 0 if not.
 */
static int setheight (LevelMap *map, int height)
{
    /* validation */
    if (map->blocks)
	return 0;
    if (height < 9 || height > 28)
	return 0;
    if (map->width * height > 256)
	return 0;

    /* set the height, and also the width if it is unset */
    map->height = height;
    if (! map->width)
	map->width = height / 256;

    /* return the code for success */
    return 1;
}

/**
 * Set the chance of a cell being a corridor rather than a chamber.
 * If not used, the chance is 50%.
 * @param chance is the percentage chance.
 */
static int setcorridor (int chance)
{
    if (chance < 0 || chance > 100)
	return 0;
    corridorchance = chance;
    return 1;
}

/**
 * Set the chance of adjacent chambers being combined together.
 * If not used, the chance is 50%.
 * @param chance is the percentage chance.
 * @returns 1 if successful, 0 if not.
 */
static int setcombined (int chance)
{
    if (chance < 0 || chance > 100)
	return 0;
    combinedchamberchance = chance;
    return 1;
}

/**
 * Set the minimum height/width of a cell excluding walls.
 * Normally set at 2 to prevent 1-wide chambers.
 * @param minimum is the new minimum size.
 * @returns 1 if successful, 0 if not.
 */
static int setcellsize (int minimum)
{
    if (minimum < 1)
	return 0;
    mincellsize = minimum;
    return 1;
}

/**
 * Generate the map.
 * @param map is the map to affect.
 */
static int generate (LevelMap *map)
{
    /* local variables */
    CellData *cells; /* temporary data on cells */
    int generated; /* 1 if the map was generated */

    /* validation */
    if (map->blocks)
	return 0;

    /* main generation stages */
    generatesize (map);
    if (! (cells = generatecells (map)))
	return 0;
    generated = generatemapfromcells (map, cells);
    destroycelldata (cells);

    /* tell the calling process if the cells were generated */
    return generated;
}

/**
 * Turn fully enclosed wall spaces into voids.
 * @param map is the map to affect.
 * @returns 1 on success, 0 on failure.
 */
static int openvoids (LevelMap *map)
{
    /* local variables */
    int b, /* block counter */
	isvoid, /* void flag */
	x, /* x location */
	y, /* y location */
	xx, /* x offset */
	yy, /* y offset */
	adj; /* contents of adjacent block */

    /* validation */
    if (! map->blocks)
	return 0;

    /* check each wall cell to see if it should be a void instead */
    for (b = 0; b < map->width * map->height; ++b)
	if (map->blocks[b] == LEVELMAP_WALL) {

	    /* initialise */
	    isvoid = 1;
	    x = b % map->width;
	    y = b / map->width;

	    /* scan adjacent cells for non-wall non-void blocks */
	    for (xx = -1; xx <= +1; ++xx)
		for (yy = -1; yy <= +1; ++yy) {
		    adj = map->getblock (map, x + xx, y + yy);
		    if (adj != LEVELMAP_WALL &&
			adj != LEVELMAP_VOID)
			isvoid = 0;
		}

	    /* if this wall is enclosed, turn it into a void */
	    if (isvoid)
		map->blocks[b] = LEVELMAP_VOID;
	}

    /* successful */
    return 1;
}

/**
 * Connect the textures of the blocks.
 * @param map is the map whose blocks should be connected.
 * @returns 1 on success, 0 on failure.
 */
static int connecttextures (LevelMap *map)
{
    /* local variables */
    int b; /* block counter */

    /* validation */
    if (! map->blocks)
	return 0;

    /* connect each block in each direction */
    for (b = 0; b < map->width * map->height; ++b)

	/* walls should connect to doors and not to exterior */
	if (map->blocks[b] == LEVELMAP_WALL ||
	    map->blocks[b] == LEVELMAP_DOOR) {
	    connectwall (map, b, 1);
	    connectwall (map, b, 2);
	    connectwall (map, b, 4);
	    connectwall (map, b, 8);
	}

	/* other blocks should connect with same and exterior */
	else {
	    connectblock (map, b, 1);
	    connectblock (map, b, 2);
	    connectblock (map, b, 4);
	    connectblock (map, b, 8);
	}

    /* successful */
    return 1;
}

/**
 * Get the contents of an individual block.
 * @param map is the map to inspect.
 * @param x is the x location of the block.
 * @param y is the y location of the block.
 * @returns 0 if the block is passable.
 * @returns 0x1-0xf if the block is a wall.
 * @returns 0xff if outside the map.
 */
static int getblock (LevelMap *map, int x, int y)
{
    /* validation */
    if (! map->blocks)
	return LEVELMAP_VOID;
    if (x < 0 || x >= map->width)
	return LEVELMAP_VOID;
    if (y < 0 || y >= map->height)
	return LEVELMAP_VOID;

    /* return the block */
    return map->blocks[x + map->width * y];
}

/*----------------------------------------------------------------------
 * Constructors.
 */

/**
 * Create a new map object.
 * @returns an unsized and empty map.
 */
LevelMap *new_LevelMap (void)
{
    /* local variables */
    LevelMap *map;

    /* allocate the memory */
    if (! (map = malloc (sizeof (LevelMap))))
	return NULL;

    /* initialise the attributes */
    map->width = 0;
    map->height = 0;
    map->blocks = NULL;

    /* initialise the methods */
    map->destroy = destroy;
    map->write = write;
    map->read = read;
    map->setwidth = setwidth;
    map->setheight = setheight;
    map->setcorridor = setcorridor;
    map->setcombined = setcombined;
    map->setcellsize = setcellsize;
    map->generate = generate;
    map->openvoids = openvoids;
    map->connecttextures = connecttextures;
    map->getblock = getblock;

    /* return the newly generated structure */
    return map;
}
