/*======================================================================
 * Dungeon Level Map Generation Library.
 * Generates small generic dungeon-like level maps.
 *
 * Released into the Public Domain by Damian Gareth Walker, 2020.
 * Created: 29-Jul-2020.
 *
 * This is the main header.
 */

#ifndef __LEVELMAP_H__
#define __LEVELMAP_H__

/*----------------------------------------------------------------------
 * Constants
 */

/* base values in the map array */
#define LEVELMAP_VOID 0x00
#define LEVELMAP_OPEN 0x10
#define LEVELMAP_WALL 0x20
#define LEVELMAP_DOOR 0x30

/*----------------------------------------------------------------------
 * Data Definitions
 */

/** @struct level_map is the main dungeon level structure */
typedef struct levelmap LevelMap;
struct levelmap {

    /** @var width is the width of the map. */
    int width;

    /** @var height is the height of the map. */
    int height;

    /** @var blocks is the map block storage buffer. */
    unsigned char *blocks;

    /**
     * Destroy the level map when it is no longer needed.
     * @param map is the map to destroy.
     */
    void (*destroy) (LevelMap *map);

    /**
     * Write the level to an open file.
     * @param map    The level map to write.
     * @param output The output file handle.
     * @return       1 if successful, 0 if not.
     */
    int (*write) (LevelMap *map, FILE *output);

    /**
     * Read the level from an open file.
     * @param map   The level map to read.
     * @param input The output file handle.
     * @return      1 if successful, 0 if not.
     */
    int (*read) (LevelMap *map, FILE *input);

    /**
     * Set the width of the level map prior to generation.
     * If not used, the width of the map will be randomised, or will
     * depend on an explicitly-set height.
     * Usage after generation has no effect.
     * @param map is the map to affect.
     * @param width is the width to set.
     * @returns 1 if successful, 0 if not.
     */
    int (*setwidth) (LevelMap *map, int width);

    /**
     * Set the height of the level map prior to generation.
     * If not used, the height of the map will be randomised, or will
     * depend on an explicitly-set width.
     * Usage after generation has no effect.
     * @param map is the map to affect.
     * @param height is the height to set.
     * @returns 1 if successful, 0 if not.
     */
    int (*setheight) (LevelMap *map, int height);

    /**
     * Set the chance of a cell being a corridor rather than a chamber.
     * If not used, the chance is 50%.
     * @param chance is the percentage chance.
     * @returns 1 if successful, 0 if not.
     */
    int (*setcorridor) (int chance);

    /**
     * Set the chance of adjacent chambers being combined together.
     * If not used, the chance is 50%.
     * @param chance is the percentage chance.
     * @returns 1 if successful, 0 if not.
     */
    int (*setcombined) (int chance);

    /**
     * Set the minimum height/width of a cell excluding walls.
     * Normally set at 2 to prevent 1-wide chambers.
     * @param minimum is the new minimum size.
     * @returns 1 if successful, 0 if not.
     */
    int (*setcellsize) (int minimum);

    /**
     * Generate the map.
     * @param map is the map to affect.
     */
    int (*generate) (LevelMap *map);

    /**
     * Turn fully enclosed wall spaces into voids.
     * @param map is the map to affect.
     * @returns 1 on success, 0 on failure.
     */
    int (*openvoids) (LevelMap *map);
    
    /**
     * Connect the walls.
     * @param map is the map whose walls should be connected.
     * @returns 1 on success, 0 on failure.
     */
    int (*connecttextures) (LevelMap *map);

    /**
     * Get the contents of an individual block.
     * @param map is the map to inspect.
     * @param x is the x location of the block.
     * @param y is the y location of the block.
     * @returns 0 if the block is impassable or outside map.
     */
    int (*getblock) (LevelMap *map, int x, int y);

};

/*----------------------------------------------------------------------
 * Constructors.
 */

/**
 * Create a new map object.
 * @returns an unsized and empty map.
 */
LevelMap *new_LevelMap (void);

#endif
