/*======================================================================
 * CGALib - Watcom C Version.
 * Music Editor Program.
 * 
 * Released as Public Domain by Damian Gareth Walker, 2025.
 * Created 18-Sep-2025.
 */

/*----------------------------------------------------------------------
 * Required Headers.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

/* project headers */
#include "cgalib.h"

/*----------------------------------------------------------------------
 * File Level Variables.
 */

/** @var scr The screen object. */
static Screen *scr = NULL;

/** @var mainfont The main font object. */
static Font *mainfont = NULL;

/** @var durfont The duration font object. */
static Font *durfont = NULL;

/** @var headfont The header font object. */
static Font *headfont = NULL;

/** @var highfont The highlight font object. */
static Font *highfont = NULL;

/** @var copyfont The copy mark font object. */
static Font *copyfont = NULL;

/** @var keys The keyboard handler. */
static KeyHandler *keys;

/** @var tune The edited tune. */
static Tune *tune;

/** @var indexnotes The first notes on each row of the tune. */
static Note **indexnotes;

/** @var indexcount The number of index notes in the tune. */
static int indexcount;

/** @var pageindex The first note on the page */
static int pageindex;

/** @var cursor The note row cursor */
static int cursor;

/** @var copymark The note row to copy from */
static int copymark;

/** @var mono 1 if the user wants a monochrome screen. */
static int mono;

/** @var path Path to the executable. */
static char path[128];

/** @var filename The filename. */
static char filename[128];

/** @var sharpnames Names of the notes, using sharps. */
static char *sharpnames[] = {
    "C.",
    "C#",
    "D.",
    "D#",
    "E.",
    "F.",
    "F#",
    "G.",
    "G#",
    "A.",
    "A#",
    "B."
};

/** @var sharpnames Names of the notes, using sharps. */
static char *flatnames[] = {
    "C.",
    "D$",
    "D.",
    "E$",
    "E.",
    "F.",
    "G$",
    "G.",
    "A$",
    "A.",
    "B$",
    "B."
};

/** @var notenames Pointer to the selected note names. */
static char **notenames = sharpnames;

/*----------------------------------------------------------------------
 * Service Routines.
 */

/**
 * Error Handler.
 * @param errorlevel is the error level to return to the OS.
 * @param message is the message to print.
 */
void error_handler (int errorlevel, char *message)
{
    if (scr)
        scr->destroy (scr);
    puts (message);
    exit (errorlevel);
}

/**
 * Print a note value on the screen.
 * @param x     Location to print to, x coordinate.
 * @param y     Location to print to, y coordinate.
 * @param pitch The pitch to print.
 */
static void printnote (int x, int y, int pitch)
{
    char notestr[4];
    sprintf (notestr, "%-2s%1d", notenames[pitch % 12], pitch / 12);
    scr->print (scr, x, y, notestr);
}

/**
 * Get a number.
 * @param address Pointer to the number store.
 * @param x       X location on the screen.
 * @param y       Y location on the screen.
 * @param min     Minimum value.
 * @param max     Maximum value.
 * @param step    Step of increase/decrease.
 */
void getnum (int *address, int x, int y, int min, int max, int step)
{
    int key; /* keypress */
    char numstr[6], /* number expressed as a string */
	*format, /* number format */
	msg[81]; /* error message */

    /* determine number format based on maximum value */
    if (max < 10)
	format = "%1d";
    else if (max < 100)
	format = "%02d";
    else if (max < 1000)
	format = "%03d";
    else if (max < 10000)
	format = "%04d";
    else
	format = "%05d";

    /* main entry loop */
    scr->font = highfont;
    do {
	sprintf (numstr, format, *address);
	scr->print (scr, x, y, numstr);
	scr->update (scr);
	keys->wait ();
	key = keys->ascii ();
	if (key == 11 && *address < max)
	    *address += step;
	else if (key == 10 && *address > min)
	    *address -= step;
	else if ((key >= '0' && key <= '9') &&
		 key - '0' + *address * 10 <= max)
	    *address = key - '0' + *address * 10;
	else if (key == 8)
	    *address /= 10;
	else if (key == 13 && (*address < min || *address > max)) {
	    sprintf (msg, "Value must be in range %d..%d", min, max);
	    scr->font = mainfont;
	    scr->print (scr, 0, 192, msg);
	    scr->update (scr);
	    scr->font = highfont;
	}
    } while (key != 13 || *address < min || *address > max);
    scr->box (scr, 0, 192, 320, 8, CGALIB_BOX_BLANK);
    scr->font = mainfont;
    scr->print (scr, x, y, numstr);
}

/**
 * Get a note value.
 * @param address Pointer to the note value.
 * @param x       X location on the screen.
 * @param y       Y location on the screen.
 * @return        1 if ENTER pressed, 0 if ESC.
 */
int getnote (int *address, int x, int y)
{
    char key; /* ASCII value of key pressed */
    int pitch; /* note pitch within the octave */
    scr->font = highfont;
    do {
	printnote (x, y, *address);
	scr->update (scr);
	keys->wait ();
	key = keys->ascii ();
	if (key == 11 && *address < 255)
	    ++*address;
	else if (key == 10 && *address > 0)
	    --*address;
	else if (key >= '0' && key <= '9')
	    *address = *address % 12
		+ 12 * (key - '0');
	else if (tolower (key) >= 'a' && tolower (key) <= 'g') {
	    pitch = (tolower (key) >= 'c')
		? tolower (key) - 'c'
		: 5 + tolower (key) - 'a';
	    *address = (*address / 12 * 12)
		+ pitch
		+ (pitch > 0)
		+ (pitch > 1)
		+ (pitch > 3)
		+ (pitch > 4)
		+ (pitch > 5);
	}
    } while (key != 13 && key != 27);
    scr->font = mainfont;
    printnote (x, y, *address);
    return (key == 13);
}

/**
 * Print a numeric value on the screen.
 * @param x     Location to print to, x coordinate.
 * @param y     Location to print to, y coordinate.
 * @param value The value to print.
 */
static void printval (int x, int y, int value)
{
    char numstr[4];
    sprintf (numstr, "%03d", value);
    scr->print (scr, x, y, numstr);
}

/*----------------------------------------------------------------------
 * Level 4 Routines.
 */

/**
 * Show a row of notes.
 * @param row  The screen row.
 * @param note The first note of the series.
 * @param high 1 if the row is highlighted, otherwise 0.
 * @param copy 1 if the row is marked for copying, otherwise 0.
 * @return     The first note of the next row, or NULL if none.
 */
static Note *shownotes (int row, Note *note, int high, int copy)
{
    int c = 0, /* counter */
	x, /* x location */
	y; /* y location */

    /* blank out the line or highlight it */
    x = 164 * (row / 23);
    y = 8 + 8 * (row % 23);
    scr->paper = high ? 2 : copy ? 1 : 0;
    scr->box (scr, x, y, 156, 8, CGALIB_BOX_BLANK);
    scr->paper = 0;

    /* make sure we're looking at a note */
    if (! note)
	return 0;

    /* print the note row duration */
    scr->font = high ? highfont : copy ? copyfont : durfont;
    printval (x, y, note->duration);
    scr->font = high ? highfont : copy ? copyfont : mainfont;

    /* print the notes */
    do {
	printnote (16 + x + 16 * c, y, note->pitch);
	note = note->next;
	++c;
    } while (note && ! note->duration);
    scr->font = mainfont;

    /* return the number of notes printed */
    return note;
}

/**
 * Show blank row in place of notes.
 * @param row The screen row, 0 to 45.
 */
static void showblank (int row)
{
    int x, /* x location */
	y; /* y location */
    x = 164 * (row / 23);
    y = 8 + 8 * (row % 23);
    scr->box (scr, x, y, 156, 8, CGALIB_BOX_BLANK);
}

/*----------------------------------------------------------------------
 * Level 3 Routines
 */

/**
 * Load the font and create recoloured versions.
 * @param filename is the name of the font file.
 * @returns the loaded font.
 * The font file loaded by this function requires an 8-byte header,
 * consisting of the text CGA100F and a null byte. Then follows two
 * bytes determining the first and last character codes supported by
 * the font. After that is the font pixel data.
 */
static int load_font (void)
{
    FILE *fp; /* file pointer */
    char filename[256], /* nam of the font file */
	header[8]; /* font file header */

    /* attempt to open the file, and read and verify the header */
    sprintf (filename, "%sfnt/maketune.fnt", path);
    if (! (fp = fopen (filename, "rb")))
        return 0;
    else if (! fread (header, 8, 1, fp)) {
        fclose (fp);
        return 0;
    } else if
	    (strcmp (header, "CGA100F")
	     && strcmp (header, "CGA200F")) {
        fclose (fp);
        return 0;
    }

    /* read the font and create recoloured versions */
    mainfont = read_Font (fp, header[3] - '0');
    durfont = mainfont->clone (mainfont);
    durfont->recolour (durfont, 1, 0);
    headfont = mainfont->clone (mainfont);
    headfont->recolour (headfont, 2, 0);
    highfont = mainfont->clone (mainfont);
    highfont->recolour (highfont, 3, 2);
    copyfont = mainfont->clone (mainfont);
    copyfont->recolour (copyfont, 0, 1);
    fclose (fp);
    return 1;
}

/**
 * Load the sound effects to edit
 * @param filename The sound effect file.
 * @return         1 if successful, 0 on failure.
 */
static int load_tune (char *filename)
{
    FILE *fp; /* file pointer */
    char header[8]; /* effect file header */
    int c; /* counter */
    Note *note; /* pointer to a note */

    /* attempt to open the file, and read and verify the header */
    if (! (fp = fopen (filename, "rb")))
        return 0;
    else if (! fread (header, 8, 1, fp)) {
        fclose (fp);
        return 0;
    } else if (strcmp (header, "CGA100T") &&
	       strcmp (header, "SPK100T"))
    {
	fclose (fp);
        return 0;
    }

    /* read the tune */
    if (! (tune = new_Tune ())) {
	fclose (fp);
	return 0;
    }
    if (! tune->read (tune, fp)) {
	fclose (fp);
	return 0;
    }
    fclose (fp);

    /* index the tune */
    for (note = tune->notes; note; note = note->next)
	if (note->duration)
	    ++indexcount;
    if (! (indexnotes = malloc (indexcount * sizeof (Note *))))
	return 0;
    for (c = 0, note = tune->notes; note; note = note->next)
	if (note->duration)
	    indexnotes[c++] = note;

    /* return */
    return 1;
}

/**
 * Show all notes on the screen.
 */
static void showallnotes (void)
{
    int c; /* counter */
    for (c = pageindex; c < pageindex + 46; ++c)
	if (c < indexcount)
	    shownotes
		(c - pageindex,
		 indexnotes[c],
		 c == cursor,
		 c == copymark);
	else if (c == indexcount)
	    shownotes (c - pageindex, NULL, c == cursor, c == copymark);
	else
	    showblank (c - pageindex);
}

/*----------------------------------------------------------------------
 * Level 2 Routines.
 */

static void initialise_args (int argc, char **argv)
{
    char errormsg[81]; /* error message */
    while (argc-- > 1)
	if (! strcmp (argv[argc], "-m") ||
	    ! strcmp (argv[argc], "-M"))
	    mono = 1;
	else if (*argv[argc] == '-') {
	    sprintf (errormsg, "Invalid option %s", argv[argc]);
	    error_handler (1, errormsg);
	} else
	    strcpy (filename, argv[argc]);
    if (strrchr (argv[0], '\\')) {
	strcpy (path, argv[0]);
	*(strrchr (path, '\\') + 1) = '\0';
    } else
	*path = '\0';
}

/**
 * Initialise the screen.
 * @param mono is true if mono mode was requested.
 */
void initialise_screen (int mono)
{
    /* initialise screen and assets */
    if (! (scr = new_Screen (mono ? 6 : 4, CGALIB_SHOWN)))
        error_handler (1, "Cannot initialise graphics mode!");
    if (! (load_font ()))
        error_handler (1, "Cannot load font");

    /* initial screen display */
    scr->font = headfont;
    scr->print (scr, 0, 0, "Dur Notes");
    scr->print (scr, 164, 0, "Dur Notes");
    scr->font = mainfont;
    scr->updates = 1;
    showallnotes ();
}

/**
 * Edit a note row.
 */
static void editrow (void)
{
    int c = 0, /* counter */
	x, /* x location */
	y; /* y location */
    Note *note, /* note being edited */
	*prev; /* the note before the one being edited */

    /* ensure there are notes on this row */
    if (cursor == indexcount) {
	indexnotes = realloc (indexnotes, ++indexcount * sizeof (Note *));
	note = indexnotes[indexcount - 1] = new_Note (0, 0);
	tune->add (tune, note);
    } else if (! (note = indexnotes[cursor]))
	return; /* will insert? */

    /* display the row without highlighting */
    x = 164 * ((cursor - pageindex) / 23);
    y = 8 + 8 * ((cursor - pageindex) % 23);
    shownotes (cursor - pageindex, note, 0, 0);

    /* get the duration and initial pitch */
    getnum (&note->duration, x, y, 1, 255, 1);
    scr->font = durfont;
    printval (x, y, note->duration);
    scr->font = mainfont;
    getnote (&note->pitch, x + 16, y);

    /* get the rest of the existing notes */
    prev = note;
    while ((note = note->next) && note->duration == 0) {
	++c;
	getnote (&note->pitch, x + 16 * (c + 1), y);
	prev = note;
    }

    /* allow more notes to be added */
    while (c < 9) {
	note = new_Note (0, 0);
	++c;
	if (getnote (&note->pitch, x + 16 * (c + 1), y)) {
	    note->next = prev->next;
	    prev->next = note;
	} else {
	    free (note);
	    break;
	}
	prev = note;
	note = note->next;
    }

    /* redisplay the row */
    shownotes (cursor - pageindex, indexnotes[cursor], 1, 0);
}

/**
 * Copy note row from the copy mark to the cursor.
 */
static void copynotes (void)
{
    Note *fromnote, /* note to copy from */
	*tonote, /* note to copy to */
	*note, /* temporary pointer to new note */
	*prev; /* note at end of previous row */
    int c; /* insertion counter */

    /* point to first note of the copy marked row */
    fromnote = indexnotes[copymark];

    /* append a row */
    if (cursor == indexcount) {
	indexnotes = realloc (indexnotes, ++indexcount * sizeof (Note *));
	tonote = indexnotes[indexcount - 1]
	    = new_Note (fromnote->pitch, fromnote->duration);
	tune->add (tune, tonote);
	while (fromnote->next && ! fromnote->next->duration) {
	    fromnote = fromnote->next;
	    note = new_Note (fromnote->pitch, fromnote->duration);
	    tune->add (tune, note);
	    tonote = tonote->next = note;
	}
    }

    /* insert a row into the middle */
    else if (cursor > 0) {

	/* open up a gap for the pasted notes */
	for (prev = indexnotes[cursor - 1];
	     prev->next != indexnotes[cursor];
	     prev = prev->next);
	indexnotes = realloc (indexnotes, ++indexcount * sizeof (Note *));
	for (c = indexcount - 1; c > cursor; --c)
	    indexnotes[c] = indexnotes[c - 1];

	/* copy first note to the new row */
 	tonote = indexnotes[cursor]
	    = new_Note (fromnote->pitch, fromnote->duration);
	prev->next = indexnotes[cursor];
	tonote->next = indexnotes[cursor + 1];

	/* paste subsequent notes to the new row */
	while (fromnote->next && ! fromnote->next->duration) {
	    fromnote = fromnote->next;
	    tonote->next = note
		= new_Note (fromnote->pitch, fromnote->duration);
	    note->next = indexnotes[cursor + 1];
	    tonote = tonote->next;
	}

    }

    /* insert a row at the beginning */
    else {

	/* open up a gap for the pasted notes */
	indexnotes = realloc (indexnotes, ++indexcount * sizeof (Note *));
	for (c = indexcount - 1; c > cursor; --c)
	    indexnotes[c] = indexnotes[c - 1];

	/* copy first note to the new row */
 	tune->notes = tonote = indexnotes[cursor]
	    = new_Note (fromnote->pitch, fromnote->duration);
	tonote->next = indexnotes[cursor + 1];

	/* paste subsequent notes to the new row */
	while (fromnote->next && ! fromnote->next->duration) {
	    fromnote = fromnote->next;
	    tonote->next = note
		= new_Note (fromnote->pitch, fromnote->duration);
	    note->next = indexnotes[cursor + 1];
	    tonote = tonote->next;
	}

    }

    /* get ready to copy the next row */
    ++copymark;
    ++cursor;
    if (cursor >= pageindex + 46) {
	pageindex += 46;
	showallnotes ();
    } else if (cursor < indexcount) {
	showallnotes ();
    } else {
	shownotes
	    (cursor - 1 - pageindex,
	     indexnotes [cursor - 1],
	     0,
	     cursor - 1 == copymark);
	shownotes (cursor - pageindex, indexnotes[cursor], 1, 0);
    }
}

/**
 * Insert notes:
 * Open up a row and start inserting notes.
 */
static void insertnotes (void)
{
    int c; /* counter */
    Note *note; /* note pointer */

    /* insert a single note on an empty row */
    indexnotes = realloc (indexnotes, ++indexcount * sizeof (Note *));
    for (c = indexcount - 1; c > cursor; --c)
	indexnotes[c] = indexnotes[c - 1];
    indexnotes[cursor] = new_Note (0, 0);

    /* link the new note */
    indexnotes[cursor]->next = indexnotes[cursor + 1];
    for (note = tune->notes;
	 note->next != indexnotes[cursor + 1];
	 note = note->next);
    note->next = indexnotes[cursor];

    /* update the screen, then edit the new note row */
    showallnotes ();
    editrow ();
}

/**
 * Delete notes:
 * Remove a note row and close up the gap.
 */
static void deletenotes (void)
{
    int c; /* counter */
    Note *note, /* note pointer */
	*deadnote; /* the note about to be deleted */

    /* link rows before and after the one to be deleted */
    if (cursor == 0)
	tune->notes = indexnotes[cursor + 1];
    else if (cursor > 0) {
	for (note = indexnotes[cursor - 1];
	     note->next != indexnotes[cursor];
	     note = note->next);
	if (cursor == indexcount - 1)
	    note->next = NULL;
	else
	    note->next = indexnotes[cursor + 1];
    }

    /* remove the notes on the current row */
    note = indexnotes[cursor];
    while (note->next && ! note->next->duration) {
	deadnote = note;
	note = note->next;
	free (deadnote);
    }
    free (note);

    /* close the gap */
    for (c = cursor; c < indexcount - 1; ++c)
	indexnotes[c] = indexnotes[c + 1];
    indexnotes = realloc
	(indexnotes,
	 (indexcount - 1) * sizeof (Note *));
    --indexcount;

    /* refresh the display */
    showallnotes ();
}

/**
 * Save the tune.
 */
static void savetune (void)
{
    int key, /* keypress */
	len; /* length of string */
    FILE *fp; /* file pointer */

    /* get the filename */
    if (! *filename) {
	scr->print (scr, 0, 192, "Filename: ");
	scr->update (scr);
	do {
	    keys->wait ();
	    key = keys->ascii ();
	    if (key >= ' ' && key <= '~') {
		len = strlen(filename);
		filename[len] = key;
		filename[len + 1] = '\0';
		scr->print (scr, 40, 192, filename);
	    } else if (key == 8 && *filename) {
		len = strlen(filename);
		filename[len - 1] = '\0';
		len = strlen(filename);
		scr->print (scr, 40 + 4 * len, 192, " ");
	    }
	    scr->update (scr);
	} while (key != 13);
	if (*filename && ! strchr (filename, '.'))
	    strcat (filename, ".tun");
    }
    if (! *filename) return;

    /* save the file */
    if (! (fp = fopen (filename, "wb")))
	return;
    fwrite ("CGA100T", 8, 1, fp);
    tune->write (tune, fp);
    fclose (fp);
}

/*----------------------------------------------------------------------
 * Level 1 Routines.
 */

/**
 * Initialise the program.
 * @param mono 1 for monochrome, 0 for colour.
 * @return     1 if successful, 0 on failure.
 */
static int initialise (int argc, char **argv)
{
    initialise_args (argc, argv);
    if (*filename && ! load_tune (filename))
	error_handler (2, "Cannot load tune");
    else if (! *filename &&
	     ! (tune = new_Tune ()))
	error_handler (1, "Cannot create tune");
    initialise_screen (mono);
    keys = new_KeyHandler ();
    return 1;
}

/**
 * Main program loop.
 * @return 0 when finished, 1 to continue.
 */
static int main_program (void)
{
    int key; /* key press */

    /* get keypress */
    scr->update (scr);
    keys->wait ();
    key = keys->scancode ();

    /* cursor up */
    if (key == KEY_KP8 && cursor > 0) {
	shownotes
	    (cursor - pageindex,
	     cursor < indexcount ? indexnotes[cursor] : NULL,
	     0,
	     cursor == copymark);
	--cursor;
	if (cursor < pageindex) {
	    pageindex -= (pageindex < 46) ? pageindex : 46;
	    showallnotes ();
	}
	shownotes (cursor - pageindex, indexnotes[cursor], 1, 0);
    }

    /* cursor down */
    else if (key == KEY_KP2 && cursor < indexcount) {
	shownotes
	    (cursor - pageindex,
	     indexnotes[cursor],
	     0,
	     cursor == copymark);
	++cursor;
	if (cursor >= pageindex + 46) {
	    pageindex += 46;
	    showallnotes ();
	}
	shownotes
	    (cursor - pageindex,
	     cursor < indexcount ? indexnotes[cursor] : NULL,
	     1,
	     0);
    }

    /* Page Up */
    else if (key == KEY_KP9 && pageindex > 0) {
	pageindex -= 46;
	cursor -= 46;
	showallnotes ();
    }

    /* Page Down */
    else if (key == KEY_KP3 && pageindex <= indexcount - 46) {
	pageindex += 46;
	cursor += 46;
	if (cursor > indexcount)
	    cursor = indexcount;
	showallnotes ();
    }

    /* Home */
    else if (key == KEY_KP7 && cursor > 0) {
	pageindex = cursor = 0;
	showallnotes ();
    }

    /* End */
    else if (key == KEY_KP1 && cursor < indexcount) {
	cursor = indexcount;
	pageindex = cursor - (cursor % 46);
	showallnotes ();
    }

    /* ENTER (edit line) */
    else if (key == KEY_ENTER)
	editrow ();

    /* SPACE (play) */
    else if (key == KEY_SPACE && cursor < indexcount) {
	scr->print (scr, 0, 192, "Playing...");
	scr->update (scr);
	tune->note = indexnotes[cursor];
	tune->play (tune, keys);
	scr->box (scr, 0, 192, 320, 8, CGALIB_BOX_BLANK);
	scr->update (scr);
    }

    /* M (set copy marker) */
    else if (key == KEY_M && cursor < indexcount) {
	if (copymark >= pageindex && copymark < pageindex + 46)
	    shownotes
		(copymark - pageindex,
		 indexnotes[copymark],
		 0,
		 0);
	copymark = cursor;
    }

    /* C (copy from copy marker to the cursor) */
    else if (key == KEY_C && copymark < indexcount)
	copynotes ();

    /* Insert */
    else if (key == KEY_KP0 && cursor < indexcount)
	insertnotes ();

    /* Delete */
    else if (key == KEY_KPSTOP && cursor < indexcount)
	deletenotes ();

    /* Left bracket (use flats) */
    else if (key == KEY_LBRACK) {
	notenames = flatnames;
	showallnotes ();
    }

    /* Right bracket (use sharps) */
    else if (key == KEY_RBRACK) {
	notenames = sharpnames;
	showallnotes ();
    }
    
    /* ESC (quit) */
    else if (key == KEY_ESC)
	return 0;

    /* return */
    return 1;
}

/**
 * End the program.
 */
static void end_program (void)
{
    Speaker *speaker; /* check speaker */
    savetune ();
    if (tune)
	tune->destroy (tune);
    if (indexnotes)
	free (indexnotes);
    if ((speaker = get_Speaker ()))
	speaker->destroy ();
    keys->destroy ();
    mainfont->destroy (mainfont);
    durfont->destroy (durfont);
    headfont->destroy (headfont);
    highfont->destroy (highfont);
    copyfont->destroy (copyfont);
    scr->destroy (scr);
}

/*----------------------------------------------------------------------
 * Top Level Routine.
 */

/**
 * Main program.
 * @param argc is the number of command line argumets.
 * @param argv is the command line arguments.
 * No return value as exit () is used to terminate abnormally.
 */
int main (int argc, char **argv)
{
    if (initialise (argc, argv)) {
	while (main_program ());
	end_program ();
    }
    return 0;
}
