/*======================================================================
 * CGALIB version 2
 * Font handling module.
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 06-Dec-2022.
 */

/*----------------------------------------------------------------------
 * Required Headers.
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "cgalib.h"

/*----------------------------------------------------------------------
 * Public Method Level Definitions.
 */

/**
 * Destroy a font when it is no longer needed.
 * @param font The font to destroy.
 */
static void destroy (Font *font)
{
    if (font) {
        if (font->pixels)
            free (font->pixels);
        free (font);
    }
}

/**
 * Clone a font.
 * @param  font The font to clone.
 * @return      The new font.
 */
static Font *clone (Font *font)
{
    Font *newfont; /* destination font */
    int s; /* font character size in bytes */

    /* attempt to reserve memory */
    s = font->width / 4 * font->height;
    newfont = new_Font (font->first, font->last, font->width, font->height);
    if (! newfont)
	return NULL;

    /* set the font information */
    memcpy
	(newfont->pixels,
	 font->pixels, s * (font->last - font->first + 1));

    /* return the font */
    return newfont;
}

/**
 * Write a font to an already open file.
 * @param  font   The font to write.
 * @param  output The output file handle.
 * @return        1 on success, 0 on failure.
 */
static int write (Font *font, FILE *output)
{
    if (! fwrite (&font->first, 1, 1, output))
        return 0;
    if (! fwrite (&font->last, 1, 1, output))
        return 0;
    if (! fwrite (&font->width, 1, 1, output))
        return 0;
    if (! fwrite (&font->height, 1, 1, output))
        return 0;
    if (! fwrite
	(font->pixels,
	 font->width / 4 * font->height
	 * (font->last - font->first + 1),
	 1,
	 output))
        return 0;
    return 1;
}

/**
 * Put a character into the font.
 * @param  font   The font to modify.
 * @param  bitmap The bitmap containing the character.
 * @param  code   The character code to modify.
 */
static void put (Font *font, Bitmap *bitmap, int code)
{
    memcpy
	(font->pixels + (font->width / 4 * font->height)
	 * (code - font->first),
	 bitmap->pixels,
	 font->width / 4 * font->height);
}

/**
 * Get a character from the font.
 * @param  font The font to extract from.
 * @param  code The character code to extract.
 * @return      A bitmap resembling the character.
 */
static Bitmap *get (Font *font, int code)
{
    Bitmap *bitmap; /* the bitmap to create */
    bitmap = new_Bitmap (font->width, font->height);
    memcpy
	(bitmap->pixels,
	 font->pixels + (font->width / 4 * font->height)
	 * (code - font->first),
	 font->width / 4 * font->height);
    return bitmap;
}

/**
 * Recolour a whole font permanently.  This is faster than
 * printing the standard font with alternative ink and paper
 * colours. The font must be in colours 3 ink and 0 paper before
 * the conversion.
 * @param  font  The font to recolour.
 * @param  ink   The new ink colour.
 * @param  paper The new paper colour.
 */
static void recolour (Font *font, int ink, int paper)
{
    int ch; /* character count */
    int r; /* row count */
    char p; /* paper mask */
    char i; /* ink mask */
    char b; /* background of recoloured character */
    char f; /* foreground of recoloured character */
    int s; /* font character size in bytes */

    /* font is assumed to be colour 3 on colour 0 already */
    if (ink == 3 && paper == 0)
	return;

    /* work out the ink and paper masks */
    i = ink * 0x55;
    p = paper * 0x55;
    s = font->width / 4 * font->height;

    /* loop through all the characters */
    for (ch = 0; ch < font->last - font->first; ++ch)
        for (r = 0; r < s; ++r) {
            f = font->pixels[s * ch + r] & i;
            b = (0xff ^ font->pixels[s * ch + r]) & p;
            font->pixels[s * ch + r] = f | b;
        }
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Cronstruct a new font.
 * @param  first The first character in the font.
 * @param  last  The last character in the font.
 * @param  width  The width of the font.
 * @param  height The height of the font.
 * @return       The new font.
 */
Font *new_Font (int first, int last, int width, int height)
{
    /* local variables */
    Font *font; /* the font to return */

    /* attempt to allocate memory */
    if (! (font = malloc (sizeof (Font))))
        return NULL;
    if (! (font->pixels = malloc
	   (width / 4 * height * (last - first + 1)))) {
        free (font);
        return NULL;
    }

    /* set the methods */
    font->destroy = destroy;
    font->clone = clone;
    font->write = write;
    font->put = put;
    font->get = get;
    font->recolour = recolour;

    /* set the rest of the attributes */
    font->first = first;
    font->last = last;
    font->width = width;
    font->height = height;

    /* return the font */
    return font;
}

/**
 * Construct a new font by reading it from an already open file.
 * @param  input   The input file handle.
 * @param  version CGALIB version of the font file.
 * @return         The loaded font.
 */
Font *read_Font (FILE *input, int version)
{
    Font *font; /* the font to return */
    char f; /* first character code read from file */
    char l; /* last character code read from file */
    char w; /* width of the font */
    char h; /* height of the font */

    /* read first and last character codes */
    if (! fread (&f, 1, 1, input))
        return NULL;
    if (! fread (&l, 1, 1, input))
        return NULL;
    if (version >= 2) {
	if (! fread (&w, 1, 1, input))
	    return NULL;
	if (! fread (&h, 1, 1, input))
	    return NULL;
    } else {
	w = 4;
	h = 8;
    }

    /* attempt to allocate memory */
    if (! (font = new_Font (f, l, w, h)))
        return NULL;

    /* attempt to read the pixel data */
    if (! fread (font->pixels, (w / 4 * h) * (l - f + 1), 1, input)) {
        free (font->pixels);
        free (font);
        return NULL;
    }

    /* set the other font information */
    font->first = f;
    font->last = l;
    font->width = w;
    font->height = h;

    /* return the font */
    return font;
}
