/*======================================================================
 * CGALIB version 2
 * x11 wrapper module.
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 18-Jan-2024.
 */

/*----------------------------------------------------------------------
 * Required headers.
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* compiler headers */
#include <unistd.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/XKBlib.h>

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var dis A pointer to the display. */
static Display *xdis = NULL;

/** @var xscreen The screen ID number. */
static int xscreen;

/** @var xwin The window. */
static Window xwin;

/** @var xgc The graphics context. */
static GC xgc;

/** @var status 1 if X11 initialised, 0 if not. */
static int status = 0;

/** @var cga_palette The full 16-colour CGA palette. */
static char *cga_palette[] = {
    "#000000", /* 0: black */
    "#0000aa", /* 1: low intensity blue */
    "#00aa00", /* 2: low intensity green */
    "#00aaaa", /* 3: low intensity cyan */
    "#aa0000", /* 4: low intensity red */
    "#aa00aa", /* 5: low intensity magenta */
    "#aa5500", /* 6: brown */
    "#aaaaaa", /* 7: low intensity white */
    "#555555", /* 8: grey */
    "#5555ff", /* 9: high intensity blue */
    "#55ff55", /* 10: high intensity green */
    "#55ffff", /* 11: high intensity cyan */
    "#ff5555", /* 12: high intensity red */
    "#ff55ff", /* 13: high intensity magenta */
    "#ffff55", /* 14: yellow */
    "#ffffff" /* 15: high intensity white */
};

/** @var fg_palettes The CGA 3-colour foreground palettes. */
static int fg_palettes[6][3] = {
    {2, 4, 6},
    {3, 5, 7},
    {3, 4, 7},
    {10, 12, 14},
    {11, 13, 15},
    {11, 12, 15}
};

/** @var keysyms Scancode to keysym conversion table. */
static int keysyms[] = {

    /* no key */
    0,

    /* keyboard top row */
    XK_Escape, /* 0x01 */
    XK_1,
    XK_2,
    XK_3,
    XK_4,
    XK_5,
    XK_6,
    XK_7,
    XK_8,
    XK_9,
    XK_0,
    XK_minus,
    XK_equal,
    XK_BackSpace,

    /* keyboard second row */
    XK_Tab,
    XK_q,
    XK_w,
    XK_e,
    XK_r,
    XK_t,
    XK_y,
    XK_u,
    XK_i,
    XK_o,
    XK_p,
    XK_bracketleft,
    XK_bracketright,
    XK_Return,

    /* keyboard third row */
    XK_Control_L,
    XK_a,
    XK_s,
    XK_d,
    XK_f,
    XK_g,
    XK_h,
    XK_j,
    XK_k,
    XK_l,
    XK_semicolon,
    XK_apostrophe,
    XK_numbersign, /* there is no XK_tilde */

    /* keyboard fourth row */
    XK_Shift_L,
    XK_backslash,
    XK_z,
    XK_x,
    XK_c,
    XK_v,
    XK_b,
    XK_n,
    XK_m,
    XK_comma,
    XK_period,
    XK_slash,
    XK_Shift_R,

    /* keyboard special keys */
    XK_KP_Multiply,
    XK_Alt_L,
    XK_space,
    XK_Caps_Lock,

    /* keyboard function keys */
    XK_F1,
    XK_F2,
    XK_F3,
    XK_F4,
    XK_F5,
    XK_F6,
    XK_F7,
    XK_F8,
    XK_F9,
    XK_F10,

    /* keyboard keypad */
    XK_Num_Lock,
    XK_Scroll_Lock,
    XK_KP_7,
    XK_KP_8,
    XK_KP_9,
    XK_KP_Subtract,
    XK_KP_4,
    XK_KP_5,
    XK_KP_6,
    XK_KP_Add,
    XK_KP_1,
    XK_KP_2,
    XK_KP_3,
    XK_KP_0,
    XK_KP_Decimal,
    XK_Sys_Req

};

/** @var keytrans Translation table for extended keys. */
static int keytrans[24][2] = {
    {XK_KP_Left, XK_KP_4},
    {XK_KP_Right, XK_KP_6},
    {XK_KP_Up, XK_KP_8},
    {XK_KP_Down, XK_KP_2},
    {XK_KP_Home, XK_KP_7},
    {XK_KP_End, XK_KP_1},
    {XK_KP_Page_Up, XK_KP_9},
    {XK_KP_Page_Down, XK_KP_3},
    {XK_KP_Insert, XK_KP_0},
    {XK_KP_Delete, XK_KP_Decimal},
    {XK_KP_Enter, XK_Return},
    {XK_KP_Begin, XK_KP_5},
    {XK_Alt_R, XK_Alt_L},
    {XK_Control_R, XK_Control_L},
    {XK_Left, XK_KP_4},
    {XK_Right, XK_KP_6},
    {XK_Up, XK_KP_8},
    {XK_Down, XK_KP_2},
    {XK_Home, XK_KP_7},
    {XK_End, XK_KP_1},
    {XK_Page_Up, XK_KP_9},
    {XK_Page_Down, XK_KP_3},
    {XK_Insert, XK_KP_0},
    {XK_Delete, XK_KP_Decimal}
};

/** @var x11_palette The X11 palette values */
static long int x11_palette[16];

/** @var foreground The foreground palette */
static int foreground = 4;

/** @var background The background colour */
static int background = 0;

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Initialise X11 and open a window.
 */
void cgalib_x11_init (void)
{
    unsigned long
	black,
	white;
    int i;
    XColor tmp;

    /* initialise the screen and display */
    xdis = XOpenDisplay ((char *) 0);
    xscreen = DefaultScreen (xdis);

    /* get black and white colours */
    black = BlackPixel (xdis, xscreen);
    white = WhitePixel (xdis, xscreen);

    /* create and initialise the window */
    xwin = XCreateSimpleWindow
	(xdis, DefaultRootWindow (xdis), 0, 0, 640, 400, 5, white,
	 black);
    XSetStandardProperties
	(xdis, xwin, "CGALIB2","CGALIB2", None, NULL, 0, NULL);
    XSelectInput
	(xdis, xwin,
	 FocusChangeMask
	 | ExposureMask
	 | ButtonPressMask
	 | KeyPressMask
	 | KeyReleaseMask);

    /* create the graphics context */
    xgc = XCreateGC(xdis, xwin, 0, 0);
    XSetBackground (xdis, xgc, white);
    XSetForeground (xdis, xgc, black);

    /* initialise the palette */
    for (i = 0; i < 16; ++i) {
	XParseColor
	    (xdis, DefaultColormap (xdis, xscreen), cga_palette[i],
	     &tmp);
	XAllocColor (xdis, DefaultColormap (xdis, xscreen), &tmp);
	x11_palette[i] = tmp.pixel;
    }

    /* make the window visible */
    XClearWindow (xdis, xwin);
    XMapRaised (xdis, xwin);
    XFlush (xdis);
    //XSync (xdis, True);
    sleep (1);

    /* remember that X is now initialised */
    status = 1;
}

/**
 * Close X11
 */
void cgalib_x11_close (void)
{
    XFreeGC (xdis, xgc);
    XDestroyWindow (xdis, xwin);
    XCloseDisplay (xdis);
    status = 0;
}

/**
 * Check status
 * @return 1 if X is initialised, 0 if not.
 */
int cgalib_x11_status (void)
{
    return status;
}

/**
 * Plot a pixel in a given colour.
 * @param x The x coordinate of the pixel.
 * @param y The y coordinate of the pixel.
 * @param c The 2-bit colour value of the pixel.
 */
void cgalib_x11_plot (int x, int y, int c)
{
    int i; /* colour index in the full palette */
    i = c ? fg_palettes[foreground][c - 1]
	: background;
    XSetForeground (xdis, xgc, x11_palette[i]);
    XFillRectangle (xdis, xwin, xgc, 2 * x, 2 * y, 2, 2);
}

/**
 * Force a screen update.
 */
void cgalib_x11_update (void)
{
    XFlush (xdis);
}

/**
 * Change the CGA palette.
 * @param inforeground The foresground palette.
 * @param inbackground The background colour.
 */
void cgalib_x11_palette (int inforeground, int inbackground)
{
    foreground = inforeground;
    background = inbackground;
}

/**
 * Process an X11 event and return any keyboard events.
 * @return A PC/XT compatible scan code value.
 */
int cgalib_x11_event (void)
{
    XEvent event; /* event object */
    int keysym, /* X11 keyboard symbol value */
	scancode = 0, /* BIOS keyboard scancode equivalent */
	i; /* key translate index */

    /* check for an event */
    if (! XPending (xdis)) return 0;
    XNextEvent (xdis, &event);

    /* interpret keyboard event */
    if (event.type == KeyPress || event.type == KeyRelease) {
	keysym = XkbKeycodeToKeysym
	    (xdis, event.xkey.keycode, 0, 0);
	for (i = 0; i < 24; ++i)
	    if (keytrans[i][0] == keysym) {
		keysym = keytrans[i][1];
		break;
	    }
	for (i = 0; i <= 0x54; ++i)
	    if (keysyms[i] == keysym)
		scancode = i;
	if (event.type == KeyRelease)
	    scancode |= 0x80;
    }

    /* interpret gain focus event */
    else if (event.type == FocusIn || event.type == FocusOut) {
	return 0xb8; /* gross hack - AltL released */
    }

    return scancode;
}
