/*======================================================================
 * CGALIB version 2
 * Screen data handling module.
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 06-Dec-2022.
 */

/*----------------------------------------------------------------------
 * Required headers.
 */

/* standard headers */
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "cgalib.h"
#include "screen_u.h"

/*----------------------------------------------------------------------
 * External function prototypes.
 */

/**
 * Draw a display list entry (present in hardware modules).
 * @param screen The screen to draw.
 * @param x      The x coordinate of the area.
 * @param y      The y coordinate of the area.
 * @param w      The width of the area.
 * @param h      The height of the area.
 */
void draw_display_entry
(Screen *screen, int x, int y, int w, int h);

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var current The current screen being displayed. */
static Screen *current = NULL;

/** @var displaylist A list of areas to update. */
DisplayList *displaylist = NULL;

/*----------------------------------------------------------------------
 * Public Method Level Definitions.
 */

/**
 * Destroy the screen and return to text mode.
 * @param screen The screen to destroy.
 */
static void destroy (Screen *screen)
{
    if (screen) {
	if (current == screen) {
	    cgalib_clear_dl ();
	    current = NULL;
	}
	if (screen->buffer)
	    screen->buffer->destroy (screen->buffer);
	free (screen);
    }
}

/**
 * Read the screen buffer from an already open file.
 * @param  screen The screen to read.
 * @param  format The file format:
 *                CGALIB_BITMAP is in bitmap format.
 *                CGALIB_BSAVE is in BSAVE format.
 * @param  input  The output file handle.
 * @return        1 on success, 0 on failure.
 */
static int read (Screen *screen, int format, FILE *input)
{
    Bitmap *bitmap; /* bitmap read from file */
    unsigned char header[7]; /* BSAVE header to read */
    unsigned char *gap; /* 192-byte gap for BSAVE files */
    int i, /* interlace number */
	y; /* y line to write */
    char *line; /* pointer to present line */

    /* read as a bitmap */
    if (format == CGALIB_BITMAP_FILE) {
	if (! (bitmap = read_Bitmap (input)))
	    return 0;
	screen->put (screen, bitmap, 0, 0, CGALIB_SET);
	bitmap->destroy (bitmap);
    }

    /* read as a BSAVE image */
    else {
	if (! fread (header, 1, 7, input))
	    return 0;
	for (i = 0; i < 2; ++i) {
	    for (y = 0; y < 199; y +=2) {
		line = screen->buffer->pixels + 80 * (i + y);
		if (! fread (line, 1, 80, input))
		    return 0;
	    }
	    gap = calloc (1, 192);
	    if (! i && ! fread (gap, 1, 192, input)) {
		free (gap);
		return 0;
	    }
	    free (gap);
	}
    }

    /* if we got here, everything went OK */
    cgalib_update_dl (screen, 0, 0, 320, 200);
    return 1;
}

/**
 * Write the screen to an already open file.
 * @param screen The screen to write.
 * @param  format The file format:
 *                CGALIB_BITMAP is in bitmap format.
 *                CGALIB_BSAVE is in BSAVE format.
 * @param output The output file handle.
 */
static int write (Screen *screen, int format, FILE *output)
{
    unsigned char header[7] = { /* BSAVE header */
	'\xfd', '\x00', '\xb8', '\x00', '\x00', '\x40', '\x3f'
    };
    unsigned char *gap; /* 192-byte gap for BSAVE files */
    int i, /* interlace number */
	y; /* y line to write */
    char *line; /* pointer to present line */

    /* write the screen buffer as a 320x200 bitmap */
    if (format == CGALIB_BITMAP_FILE)
	return screen->buffer->write (screen->buffer, output);

    /* write the screen buffer as a BSAVE */
    if (! fwrite (header, 1, 7, output))
	return 0;
    for (i = 0; i < 2; ++i) {
	for (y = 0; y < 199; y +=2) {
	    line = screen->buffer->pixels + 80 * (i + y);
	    if (! fwrite (line, 1, 80, output))
		return 0;
	}
	gap = calloc (1, 192);
	if (! i && ! fwrite (gap, 1, 192, output)) {
	    free (gap);
	    return 0;
	}
	free (gap);
    }

    /* if we got here, everything went OK */
    return 1;
}

/**
 * Show another screen.
 * @param screen The screen to show.
 */
static void show (Screen *screen)
{
    current = screen;
    cgalib_clear_dl ();
}

/**
 * Hide a screen.
 * @param screen The screen to hide.
 */
static void hide (Screen *screen)
{
    if (screen != current)
	return;
    cgalib_clear_dl ();
    current = NULL;
}

/**
 * Determine if a screen is shown.
 * @param screen The screen to check.
 */
static int shown (Screen *screen)
{
    return screen == current;
}

/**
 * Copy updates from the buffer to the screen.
 * @param screen The screen to update.
 */
static void update (Screen *screen)
{
    cgalib_clear_dl ();
}

/**
 * Set the screen palette.
 * @param screen     The screen for which to set the palette.
 * @param foreground The foreground colour set to choose:
 *                   0: low intensity red, green, brown;
 *                   1: low intensity cyan, magenta, white;
 *                   2: low intensity cyan, red, white;
 *                   3: high intensity red, green, yellow;
 *                   4: high intensity cyan, magenta, white;
 *                   5: high intensity cyan, red, white.
 * @param background The background colour to choose.
 */
static void palette (Screen *screen, int foreground, int background)
{
    /* update the palette details in the screen data */
    screen->foreground = foreground;
    screen->background = background;
}

/**
 * Get a bitmap from another bitmap.
 * @param  screen The screen to extract from.
 * @param  x      The x coordinate on the source bitmap.
 * @param  y      The y coordinate on the destination bitmap.
 * @param  w      The width of the bitmap to exstract.
 * @param  h      The height of the bitmap to extract.
 * @return        The new bitmap.
 */
static Bitmap *get (Screen *screen, int x, int y, int w, int h)
{
    return screen->buffer->get (screen->buffer, x, y, w, h);
}

/**
 * Put a bitmap on the screen.
 * @param  screen The destination screen.
 * @param  source The source bitmap.
 * @param  x      The x coordinate on the destination bitmap.
 * @param  y      The y coordinate on the destination bitmap.
 * @param  mode   The draw mode.
 */
static void put (Screen *screen, Bitmap *bitmap, int x, int y, int mode)
{
    screen->buffer->put (screen->buffer, bitmap, x, y, mode);
    cgalib_update_dl (screen, x, y, bitmap->width, bitmap->height);
}

/**
 * Put part of another bitmap onto this one.
 * @param screen The destination screen.
 * @param bitmap The source bitmap.
 * @param xd     The x coordinate on the destination bitmap.
 * @param yd     The y coordinate on the destination bitmap.
 * @param xs     The x coordinate on the source bitmap.
 * @param ys     The y coordinate on the source bitmap.
 * @param w      The width of the area to copy.
 * @param h      The height of the area to copy.
 * @param mode   The draw mode.
 */
static void transfer (Screen *screen, Bitmap *bitmap, int xd, int yd,
		      int xs, int ys, int w, int h, int draw)
{
    screen->buffer->transfer (screen->buffer, bitmap, xd, yd, xs, ys,
			      w, h, draw);
    cgalib_update_dl (screen, xd, yd, w, h);
}

/**
 * Draw a filled box onto the screen.
 * @param screen  The screen to draw onto.
 * @param x       The x coordinate of the left edge of the box.
 * @param y       The y coordinate of the top edit of the box.
 * @param width   The width of the box.
 * @param height  The height of the box.
 * @param pattern The 16-bit pattern for the box.
 */
static void box (Screen *screen, int x, int y, int width, int height,
		 unsigned int pattern)
{
    screen->buffer->ink = screen->ink;
    screen->buffer->paper = screen->paper;
    screen->buffer->box (screen->buffer, x, y, width, height, pattern);
    cgalib_update_dl (screen, x, y, width, height);
}

/**
 * Print a message onto the screen.
 * @param screen  The screen to print onto.
 * @param x       The x coordinate of the message.
 * @param y       The y coordinate of the message.
 * @param message The message to print.
 */
static void print (Screen *screen, int x, int y, char *message)
{
    screen->buffer->ink = screen->ink;
    screen->buffer->paper = screen->paper;
    screen->buffer->font = screen->font;
    screen->buffer->print (screen->buffer, x, y, message);
    cgalib_update_dl
	(screen, x, y, screen->font->width * strlen (message),
	 screen->font->height);
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Create a new screen object and initialise its data.
 * @param  mode    The CGA screen mode: 4, 5, or 6.
 * @param  isshown 1 to show the screen on creation, 0 not to do so.
 * @return         The new screen.
 */
Screen *cgalib_init (int mode, int isshown)
{
    Screen *screen; /* the screen to return */

    /* reserve memory for the screen */
    if (! (screen = malloc (sizeof (Screen))))
        return NULL;
    if (! (screen->buffer = new_Bitmap (320, 200))) {
	free (screen);
	return NULL;
    }

    /* initialise methods */
    screen->destroy = destroy;
    screen->write = write;
    screen->read = read;
    screen->show = show;
    screen->hide = hide;
    screen->shown = shown;
    screen->update = update;
    screen->palette = palette;
    screen->get = get;
    screen->put = put;
    screen->transfer = transfer;
    screen->box = box;
    screen->print = print;

    /* initialise attributes */
    screen->foreground = (mode == CGALIB_MEDIUM_MONO)
	? CGALIB_LIGHT_CYAN_RED_WHITE
	: CGALIB_LIGHT_CYAN_MAGENTA_WHITE;
    screen->background = CGALIB_BLACK;
    screen->ink = CGALIB_INK_WHITE;
    screen->paper = CGALIB_PAPER_BACKGROUND;
    screen->font = NULL;
    screen->updates = 0;

    /* if shown, set as the current screen */
    if (isshown) {
	current = screen;
	cgalib_clear_dl ();
    }

    /* return the new screen */
    return screen;
}

/**
 * Clear the display list without updating the display.
 */
void cgalib_clear_dl (void)
{
    DisplayList *entry, /* current display list entry */
	*next; /* next display list entry */
    entry = displaylist;
    while (entry) {
	next = entry->next;
	free (entry);
	entry = next;
    }
    displaylist = NULL;
}

/**
 * Add a screen area to the display list (areas to be updated).
 * @param screen The screen we should be showing.
 * @param x      The x coordinate of the area.
 * @param y      The y coordinate of the area.
 * @param w      The width of the area.
 * @param h      The height of the area.
 */
void cgalib_update_dl (Screen *screen, int x, int y,
				 int w, int h)
{
    DisplayList *entry, /* new entry for display list */
	*next; /* next entry from display next */

    /* ignore if the requested screen is not shown. */
    if (screen != current)
	return;

    /* update immediately if this screen has no display list */
    else if (! screen->updates) {
	draw_display_entry (screen, x, y, w, h);
	return;
    }

    /* reserve memory for new display list entry */
    if (! (entry = malloc (sizeof (DisplayList))))
	return;

    /* fill in the display list entry */
    entry->x = x;
    entry->y = y;
    entry->w = w;
    entry->h = h;
    entry->id = displaylist ? displaylist->id + 1 : 1;

    /* add to the display list */
    entry->next = displaylist;
    displaylist = entry;

    /* if the display list is too big, aggregate it */
    if (current && entry->id > current->updates) {

	/* aggregate and free the entries one by one */
	for (entry = displaylist->next;
	     entry;
	     entry = next) {

	    /* expand left of update area */
	    if (entry->x < displaylist->x) {
		displaylist->w += displaylist->x - entry->x;
		displaylist->x = entry->x;
	    }

	    /* expand top of update area */
	    if (entry->y < displaylist->y) {
		displaylist->h += displaylist->y - entry->y;
		displaylist->y = entry->y;
	    }

	    /* expand right of update area */
	    if (entry->x + entry->w > displaylist->x + displaylist->w)
		displaylist->w
		    += (entry->x + entry->w)
		    - (displaylist->x + displaylist->w);

	    /* expand bottom of update area */
	    if (entry->y + entry->h > displaylist->y + displaylist->h)
		displaylist->h
		    += (entry->y + entry->h)
		    - (displaylist->y + displaylist->h);

	    /* look at next entry */
	    next = entry->next;
	    free (entry);
	}

	/* truncate linked list */
	displaylist->next = NULL;
	displaylist->id = 1;
    }
}

/**
 * Get the start of the display list.
 * @return The start of the display list.
 */
DisplayList *cgalib_get_dl (void)
{
    return displaylist;
}
