/*======================================================================
 * CGALIB version 2
 * Demonstration Program
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 10-Dec-2022.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "cgalib.h"

/*----------------------------------------------------------------------
 * File Level Variables.
 */

/** @var scr is the screen data */
static Screen *screen;

/** @var fnt is an array of fonts */
static Font *font;

/** @var bit is an array of bitmaps */
static Bitmap *bitmap[16];

/** @var keyboard is the keyboard object */
static KeyHandler *keyboard;

/** @var speaker is the speaker object */
static Speaker *speaker;

/** @var effects are the sound effects */
static Effect *effects[3];

/** @var map is the map level as a 2D array */
static int map[18][10];

/** @var player_x is the player's X position */
static int player_x;

/** @var player_y is the player's Y position */
static int player_y;

/** @var map_img is the persistent map image */
static Bitmap *map_img;

/** @var level is the number of droids to generate. */
static int droids;

/** @var score is the game score. */
static int score;

/** @var effect_data is the data for the sound effects. */
static int effect_data[3][6] = {
    {EFFECT_RISE, 3, 36, 48, 4, 0}, /* start of teleport noise */
    {EFFECT_FALL, 1, 24, 24, 1, 0}, /* step noise */
    {EFFECT_FALL, 1, 36, 48, 7, 0} /* player death noise */
};

/*----------------------------------------------------------------------
 * Service Routines.
 */

/**
 * Error Handler.
 * @param errorlevel The error level to return to the OS.
 * @param message The message to print.
 */
void error_handler (int errorlevel, char *message)
{
    if (screen)
        screen->destroy (screen);
    if (keyboard)
	keyboard->destroy ();
    puts (message);
    exit (errorlevel);
}

/*----------------------------------------------------------------------
 * Level 3 Routines.
 */

/**
 * Show the score.
 */
void display_score (void) {
    char scoreboard[6];
    sprintf (scoreboard, "%05d", score);
    screen->print (screen, 300, 192, scoreboard);
}

/*
 * Level 2 Routines
 */

/**
 * Load and validate a font.
 * @param filename The name of the font file.
 * @return the loaded font.
 * The font file loaded by this function requires an 8-byte header,
 * consisting of the text CGA100F and a null byte. Then follows two
 * bytes determining the first and last character codes supported by
 * the font. After that is the font pixel data.
 */
static Font *load_font (char *filename)
{
    /* local variables */
    Font *font;
    FILE *fp;
    char header[8];

    /* attempt to open the file, and read and verify the header */
    if (! (fp = fopen (filename, "rb")))
        return NULL;
    else if (! fread (header, 8, 1, fp)) {
        fclose (fp);
        return NULL;
    } else if (strcmp (header, "CGA200F") && strcmp (header, "CGA100F")) {
        fclose (fp);
        return NULL;
    }

    /* read the font and return it */
    font = read_Font (fp, header[3] - '0');
    fclose (fp);
    return font;
}

/**
 * Load the bitmaps
 * @param filename The bimap file.
 */
static int load_bitmaps (char *filename)
{
    /* local variables */
    FILE *fp;
    char header[8];
    int c;

    /* attempt to open the file, and read and verify the header */
    if (! (fp = fopen (filename, "rb")))
        return 0;
    else if (! fread (header, 8, 1, fp)) {
        fclose (fp);
        return 0;
    } else if (strcmp (header, "CGA100B")) {
        fclose (fp);
        return 0;
    }

    /* read the five bitmaps */
    for (c = 0; c < 16; ++c)
        bitmap[c] = read_Bitmap (fp);
    fclose (fp);
    return 1;
}

/**
 * Move the player.
 * @return 1 if the level is finished, 0 if not.
 */
static int move_player (void)
{
    int finished, /* true if the player has won or died */
        xd, /* calculated x direction */
        yd, /* calculated y direction */
        x, /* teleport x coordinate */
        y, /* teleport y coordinate */
        key; /* key pressed */

    /* wait for a key */
    finished = 0;
    keyboard->wait ();
    key = keyboard->scancode ();
    screen->box (screen, 148, 192, 24, 8, CGALIB_BOX_BLANK);

    /* determine vertical direction */
    if (key == KEY_KP7 || key == KEY_KP8 || key == KEY_KP9
        || key == KEY_Q || key == KEY_W || key == KEY_E)
        yd = -1;
    else if (key == KEY_KP1 || key == KEY_KP2 || key == KEY_KP3
        || key == KEY_Z || key == KEY_X || key == KEY_C || key == KEY_S)
        yd = 1;
    else
        yd = 0;

    /* determine horizontal direction */
    if (key == KEY_KP7 || key == KEY_KP4 || key == KEY_KP1
        || key == KEY_Q || key == KEY_A || key == KEY_Z)
        xd = -1;
    else if (key == KEY_KP9 || key == KEY_KP6 || key == KEY_KP3
        || key == KEY_E || key == KEY_D || key == KEY_C)
        xd = 1;
    else
        xd = 0;

    /* move the player */
    if ((xd || yd)
        && player_x + xd >= 0 && player_x + xd <= 17
        && player_y + yd >= 0 && player_y + yd <= 9) {
        map[player_x][player_y] = 0;
        map_img->put (map_img, bitmap[0], 16 * player_x, 16 * player_y,
            CGALIB_SET);
        if (map[player_x + xd][player_y + yd] == 0) {
            map[player_x + xd][player_y + yd] = 1;
            map_img->put (map_img, bitmap[2], 16 * (player_x + xd),
                16 * (player_y + yd), CGALIB_AND);
            map_img->put (map_img, bitmap[1], 16 * (player_x + xd),
                16 * (player_y + yd), CGALIB_OR);
            player_x += xd;
            player_y += yd;
            screen->put (screen, map_img, 16, 16, CGALIB_SET);
        } else
            screen->print (screen, 148, 192, "OUCH!!");
    }

    /* player tried to move into a wall */
    else if (xd || yd)
        screen->print (screen, 148, 192, "OUCH!!");

    /* teleport the player */
    else if (key == KEY_SPACE) {

        /* find an empty place */
        do {
            x = rand () % 18;
            y = rand () % 10;
        } while (map[x][y]);

        /* move the player there */
        map[player_x][player_y] = 0;
        map[x][y] = 1;
        map_img->put (map_img, bitmap[0], 16 * player_x, 16 * player_y,
            CGALIB_SET);
        map_img->put (map_img, bitmap[2], 16 * x, 16 * y, CGALIB_AND);
        map_img->put (map_img, bitmap[1], 16 * x, 16 * y, CGALIB_OR);
        player_x = x;
        player_y = y;
        screen->put (screen, map_img, 16, 16, CGALIB_SET);

        /* apply and display the score penalty */
        score -= (score > droids / 2) ? droids / 2 : score;
        display_score ();
	effects[0]->play (effects[0]);
    }

    /* return true if finished */
    return finished;
}

/**
 * Move the droids.
 * @return 1 if the level is finished, 0 if not.
 */
int move_droids (void)
{
    /* local variables */
    int x, y, xd, yd, c, finished, new_map[18][10];

    /* initialise the new map */
    for (x = 0; x < 18; ++x)
        for (y = 0; y < 10; ++y)
            new_map[x][y] = map[x][y] * (map[x][y] != 3);

    /* move all the robots on the map */
    finished = 0;
    for (x = 0; x < 18; ++x)
        for (y = 0; y < 10; ++y)
            if (map[x][y] == 3) {

                /* update the old position with whatever will be here */
                map_img->put (map_img, bitmap[0], 16 * x, 16 * y, CGALIB_SET);
                map_img->put (map_img, bitmap[new_map[x][y] + 1], 16 * x,
                    16 * y, CGALIB_AND);
                map_img->put (map_img, bitmap[new_map[x][y]], 16 * x, 16 * y,
                    CGALIB_OR);

                /* work out the direction of movement */
                xd = (player_x > x) - (player_x < x);
                yd = (player_y > y) - (player_y < y);

                /* moving on to a blank square or the player */
                if (new_map[x + xd][y + yd] == 0
                    || new_map[x + xd][y + yd] == 1) {
                    new_map[x + xd][y + yd] = 3;
                    map_img->put (map_img, bitmap[4], 16 * (x + xd),
                        16 * (y + yd), CGALIB_AND);
                    map_img->put (map_img, bitmap[3], 16 * (x + xd),
                        16 * (y + yd), CGALIB_OR);
                    if (x + xd == player_x && y + yd == player_y)
                        finished = 1;
                }

                /* crashing into another droid */
                else if (new_map[x + xd][y + yd] == 3) {
                    new_map[x + xd][y + yd] = 5;
                    map_img->put (map_img, bitmap[6], 16 * (x + xd),
                        16 * (y + yd), CGALIB_AND);
                    map_img->put (map_img, bitmap[5], 16 * (x + xd),
                        16 * (y + yd), CGALIB_OR);
                    score += 2;
                }

                /* crashing into debris */
                else if (new_map[x + xd][y + yd] == 5)
                    score += 1;
            }
    
    /* copy the new map over the old one */
    c = 0;
    for (x = 0; x < 18; ++x)
        for (y = 0; y < 10; ++y) {
            map[x][y] = new_map[x][y];
            c += (map[x][y] == 3);
        }
    if (!c)
        finished = 1;

    /* redraw the screen and return */
    screen->put (screen, map_img, 16, 16, CGALIB_SET);
    display_score ();
    return finished;
}

/*----------------------------------------------------------------------
 * Level 1 Routines.
 */

/**
 * Initialise the screen.
 * @param mono True if mono mode was requested.
 */
void initialise_screen (int mode)
{
    /* local variables */
    int e; /* effect counter */

    /* initialise keyboard handler */
    if (! (keyboard = new_KeyHandler ()))
	error_handler (1, "Cannot initialise keyboard");

    /* initialise screen */
    if (! (screen = new_Screen (mode, CGALIB_SHOWN)))
        error_handler (1, "Cannot initialise graphics mode!");
    screen->palette (screen,
		     CGALIB_LIGHT_CYAN_MAGENTA_WHITE,
		     CGALIB_RED);

    /* load font and bitmaps */
    if (! (font = load_font ("fnt/future.fnt")))
        error_handler (1, "Cannot load font");
    if (! load_bitmaps ("bit/demo.bit"))
        error_handler (1, "Cannot load bitmaps");

    /* initialise sound effects */
    speaker = get_Speaker ();
    for (e = 0; e < 3; ++e) {
	effects[e] = new_Effect ();
	effects[e]->pattern = effect_data[e][0];
	effects[e]->repetitions = effect_data[e][1];
	effects[e]->low = effect_data[e][2];
	effects[e]->high = effect_data[e][3];
	effects[e]->duration = effect_data[e][4];
	effects[e]->pause = effect_data[e][5];
    }
}

/**
 * Initialise a game.
 */
void initialise_game (void)
{
    /* local variables */
    Bitmap *hidden; /* hidden copy of the screen */
    int x, y; /* X and Y game coordinates */
    int walls[8] = {8, 8, 8, 8, 8, 8, 14, 15}; /* random wall pieces */

    /* display please wait message */
    screen->font = font;
    screen->print (screen, 132, 192, "Please wait...");

    /* prepare the game field */
    if (! (hidden = new_Bitmap (320, 192)))
        error_handler (1, "Out of memory creating game screen");
    hidden->put (hidden, bitmap[7], 0, 0, CGALIB_SET);
    hidden->put (hidden, bitmap[9], 304, 0, CGALIB_SET);
    hidden->put (hidden, bitmap[12], 0, 176, CGALIB_SET);
    hidden->put (hidden, bitmap[13], 304, 176, CGALIB_SET);
    for (x = 1; x < 19; ++x) {
        hidden->put (hidden, bitmap[walls[rand () % 8]], 16 * x, 0, CGALIB_SET);
        hidden->put (hidden, bitmap[walls[rand () % 8]], 16 * x, 176,
            CGALIB_SET);
    }
    for (y = 1; y < 11; ++y) {
        hidden->put (hidden, bitmap[10], 0, 16 * y, CGALIB_SET);
        hidden->put (hidden, bitmap[11], 304, 16 * y, CGALIB_SET);
    }
    hidden->box (hidden, 16, 16, 288, 160, CGALIB_BOX_BLANK);
    screen->put (screen, hidden, 0, 0, CGALIB_SET);
    hidden->destroy (hidden);

    /* create the map image */
    if (! (map_img = new_Bitmap (288, 160)))
    error_handler (1, "Out of memory creating game map");

    /* initialise score and such */
    score = 0;
    droids = 12;
}

/**
 * Initialise a single play level
 */
void initialise_level (void)
{
    /* local variables */
    int x, y, /* temporary x and y coordinates */
        c; /* counter for droids */

    /* (re-)display please wait message */
    screen->font = font;
    screen->print (screen, 132, 192, "Please wait...");

    /* clear the map */
    for (x = 0; x < 18; ++x)
        for (y = 0; y < 10; ++y)
            map[x][y] = 0;

    /* place the player */
    player_x = rand () % 18;
    player_y = rand () % 10;
    map[player_x][player_y] = 1;

    /* place the droids */
    for (c = 0; c < droids; ++c) {
        do {
            x = rand () % 18;
            y = rand () % 10;
        } while (abs (x - player_x) + abs (y - player_y) < 8
            || map[x][y] != 0);
        map[x][y] = 3;
    }

    /* prepare the game map */
    for (x = 0; x < 18; ++x)
        for (y = 0; y < 10; ++y) {
            map_img->put (map_img, bitmap[0], 16 * x, 16 * y, CGALIB_SET);
            if (map[x][y]) {
                map_img->put (map_img, bitmap[map[x][y] + 1],
			      x * 16, y * 16, CGALIB_AND);
                map_img->put (map_img, bitmap[map[x][y]],
			      x * 16, y * 16, CGALIB_OR);
            }
        }
    screen->put (screen, map_img, 16, 16, CGALIB_SET);

    /* clear the "please wait" message and display the score */
    screen->box (screen, 132, 192, 56, 8, CGALIB_BOX_BLANK);
    display_score ();
}

/**
 * Play a single level through.
 */
void play_level (void)
{
    int finished; /* true if the player has won or died */

    /* move player and droids till the level is done */
    do {
        finished = move_player ();
        if (! finished)
            finished = move_droids ();
	if (! finished)
	    effects[1]->play (effects[1]);
    } while (! finished);
}

/**
 * Process the end of a level
 */
int end_level (void)
{
    /* display the victory or defeat message */
    if (map[player_x][player_y] == 1) {
        screen->print (screen, 132, 192, "Level cleared!");
	effects[0]->play (effects[0]);
    } else {
        screen->print (screen, 128, 192, "You are defeated");
	effects[2]->play (effects[2]);
    }
    keyboard->wait ();
    keyboard->ascii (); /* discard keypress */
    screen->box (screen, 128, 192, 64, 8, CGALIB_BOX_BLANK);
    ++droids;

    /* return true if player is dead */
    return map[player_x][player_y] != 1;
}

/**
 * Process the end of a game
 */
int end_game (void)
{
    /* local variables */
    int key; /* key that the player pressed */

    /* print the prompt, and get the key */
    screen->print (screen, 124, 192, "Play again (Y/N) ?");
    do {
	keyboard->wait ();
        key = keyboard->ascii ();
    } while (key != 'Y' && key != 'y' && key != 'N' && key != 'n');
    screen->box (screen, 124, 192, 72, 8, CGALIB_BOX_BLANK);

    /* return true if key is 'N' to quit */
    return (key == 'N' || key == 'n');
}

/**
 * Clean up when the program is finished.
 */
void clean_up (void)
{
    int c;
    map_img->destroy (map_img);
    keyboard->destroy ();
    font->destroy (font);
    screen->destroy (screen);
    for (c = 0; c < 16; ++c)
	bitmap[c]->destroy (bitmap[c]);
    for (c = 0; c < 3; ++c)
	effects[c]->destroy (effects[c]);
    speaker->destroy ();
}

/*----------------------------------------------------------------------
 * Top Level Routine.
 */

/**
 * Main program.
 * @param argc The number of command line argumets.
 * @param argv The command line arguments.
 * @return     0 for success, !0 for error.
 */
int main (int argc, char **argv)
{
    /* local variables */
    int dead, quit;

    /* initialise */
    initialise_screen ((argc == 2 && ! strcmp (argv[1], "-m")) ? 6 : 4);
    srand (time (NULL));

    /* game loop */
    do {
        initialise_game ();
        do {
            initialise_level ();
            play_level ();
            dead = end_level ();
        } while (! dead);
        quit = end_game ();
    } while (! quit);
    clean_up ();
    return 0;
}
