/*======================================================================
 * CGALIB version 2
 * Bitmap handling module.
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 06-Dec-2022.
 */

/*----------------------------------------------------------------------
 * Required Headers.
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "cgalib.h"

/*----------------------------------------------------------------------
 * Public Method Level Definitions.
 */

/**
 * Destroy the bitmap and return to text mode.
 * @param bitmap The bitmap to destroy.
 */
static void destroy (Bitmap *bitmap)
{
    if (bitmap) {
	if (bitmap->pixels)
	    free (bitmap->pixels);
	free (bitmap);
    }
}

/**
 * Clone the bitmap.
 * @param  bitmap The bitmap to clone.
 * @return        The new bitmap.
 */
static Bitmap *clone (Bitmap *bitmap)
{
    Bitmap *newbitmap;

    /* reserve memory */
    if (! (newbitmap = new_Bitmap (bitmap->width, bitmap->height)))
        return NULL;

    /* initialise the data */
    newbitmap->ink = bitmap->ink;
    newbitmap->paper = bitmap->paper;
    newbitmap->font = bitmap->font;
    memcpy
	(newbitmap->pixels, bitmap->pixels,
	 bitmap->width / 4 * bitmap->height);

    /* return the bitmap */
    return newbitmap;
}

/**
 * Write the bitmap to an already open file.
 * @param  bitmap The bitmap to write.
 * @param  output The output file handle.
 * @return        1 on success, 0 on failure.
 */
static int write (Bitmap *bitmap, FILE *output)
{
    if (! fwrite (&bitmap->width, 2, 1, output))
        return 0;
    if (! fwrite (&bitmap->height, 2, 1, output))
        return 0;
    if (! fwrite
	(bitmap->pixels, bitmap->width / 4 * bitmap->height, 1, output))
        return 0;
    return 1;
}

/**
 * Get a bitmap from another bitmap.
 * @param  bitmap The bitmap to extract from.
 * @param  x      The x coordinate on the source bitmap.
 * @param  y      The y coordinate on the destination bitmap.
 * @param  w      The width of the bitmap to exstract.
 * @param  h      The height of the bitmap to extract.
 * @return        The new bitmap.
 */
static Bitmap *get (Bitmap *bitmap, int x, int y, int w, int h)
{
    char *s; /* address to copy data from */
    char *d; /* address to copy data to */
    int r; /* row counter */
    Bitmap *newbitmap; /* destination bitmap */
    if (! (newbitmap = new_Bitmap (w, h))) return NULL;
    for (r = 0; r < newbitmap->height; ++r) {
        s = bitmap->pixels + x / 4 + (y + r) * (bitmap->width / 4);
        d = newbitmap->pixels + newbitmap->width / 4 * r;
        memcpy (d, s, newbitmap->width / 4);
    }
    return newbitmap;
}

/**
 * Put another bitmap onto this one.
 * @param  bitmap The destination bitmap.
 * @param  source The source bitmap.
 * @param  x      The x coordinate on the destination bitmap.
 * @param  y      The y coordinate on the destination bitmap.
 * @param  mode   The draw mode.
 */
static void put (Bitmap *bitmap, Bitmap *source, int x, int y, int mode)
{
    /* local variables */
    char *d; /* address to copy data to */
    char *s; /* address to copy data from */
    int r; /* row counter */
    int b; /* byte counter */

    /* copy the pixels */
    for (r = 0; r < source->height; ++r)

        /* DRAW_PSET can be copied by a quicker method */
        if (mode == CGALIB_SET) {
            d = bitmap->pixels + x / 4 + (y + r) * (bitmap->width / 4);
            s = source->pixels + source->width / 4 * r;
            memcpy (d, s, source->width / 4);
        }
        
        /* the other draw operations need doing byte by byte */
        else 
            for (b = 0; b < source->width / 4; ++b) {
                d = bitmap->pixels + b + x / 4
                    + (y + r) * (bitmap->width / 4);
                s = b + source->pixels + source->width / 4 * r;
                switch (mode) {
                    case CGALIB_RESET:
                        *d = ~*s;
                        break;
                    case CGALIB_AND:
                        *d &= *s;
                        break;
                    case CGALIB_OR:
                        *d |= *s;
                        break;
                    case CGALIB_XOR:
                        *d ^= *s;
                        break;
                }
            }
}

/**
 * Put part of another bitmap onto this one.
 * @param bitmap The destination bitmap.
 * @param source The source bitmap.
 * @param xd     The x coordinate on the destination bitmap.
 * @param yd     The y coordinate on the destination bitmap.
 * @param xs     The x coordinate on the source bitmap.
 * @param ys     The y coordinate on the source bitmap.
 * @param w      The width of the area to copy.
 * @param h      The height of the area to copy.
 * @param mode   The draw mode.
 */
static void transfer (Bitmap *bitmap, Bitmap *source, int xd, int yd,
		      int xs, int ys, int w, int h, int mode)
{
    /* local variables */
    char *d; /* address to copy data to */
    char *s; /* address to copy data from */
    int r; /* row counter */
    int b; /* byte counter */

    /* copy the pixels */
    for (r = 0; r < h; ++r)

        /* DRAW_PSET can be copied by a quicker method */
        if (mode == CGALIB_SET) {
            d = bitmap->pixels + xd / 4 + (yd + r) * (bitmap->width / 4);
            s = source->pixels + (xs / 4)
		+ (source->width / 4) * (ys + r);
            memcpy (d, s, w / 4);
        }
        
        /* the other draw operations need doing byte by byte */
        else 
            for (b = 0; b < w / 4; ++b) {
                d = bitmap->pixels + b + xd / 4
                    + (yd + r) * (bitmap->width / 4);
                s = b + source->pixels + (xs / 4)
		    + (source->width) / 4 * (ys + r);
                switch (mode) {
                    case CGALIB_RESET:
                        *d = ~*s;
                        break;
                    case CGALIB_AND:
                        *d &= *s;
                        break;
                    case CGALIB_OR:
                        *d |= *s;
                        break;
                    case CGALIB_XOR:
                        *d ^= *s;
                        break;
                }
            }
}

/**
 * Draw a filled box onto the bitmap.
 * @param bitmap  The bitmap to draw onto.
 * @param x       The x coordinate of the left edge of the box.
 * @param y       The y coordinate of the top edit of the box.
 * @param width   The width of the box.
 * @param height  The height of the box.
 * @param pattern The 16-bit pattern for the box.
 */
static void box (Bitmap *bitmap, int x, int y, int width, int height,
		 unsigned int pattern)
{
    char p, /* pattern rotated */
	v, /* value to write to the screen */
	*d; /* address to write to */
   int r; /* row counter */
 
   /* fill each individual row */
   for (r = 0; r < height; ++r) {
       p = pattern >> (8 * (r & 1)) & 0xff;
       v = p & (bitmap->ink * 0x55);
       v |= (p ^ 0xff) & (bitmap->paper * 0x55);
       d = bitmap->pixels + x / 4 + (y + r) * (bitmap->width / 4);
       memset (d, v, width / 4);
   }
}

/**
 * Print a message onto the bitmap.
 * @param bitmap  The bitmap to print onto.
 * @param x       The x coordinate of the message.
 * @param y       The y coordinate of the message.
 * @param message The message to print.
 */
static void print (Bitmap *bitmap, int x, int y, char *message)
{
    int b; /* character pointer */
    int r; /* row of character */
    int c; /* column of caracter */
    int s; /* size of a character in bytes */
    char v; /* value of byte in character */
    char *d; /* pointer to destination byte on screen */

    /* only try to print if a font is selected */
    if (! bitmap->font) return;

    /* print each character */
    s = bitmap->font->width / 4 * bitmap->font->height;
    for (b = 0; message[b]; ++b)
        for (r = 0; r < bitmap->font->height; ++r)
	    for (c = 0; c < bitmap->font->width / 4; ++c) {
		d = bitmap->pixels
		    + c + b * bitmap->font->width / 4 + x / 4
		    + (y + r) * (bitmap->width / 4);
		v = bitmap->font->pixels
		    [r * bitmap->font->width / 4 + c
		     + s * (message[b] - bitmap->font->first)];
		if (bitmap->ink != CGALIB_INK_WHITE ||
		    bitmap->paper != CGALIB_INK_BACKGROUND)
		    v = (v & 0x55 * bitmap->ink)
			| ((0xff ^ v) & 0x55 * bitmap->paper);
		*d = v;
	    }
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Construct a new bitmap.
 * @param  width  The width of the bitmap.
 * @param  height The height of the bitmap.
 * @return      The new bitmap.
 */
Bitmap *new_Bitmap (int width, int height)
{
    /* local variables */
    Bitmap *bitmap;

    /* reserve memory */
    if (! (bitmap = malloc (sizeof (Bitmap))))
        return NULL;
    if (! (bitmap->pixels = calloc (1, width / 4 * height))) {
        free (bitmap);
        return NULL;
    }

    /* initialise the methods */
    bitmap->destroy = destroy;
    bitmap->clone = clone;
    bitmap->write = write;
    bitmap->get = get;
    bitmap->put = put;
    bitmap->transfer = transfer;
    bitmap->box = box;
    bitmap->print = print;

    /* initialise the remaining attributes */
    bitmap->width = width;
    bitmap->height = height;
    bitmap->ink = 3;
    bitmap->paper = 0;
    bitmap->font = NULL;

    /* return the bitmap */
    return bitmap;
}

/**
 * Construct a new bitmap read from an already open file.
 * @param  input The input file handle.
 * @return       The new bitmap.
 */
Bitmap *read_Bitmap (FILE *input)
{
    Bitmap *bitmap; /* the bitmap to return */
    int w = 0; /* the width read from a file */
    int h = 0; /* the height read from a file */

    /* attempt to read the width and height */
    if (! fread (&w, 2, 1, input))
        return NULL;
    if (! fread (&h, 2, 1, input))
        return NULL;
    
    /* create a blank bitmap of that size */
    if (! (bitmap = new_Bitmap (w, h)))
        return NULL;

    /* attempt to read the pixels */
    if (! fread (bitmap->pixels, w / 4 * h, 1, input)) {
        free (bitmap->pixels);
        free (bitmap);
        return NULL;
    }

    /* return the bitmap */
    return bitmap;
}
