/*======================================================================
 * CGALib - Watcom C Version.
 * Sound Effects Editor Program.
 * 
 * Released as Public Domain by Damian Gareth Walker, 2025.
 * Created 09-Sep-2025.
 */

/*----------------------------------------------------------------------
 * Required Headers.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

/* project headers */
#include "cgalib.h"

/*----------------------------------------------------------------------
 * File Level Variables.
 */

/** @var scr The screen object. */
static Screen *scr = NULL;

/** @var mainfont The main font object. */
static Font *mainfont = NULL;

/** @var labelfont The label font object. */
static Font *labelfont = NULL;

/** @var highfont The highlight font object. */
static Font *highfont = NULL;

/** @var keys The keyboard handler. */
static KeyHandler *keys;

/** @var bits The program's bitmaps. */
static Bitmap *bits[3];

/** @var fx The edited sound effects. */
static Effect *effects[12];

/** @var fxcursor The sound effect cursor position. */
static int fxcursor;

/** @var clipboard The sound effect most recently copied. */
static int clipboard;

/** @var mono 1 if the user wants a monochrome screen. */
static int mono;

/** @var filename The filename. */
static char filename[128];

/** @var patnames Names of the patterns. */
static char *patnames[] = {
    "Noise",
    "Fall",
    "Rise",
    "Step down",
    "Step up"
};

/*----------------------------------------------------------------------
 * Service Routines.
 */

/**
 * Error Handler.
 * @param errorlevel is the error level to return to the OS.
 * @param message is the message to print.
 */
void error_handler (int errorlevel, char *message)
{
    if (scr)
        scr->destroy (scr);
    puts (message);
    exit (errorlevel);
}

/**
 * Copy the attributes from one effect to another.
 * @param dest   The destination effect.
 * @param source The source effect.
 */
void copy_effect (Effect *dest, Effect *source)
{
    dest->pattern = source->pattern;
    dest->repetitions = source->repetitions;
    dest->low = source->low;
    dest->high = source->high;
    dest->duration = source->duration;
    dest->pause = source->pause;
}

/**
 * Get a number.
 * @param address Pointer to the number store.
 * @param x       X location on the screen.
 * @param y       Y location on the screen.
 * @param min     Minimum value.
 * @param max     Maximum value.
 * @param step    Step of increase/decrease.
 */
void getnum (int *address, int x, int y, int min, int max, int step)
{
    int key; /* keypress */
    char numstr[6], /* number expressed as a string */
	*format; /* number format */

    /* determine number format based on maximum value */
    if (max < 10)
	format = "%1d";
    else if (max < 100)
	format = "%02d";
    else if (max < 1000)
	format = "%03d";
    else if (max < 10000)
	format = "%04d";
    else
	format = "%05d";

    /* main entry loop */
    scr->font = highfont;
    do {
	sprintf (numstr, format, *address);
	scr->print (scr, x, y, numstr);
	scr->update (scr);
	keys->wait ();
	key = keys->ascii ();
	if (key == 11 && *address < max)
	    *address += step;
	else if (key == 10 && *address > min)
	    *address -= step;
	else if ((key >= '0' && key <= '9') &&
		 key - '0' + *address * 10 <= max)
	    *address = key - '0' + *address * 10;
	else if (key == 8)
	    *address /= 10;
    } while (key != 13);
    scr->font = mainfont;
    scr->print (scr, x, y, numstr);
}

/**
 * Print a numeric value on the screen.
 * @param x     Location to print to, x coordinate.
 * @param y     Location to print to, y coordinate.
 * @param value The value to print.
 */
static void printval (int x, int y, int value)
{
    char numstr[4];
    sprintf (numstr, "%03d", value);
    scr->print (scr, x, y, numstr);
}

/*----------------------------------------------------------------------
 * Level 3 Routines
 */

/**
 * Load makefx's own bitmaps.
 * @return 1 if successful, 0 on failure.
 */
static int load_prog_bitmaps (void)
{
    FILE *fp; /* file pointer */
    char header[8]; /* bitmap file header */
    int c; /* counter */
    Bitmap *bit = NULL; /* pointer to temporary bitmap */

    /* attempt to open the file, and read and verify the header */
    if (! (fp = fopen ("bit/makefx.bit", "rb")))
        return 0;
    else if (! fread (header, 8, 1, fp)) {
        fclose (fp);
        return 0;
    } else if (strcmp (header, "CGA100B")) {
        fclose (fp);
        return 0;
    }

    /* read the bitmaps */
    c = 0;
    while (c < 3 && (bit = read_Bitmap (fp))) {
	bits[c] = bit;
	++c;
    }

    /* return */
    fclose (fp);
    return 1;
}

/**
 * Load the font and create recoloured versions.
 * @param filename is the name of the font file.
 * @returns the loaded font.
 * The font file loaded by this function requires an 8-byte header,
 * consisting of the text CGA100F and a null byte. Then follows two
 * bytes determining the first and last character codes supported by
 * the font. After that is the font pixel data.
 */
static int load_font (char *filename)
{
    FILE *fp; /* file pointer */
    char header[8]; /* font file header */

    /* attempt to open the file, and read and verify the header */
    if (! (fp = fopen (filename, "rb")))
        return 0;
    else if (! fread (header, 8, 1, fp)) {
        fclose (fp);
        return 0;
    } else if
	    (strcmp (header, "CGA100F")
	     && strcmp (header, "CGA200F")) {
        fclose (fp);
        return 0;
    }

    /* read the font and create recoloured versions */
    mainfont = read_Font (fp, header[3] - '0');
    labelfont = mainfont->clone (mainfont);
    labelfont->recolour (labelfont, 1, 0);
    highfont = mainfont->clone (mainfont);
    highfont->recolour (highfont, 3, 2);
    fclose (fp);
    return 1;
}

/**
 * Load the sound effects to edit
 * @param filename The sound effect file.
 * @return         1 if successful, 0 on failure.
 */
static int load_file_effects (char *filename)
{
    FILE *fp; /* file pointer */
    char header[8]; /* effect file header */
    int c; /* counter */
    Effect *effect = NULL; /* pointer to temporary effect */

    /* attempt to open the file, and read and verify the header */
    if (! (fp = fopen (filename, "rb")))
        return 0;
    else if (! fread (header, 8, 1, fp)) {
        fclose (fp);
        return 0;
    } else if (strcmp (header, "CGA100E")) {
        fclose (fp);
        return 0;
    }

    /* read the effects */
    if (! (effect = new_Effect ())) {
	fclose (fp);
	return 0;
    }
    c = 0;
    while (c < 12 && (effect->read (effect, fp))) {
	if (! (effects[c] = new_Effect ())) {
	    fclose (fp);
	    return 0;
	}
	copy_effect (effects[c], effect);
	++c;
    }
    while (c < 12)
	effects[c++] = NULL;

    /* return */
    fclose (fp);
    effect->destroy (effect);
    return 1;
}

/**
 * Show a sound effect.
 * @param f The sound effect ID.
 */
static void showeffect (int f)
{
    int c;
    if (f == fxcursor) {
	scr->put (scr, bits[0], 0, 16 * f, CGALIB_SET);
	for (c = 1; c < 19; ++c)
	    scr->put (scr, bits[1], 16 * c, 16 * f, CGALIB_SET);
	scr->put (scr, bits[2], 304, 16 * f, CGALIB_SET);
    } else
	scr->box (scr, 0, 16 * f, 320, 16, CGALIB_BOX_BLANK);
    if (effects[f]) {
	scr->font = labelfont;
	scr->print (scr, 12, 16 * f + 4, "Pattern:");
	scr->print (scr, 88, 16 * f + 4, "Repeats:");
	scr->print (scr, 140, 16 * f + 4, "Low:");
	scr->print (scr, 176, 16 * f + 4, "High:");
	scr->print (scr, 216, 16 * f + 4, "Duration:");
	scr->print (scr, 272, 16 * f + 4, "Pause:");
	scr->font = mainfont;
	scr->print (scr, 44, 16 * f + 4, patnames[effects[f]->pattern]);
	printval (120, 16 * f + 4, effects[f]->repetitions);
	printval (156, 16 * f + 4, effects[f]->low);
	printval (196, 16 * f + 4, effects[f]->high);
	printval (252, 16 * f + 4, effects[f]->duration);
	printval (296, 16 * f + 4, effects[f]->pause);
    }
}

/**
 * Clear the current sound effect.
 */
static void clear (void)
{
    effects[fxcursor]->pattern = EFFECT_NOISE;
    effects[fxcursor]->repetitions = 0;
    effects[fxcursor]->low = 0;
    effects[fxcursor]->high = 0;
    effects[fxcursor]->duration = 0;
    effects[fxcursor]->pause = 0;
}

/**
 * Allow the player to select a pattern.
 */
static void getpattern (void)
{
    int key; /* keypress */
    char output[10]; /* number expressed as a string */

    /* main entry loop */
    scr->font = highfont;
    do {
	sprintf (output, "%-9.9s", patnames[effects[fxcursor]->pattern]);
	scr->print (scr, 44, 16 * fxcursor + 4, output);
	scr->update (scr);
	keys->wait ();
	key = keys->ascii ();
	if (key == 11 && effects[fxcursor]->pattern < EFFECT_STEP_UP)
	    ++effects[fxcursor]->pattern;
	else if (key == 10 && effects[fxcursor]->pattern > EFFECT_NOISE)
	    --effects[fxcursor]->pattern;
    } while (key != 13);
    scr->font = mainfont;
    scr->print (scr, 44, 16 * fxcursor + 4, output);
}

/*----------------------------------------------------------------------
 * Level 2 Routines.
 */

static void initialise_args (int argc, char **argv)
{
    while (argc-- > 1)
	if (! strcmp (argv[argc], "-m") ||
	    ! strcmp (argv[argc], "-M"))
	    mono = 1;
	else
	    strcpy (filename, argv[argc]);
}

/**
 * Initialise the screen.
 * @param mono is true if mono mode was requested.
 */
void initialise_screen (int mono)
{
    int e; /* effect counter */

    /* initialise screen and assets */
    if (! (scr = new_Screen (mono ? 6 : 4, CGALIB_SHOWN)))
        error_handler (1, "Cannot initialise graphics mode!");
    if (! (load_font ("fnt/present.fnt")))
        error_handler (1, "Cannot load font");
    if (! (load_prog_bitmaps ()))
	error_handler (1, "Cannot load program bitmaps");
    scr->font = mainfont;

    /* initial screen display */
    scr->updates = 1;
    for (e = 0; e < 12; ++e)
	showeffect (e);
}

/**
 * Paste the copied effect.
 */
static void paste (void)
{
    if (! effects[fxcursor])
	effects[fxcursor] = new_Effect ();
    copy_effect (effects[fxcursor], effects[clipboard]);
    showeffect (fxcursor);
}

/**
 * Insert a new bitmap.
 */
static void insert (void)
{
    /* create and clear the effect */
    if (effects[fxcursor])
	return;
    effects[fxcursor] = new_Effect ();
    clear ();

    /* update the screen */
    showeffect (fxcursor);
    scr->box (scr, 0, 192, 320, 8, CGALIB_BOX_BLANK);
}

/**
 * Change the values of the current effect.
 */
static void changevalues (void)
{
    getpattern ();
    getnum
	(&effects[fxcursor]->repetitions,
	 120, 16 * fxcursor + 4,
	 0, 255, 1);
    getnum
	(&effects[fxcursor]->low,
	 156, 16 * fxcursor + 4,
	 0, 255, 1);
    getnum
	(&effects[fxcursor]->high,
	 196, 16 * fxcursor + 4,
	 0, 255, 1);
    getnum
	(&effects[fxcursor]->duration,
	 252, 16 * fxcursor + 4,
	 0, 255, 1);
    getnum
	(&effects[fxcursor]->pause,
	 296, 16 * fxcursor + 4,
	 0, 255, 1);
}

/**
 * Save the effects.
 */
static void saveeffects (void)
{
    int key, /* keypress */
	len, /* length of string */
	c; /* effect counter */
    FILE *fp; /* file pointer */

    /* get the filename */
    if (! *filename) {
	scr->print (scr, 0, 192, "Filename: ");
	scr->update (scr);
	do {
	    keys->wait ();
	    key = keys->ascii ();
	    if (key >= ' ' && key <= '~') {
		len = strlen(filename);
		filename[len] = key;
		filename[len + 1] = '\0';
		scr->print (scr, 40, 192, filename);
	    } else if (key == 8 && *filename) {
		len = strlen(filename);
		filename[len - 1] = '\0';
		len = strlen(filename);
		scr->print (scr, 40 + 4 * len, 192, " ");
	    }
	    scr->update (scr);
	} while (key != 13);
	if (*filename && ! strchr (filename, '.'))
	    strcat (filename, ".eff");
    }
    if (! *filename) return;

    /* save the file */
    if (! (fp = fopen (filename, "wb")))
	return;
    fwrite ("CGA100E", 8, 1, fp);
    for (c = 0; c < 12; ++c)
	if (effects[c])
	    effects[c]->write (effects[c], fp);
    fclose (fp);
}

/*----------------------------------------------------------------------
 * Level 1 Routines.
 */

/**
 * Initialise the program.
 * @param mono 1 for monochrome, 0 for colour.
 * @return     1 if successful, 0 on failure.
 */
static int initialise (int argc, char **argv)
{
    initialise_args (argc, argv);
    if (*filename && ! load_file_effects (filename))
	error_handler (2, "Cannot load effects");
    initialise_screen (mono);
    keys = new_KeyHandler ();
    return 1;
}

/**
 * Main program loop.
 * @return 0 when finished, 1 to continue.
 */
static int main_program (void)
{
    int key; /* key press */

    /* get keypress */
    scr->update (scr);
    keys->wait ();
    key = keys->scancode ();

    /* cursor up */
    if (key == KEY_KP8 && fxcursor > 0) {
	showeffect (fxcursor--);
	showeffect (fxcursor);
    }

    /* cursor down */
    else if (key == KEY_KP2 && fxcursor < 11) {
	showeffect (fxcursor++);
	showeffect (fxcursor);
    }

    /* ENTER - alter sound effect values */
    else if (effects[fxcursor] && key == KEY_ENTER)
	changevalues ();

    /* C - copy */
    else if (effects[fxcursor] && key == KEY_C)
	clipboard = fxcursor;

    /* P - paste */
    else if (effects[clipboard] && key == KEY_P)
	paste ();

    /* X - clear */
    else if (effects[fxcursor] && key == KEY_X) {
	clear ();
	showeffect (fxcursor);
    }

    /* INS - insert new effect */
    else if (! effects[fxcursor] && key == KEY_KP0)
	insert ();

    /* DEL - delete effect */
    else if (effects[fxcursor] && key == KEY_KPSTOP) {
	effects[fxcursor]->destroy (effects[fxcursor]);
	effects[fxcursor] = NULL;
	showeffect (fxcursor);
    }

    /* SPACE - play back the sound */
    else if (effects[fxcursor] && key == KEY_SPACE)
	effects[fxcursor]->play (effects[fxcursor]);

    /* ESC (quit) */
    else if (key == KEY_ESC)
	return 0;

    /* return */
    return 1;
}

/**
 * End the program.
 */
static void end_program (void)
{
    int c; /* counter */
    Speaker *speaker; /* check speaker */
    saveeffects ();
    for (c = 0; c < 12; ++c)
	if (effects[c])
	    effects[c]->destroy (effects[c]);
    for (c = 0; c < 3; ++c)
	bits[c]->destroy (bits[c]);
    if ((speaker = get_Speaker ()))
	speaker->destroy ();
    keys->destroy ();
    mainfont->destroy (mainfont);
    labelfont->destroy (labelfont);
    highfont->destroy (highfont);
    scr->destroy (scr);
}

/*----------------------------------------------------------------------
 * Top Level Routine.
 */

/**
 * Main program.
 * @param argc is the number of command line argumets.
 * @param argv is the command line arguments.
 * No return value as exit () is used to terminate abnormally.
 */
int main (int argc, char **argv)
{
    if (initialise (argc, argv)) {
	while (main_program ());
	end_program ();
    }
    return 0;
}
