/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 09-May-2024.
 *
 * Map UI Module.
 */

/*----------------------------------------------------------------------
 * Included Headers.
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* project headers */
#include "sccc.h"
#include "display.h"
#include "controls.h"
#include "game.h"
#include "scores.h"
#include "ui.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var ui A pointer to the UI structure. */
static UI *ui;

/** @var display The display object. */
static Display *display;

/** @var controls The controls object. */
static Controls *controls;

/** @var game The game object. */
static Game *game;

/** @var missionscores An array of scores. */
static int missionscores[6];

/** @var performance Detailed performance scores. */
static int performance[10];

/** @var victorymenu The victory screen menu. */
static char *victorymenu[] = {
    "Cancel",
    "View scores",
    "New game",
    "Quit game"
};

/** @var victorykeys Shortcut keys for victory screen. */
static char *victorykeys = " VNQ";

/** @enum VictoryMenuID Identifiers for victory menu options. */
typedef enum {
    VICTMENU_CANCEL,
    VICTMENU_SCORES,
    VICTMENU_NEWGAME,
    VICTMENU_QUITGAME,
    VICTMENU_COUNT
} VictoryMenuID;

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Post a high score before proceeding to the high score table.
 */
static void postscore (void)
{
    Scores *scores; /* scores object */
    scores = getscores ();
    scores->submit (scores, game);
    game->state = STATE_SCORES;
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Initialise the UI.
 * @return   1 on success, 0 on failure.
 */
static int init (void)
{
    char briefing[384]; /* buffer for the briefing */

    /* initialise module-wide variables */
    display = getdisplay ();
    controls = getcontrols ();
    game = getgame ();

    /* work out the game scores */
    game->calcscore (game, missionscores, performance);

    /* show initial display */
    display->showbackground ();
    display->showmissionscores (game, missionscores, performance, 1);
    display->showtitle (getmissiontype (game->type));
    game->briefing (game, briefing);
    display->typeset (briefing);
    display->update ();

    /* success */
    return 1;
}

/**
 * Operate the UI.
 * @param ui The UI object.
 * @return   1 on success, 0 on failure.
 */
static int operate (void)
{
    int option; /* menu option chosen */
    do {

	/* wait for input */
	do {
	    controls->poll ();
	    if (controls->fire)
		option = display->menu
		    (VICTMENU_COUNT,
		     victorymenu,
		     1);
	    else if (strchr (victorykeys, toupper (controls->ascii)))
		option
		    = strchr (victorykeys, toupper (controls->ascii))
		    - victorykeys;
	    else
		option = 0;
	} while (! option);

	/* act upon input */
	switch (option) {
	case VICTMENU_SCORES:
	    postscore ();
	    return game->state;
	case VICTMENU_NEWGAME:
	    return (game->state = STATE_NEWGAME);
	case VICTMENU_QUITGAME:
	    if (display->confirm ("Quit game?"))
		return 0;
	    break;
	}
    } while (1);
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Construct Victory Screen UI.
 * @return The new AI.
 */
UI *new_VictoryUI (void)
{
    /* reserve memory for the UI */
    if (! (ui = new_UI ()))
	return NULL;

    /* initialise methods */
    ui->init = init;
    ui->operate = operate;

    /* return the UI */
    return ui;
}
