/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 07-May-2024.
 *
 * Item Stack Module.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "stack.h"
#include "fatal.h"
#include "utils.h"

/*----------------------------------------------------------------------
 * Public Method Function Defintions.
 */

/**
 * Destroy the stack.
 * @param stack The stack object.
 */
static void destroy (ItemStack *stack)
{
    if (stack->items)
	free (stack->items);
    free (stack);
}

/**
 * Write the stack to an open file.
 * @param stack  The stack object.
 * @param output The output file handle.
 * @return       1 on success, 0 on failure.
 */
static int write (ItemStack *stack, FILE *output)
{
    int c; /* counter */
    if (! writebyte (&stack->x, output))
	return 0;
    if (! writebyte (&stack->y, output))
	return 0;
    if (! writebyte (&stack->itemcount, output))
	return 0;
    for (c = 0; c < stack->itemcount; ++c)
	if (! writebyte (&stack->items[c], output))
	    return 0;
    return 1;
}

/**
 * Read the stack from an open file.
 * @param stack The stack object.
 * @param input The input file handle.
 * @return      1 on the success, 0 on failure.
 */
static int read (ItemStack *stack, FILE *input)
{
    int c; /* counter */

    /* clear out the item stack */
    if (stack->items) {
	free (stack->items);
	stack->items = NULL;
    }

    /* read the item stack */
    if (! readbyte (&stack->x, input))
	return 0;
    if (! readbyte (&stack->y, input))
	return 0;
    if (! readbyte (&stack->itemcount, input))
	return 0;
    if (! (stack->items = malloc (stack->itemcount)))
	fatal_error (FATAL_MEMORY);
    for (c = 0; c < stack->itemcount; ++c)
	if (! readbyte (&stack->items[c], input))
	    return 0;

    /* success! */
    return 1;
}

/**
 * Add an item onto the stack.
 * @param stack The stack object.
 */
static void drop (ItemStack *stack, int item)
{
    if (stack->itemcount)
	stack->items = realloc
	    (stack->items, (stack->itemcount + 1) * sizeof (int));
    else
	stack->items = malloc (sizeof (int));
    if (! stack->items)
	fatal_error (FATAL_MEMORY);
    stack->items[stack->itemcount++] = item;
}

/**
 * Remove an item from the stack.
 * @param stack The stack object.
 * @return The item removed from the stack, or 0 for none.
 */
static int take (ItemStack *stack)
{
    int item; /* item removed from the stack */
    if (! stack->itemcount)
	return 0;
    item = stack->items[--stack->itemcount];
    if (stack->itemcount)
	stack->items = realloc
	    (stack->items, stack->itemcount * sizeof (int));
    else {
	free (stack->items);
	stack->items = NULL;
    }
    return item;
}

/**
 * Return the item on top of the stack.
 * @param stack The stack object.
 * @return The item on the top of the stack, or 0 for none.
 */
static int top (ItemStack *stack)
{
    return stack->itemcount
	? stack->items[stack->itemcount - 1]
	: 0;
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Create a new stack of items.
 * @return The new stack.
 */
ItemStack *new_ItemStack (void)
{
    ItemStack *stack; /* the new stack */

    /* reserve memory for the stack. */
    if (! (stack = malloc (sizeof (ItemStack))))
	fatal_error (FATAL_MEMORY);

    /* initialise the methods */
    stack->destroy = destroy;
    stack->write = write;
    stack->read = read;
    stack->drop = drop;
    stack->take = take;
    stack->top = top;

    /* initialise the attributes */
    stack->itemcount = 0;
    stack->items = NULL;
    stack->x = 0;
    stack->y = 0;

    /* return the new stack */
    return stack;
}
