/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 16-Aug-2020.
 *
 * Display Module.
 */

/*----------------------------------------------------------------------
 * Headers.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>
#include <sys/timeb.h>

/* project-specific headers */
#include "cgalib.h"
#include "names.h"
#include "sccc.h"
#include "fatal.h"
#include "display.h"
#include "controls.h"
#include "config.h"
#include "game.h"
#include "unit.h"
#include "skill.h"
#include "item.h"
#include "cabinet.h"
#include "stack.h"
#include "scores.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/**
 * @enum MiniTileIDs
 * IDs for the menu scroll arrows.
 */
typedef enum {
    MENU_UP, /* up arrow */
    MENU_BLANK, /* no arrow */
    MENU_DOWN, /* down arrow */
    SKULL, /* skull icon */
    SKULL_MASK /* skull mask */
} MiniTileIDs;

/** @var display The display object */
static Display *display = NULL;

/** @var screen is the CGALIB screen data. */
static Screen *screen;

/** @var controls A pointer to the game control handler. */
static Controls *controls;

/** @var game A pointer to the game object. */
static Game *game;

/** @var titleimage is the title screen image bitmap */
static Bitmap *titleimage;

/** @var titletext is the title screen text bitmap */
static Bitmap *titletext;

/** @var background is the background bitmap */
static Bitmap *background;

/** @var tiles is the collection of tiles */
static Bitmap *tiles[56];

/** @var minitiles is a set of minitiles */
static Bitmap *minitiles[5];

/** @var font The normal intensity font */
static Font *font;

/** @var lowfont The low intensity font. */
static Font *lowfont;

/** @var highfont The highlight font. */
static Font *highfont;

/** @var fracfont Font for fractional numbers. */
static Font *fracfont;

/** @var gamesound 1 if there is game sound. */
static int gamesound;

/** @var tune The intro tune. */
static Tune *tune = NULL;

/** @var noises A selection of sound effect noises. */
static Effect *noises[7];

/** @enum TileID IDs of the various tiles. */
typedef enum {
    TILE_WALL,
    TILE_WALL_E,
    TILE_WALL_S,
    TILE_WALL_SE,
    TILE_WALL_W,
    TILE_WALL_WE,
    TILE_WALL_WS,
    TILE_WALL_WSE,
    TILE_WALL_N,
    TILE_WALL_NE,
    TILE_WALL_NS,
    TILE_WALL_NSE,
    TILE_WALL_NW,
    TILE_WALL_NWE,
    TILE_WALL_NWS,
    TILE_WALL_NWSE,
    TILE_FLOOR,
    TILE_CABINET,
    TILE_CHAIR,
    TILE_TABLE,
    TILE_DOOR_WE_OPEN,
    TILE_DOOR_NS_OPEN,
    TILE_DOOR_WE_CLOSED,
    TILE_DOOR_NS_CLOSED,
    TILE_UNIT,
    TILE_UNIT_MASK,
    TILE_UNIT_DEAD,
    TILE_UNIT_DEAD_MASK,
    TILE_ENEMY,
    TILE_ENEMY_MASK,
    TILE_ENEMY_DEAD,
    TILE_ENEMY_DEAD_MASK,
    TILE_MEDIKIT,
    TILE_MEDIKIT_MASK,
    TILE_BLADE,
    TILE_BLADE_MASK,
    TILE_PISTOL,
    TILE_PISTOL_MASK,
    TILE_RIFLE,
    TILE_RIFLE_MASK,
    TILE_HELMET,
    TILE_HELMET_MASK,
    TILE_ARMOUR,
    TILE_ARMOUR_MASK,
    TILE_CARD,
    TILE_CARD_MASK,
    TILE_CURSOR,
    TILE_CURSOR_MASK,
    TILE_HOSTAGE,
    TILE_HOSTAGE_MASK,
    TILE_FLASH,
    TILE_FLASH_MASK,
    TILE_HIT,
    TILE_HIT_MASK,
    TILE_PUNCH,
    TILE_PUNCH_MASK
} TileID;

/** @var itembitmapids Bitmap ID for each item. */
static int itembitmapids[ITEM_COUNT] = {
    0,
    TILE_RIFLE,
    TILE_PISTOL,
    TILE_BLADE,
    TILE_HELMET,
    TILE_ARMOUR,
    TILE_MEDIKIT,
    TILE_CARD
};

/** @var itemmaskids Bitmap ID for each item. */
static int itemmaskids[ITEM_COUNT] = {
    0,
    TILE_RIFLE_MASK,
    TILE_PISTOL_MASK,
    TILE_BLADE_MASK,
    TILE_HELMET_MASK,
    TILE_ARMOUR_MASK,
    TILE_MEDIKIT_MASK,
    TILE_CARD_MASK
};

/** @var unitbitmapids Bitmap IDs for each unit class. */
static int unitbitmapids[UNIT_CLASSES] = {
    0,
    TILE_UNIT, /* medic */
    TILE_UNIT, /* scout */
    TILE_UNIT, /* auxilliary */
    TILE_UNIT, /* soldier */
    TILE_ENEMY, /* enemy */
    TILE_HOSTAGE /* hostage */
};

/** @var unitmaskids Bitmap IDs for each unit class. */
static int unitmaskids[UNIT_CLASSES] = {
    0,
    TILE_UNIT_MASK, /* medic */
    TILE_UNIT_MASK, /* scout */
    TILE_UNIT_MASK, /* auxilliary */
    TILE_UNIT_MASK, /* soldier */
    TILE_ENEMY_MASK, /* enemy */
    TILE_HOSTAGE_MASK /* hostage */
};

/** @var unitbitmapids Bitmap IDs for each unit class. */
static int deadbitmapids[UNIT_CLASSES] = {
    0,
    TILE_UNIT_DEAD, /* medic */
    TILE_UNIT_DEAD, /* scout */
    TILE_UNIT_DEAD, /* auxilliary */
    TILE_UNIT_DEAD, /* soldier */
    TILE_ENEMY_DEAD, /* enemy */
    TILE_UNIT_DEAD /* hostage */
};

/** @var unitmaskids Bitmap IDs for each unit class. */
static int deadmaskids[UNIT_CLASSES] = {
    0,
    TILE_UNIT_DEAD_MASK, /* medic */
    TILE_UNIT_DEAD_MASK, /* scout */
    TILE_UNIT_DEAD_MASK, /* auxilliary */
    TILE_UNIT_DEAD_MASK, /* soldier */
    TILE_ENEMY_DEAD_MASK, /* enemy */
    TILE_UNIT_DEAD_MASK /* hostage */
};

/** @var mapbitmap A graphics representation of the map. */
static Bitmap *mapbitmap;

/** @var busybitmap The screen behind a busy message. */
static Bitmap *busybitmap;

/** @var behindinventory What was behind the inventory. */
static Bitmap *behindinventory;

/** @var levelnames Names of the difficulty levels. */
static char *levelnames[] = {
    "Easy",
    "Fair",
    "Hard"
};

/** @var delaytime end of a delay period */
static struct timeb delaytime;

/** @var tutorialtext Text of the tutorial messages. */
static char *tutorialtext[] = {
    "Use the 'Move' option to bring units in at the entrance.",
    "Move the cursor and use the 'Move' option to move the unit.",
    "Use the 'Move' option again to leave the area.",
    "Use the 'next Unit' option to bring another unit in."
};

/*----------------------------------------------------------------------
 * Level 1 Private Function Definitions.
 */

/**
 * Load the graphical assets.
 */
static void loadassets (void)
{
    /* local variables */
    FILE *input; /* the asset file handle */
    char header[8]; /* the asset file header to verify */
    Bitmap *logo; /* the Cyningstan logo */
    time_t start; /* time the Cyningstan logo was displayed */
    int t, /* tile count */
	c; /* sound effect count */
    Names *names; /* a pointer to the name generator */

    /* attempt to open the file and check the header */
    if (! (input = fopen ("SCCC.DAT", "rb")))
	fatal_error (FATAL_NODATA);
    if (! fread (header, 8, 1, input))
	fatal_error (FATAL_INVALIDDATA);
    if (strcmp (header, "SCM007D"))
	fatal_error (FATAL_INVALIDDATA);

    /* load and display the Cyningstan logo */
    start = time (NULL);
    if (! (logo = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    screen->put (screen, logo, 96, 92, CGALIB_SET);
    logo->destroy (logo);
    screen->update (screen);

    /* load the title screen components */
    if (! (titleimage = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (titletext = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);

    /* load the main font */
    if (! (font = read_Font (input, 2)))
	fatal_error (FATAL_INVALIDDATA);
    screen->font = font;

    /* create a low-intensity version of the font */
    if (! (lowfont = font->clone (font)))
	fatal_error (FATAL_MEMORY);
    lowfont->recolour (lowfont, 1, 0);

    /* create a highlighted version of the font */
    if (! (highfont = font->clone (font)))
	fatal_error (FATAL_MEMORY);
    highfont->recolour (highfont, 3, 2);

    /* load the fractions font */
    if (! (fracfont = read_Font (input, 2)))
	fatal_error (FATAL_INVALIDDATA);

    /* load all the tiles */
    for (t = 0; t < 56; ++t)
	if (! (tiles[t] = read_Bitmap (input)))
	    fatal_error (FATAL_INVALIDDATA);

    /* load the menu minitiles */
    for (t = 0; t < 5; ++t)
	if (! (minitiles[t] = read_Bitmap (input)))
	    fatal_error (FATAL_INVALIDDATA);

    /* load the background */
    if (! (background = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);

    /* non-display assets: the name generator */
    names = getnamegenerator ();
    if (! names->read (names, input))
	fatal_error (FATAL_INVALIDDATA);

    /* load the music */
    if (! (tune = new_Tune ()))
	fatal_error (FATAL_MEMORY);
    if (! tune->read (tune, input))
	fatal_error (FATAL_INVALIDDATA);

    /* load the sound effects */
    for (c = 0; c <= DISPLAY_COMMS; ++c) {	
	if (! (noises[c] = new_Effect ()))
	    fatal_error (FATAL_MEMORY);
	if (! (noises[c]->read (noises[c], input)))
	    fatal_error (FATAL_INVALIDDATA);
    }

    /* close the input file */
    fclose (input);

    /* clear the logo after at least three seconds */
    while (time (NULL) < start + 4);
    screen->box (screen, 96, 92, 128, 16, CGALIB_BOX_BLANK);
    screen->update (screen);
    while (time (NULL) < start + 5);
}

/**
 * Initialise the menu.
 * @return What was behind the menu.
 */
static Bitmap *initmenu (void)
{
    Bitmap *behind; /* what was behind the menu */
    if (! (behind = screen->get (screen, 248, 160, 64, 32)))
	fatal_error (FATAL_MEMORY);
    screen->transfer
	(screen, background,
	 248, 160,
	 56, 160,
	 8, 32,
	 CGALIB_SET);
    return behind;
}

/**
 * Display the menu.
 * @param count The number of options in the menu.
 * @param options The menu options.
 * @param option The selected option.
 * @param top The topmost option.
 */
static void displaymenu (int count, char **options, int option, int top)
{
    int c; /* row counter */
    char optstring[13]; /* formatted option string */

    /* display the visible menu options */
    for (c = 0; c < 4; ++c)
	if (top + c < count) {
	    sprintf (optstring, "%-12.12s", options[top + c]);
	    if (top + c == option)
		screen->font = highfont;
	    else
		screen->font = font;
	    screen->print (screen, 256, 160 + 8 * c, optstring);
	} else
	    screen->box
		(screen,
		 256, 160 + 8 * c,
		 48, 8,
		 CGALIB_BOX_BLANK);

    /* display any scroll arrows */
    screen->put
	(screen,
	 top == 0 ? minitiles[MENU_BLANK] : minitiles[MENU_UP],
	 304, 160, CGALIB_SET);
    screen->put
	(screen,
	 top >= count - 4 ? minitiles[MENU_BLANK] : minitiles[MENU_DOWN],
	 304, 184, CGALIB_SET);

    /* update the screen */
    screen->update (screen);
}

/**
 * Clean up after the menu.
 */
static void cleanupmenu (Bitmap *behind)
{
    screen->put
	(screen,
	 behind,
	 248, 160,
	 CGALIB_SET);
    behind->destroy (behind);
    screen->update (screen);
}

/**
 * Return the X and Y coordinates of an inventory item.
 * @param slot    The item slot, 0..9, 0..19, 0..21.
 * @param invtype The inventory type.
 * @param x       Pointer to store the X coordinate.
 * @param y       Pointer to store the Y coordinate.
 */
static void getinvposition (int slot, int invtype, int *x, int *y)
{
    if (slot == 0) {
	*x = 24;
	*y = 124;
    } else if (slot == 1) {
	*x = 8;
	*y = 132;
    } else if (slot == 2) {
	*x = 24;
	*y = 140;
    } else if (slot == 3) {
	*x = 40;
	*y = 132;
    } else if (slot < 10) {
	*x = 8 + 16 * ((slot - 4) % 3);
	*y = 160 + 16 * ((slot - 4) / 3);
    } else if (invtype == DISPLAY_INVENTORY_CABINET) {
	*x = 64 + 16 * ((slot - 10) % 3);
	*y = 128 + 16 * ((slot - 10) / 3);
    } else if (slot == 10) {
	*x = 80;
	*y = 124;
    } else if (slot == 11) {
	*x = 64;
	*y = 132;
    } else if (slot == 12) {
	*x = 80;
	*y = 140;
    } else if (slot == 13) {
	*x = 96;
	*y = 132;
    } else {
	*x = 64 + 16 * ((slot - 14) % 3);
	*y = 160 + 16 * ((slot - 14) / 3);
    }
}

/*----------------------------------------------------------------------
 * Public Methods.
 */

/**
 * Destroy the display when no longer needed.
 * @param display is the display to destroy.
 */
static void destroy ()
{
    int t, /* tile counter */
	c; /* general counter */
    Speaker *speaker; /* speaker object */

    /* clean up the display */
    if (display) {

	/* back to text mode */
	screen->destroy (screen);

	/* destroy any assets that were created */
	if (titleimage)
	    titleimage->destroy (titleimage);
	if (titletext)
	    titletext->destroy (titletext);
	if (font)
	    font->destroy (font);
	if (lowfont)
	    lowfont->destroy (lowfont);
	if (highfont)
	    highfont->destroy (highfont);
	if (fracfont)
	    fracfont->destroy (fracfont);
	for (t = 0; t < 54; ++t)
	    if (tiles[t])
		tiles[t]->destroy (tiles[t]);
	for (t = 0; t < 5; ++t)
	    if (minitiles[t])
		minitiles[t]->destroy (minitiles[t]);
	if (background)
	    background->destroy (background);
	if (mapbitmap)
	    mapbitmap->destroy (mapbitmap);
	if (busybitmap)
	    busybitmap->destroy (busybitmap);
	if (tune)
	    tune->destroy (tune);
	for (c = 0; c < 7; ++c)
	    if (noises[c])
		noises[c]->destroy (noises[c]);
	if ((speaker = get_Speaker ()))
	    speaker->destroy ();

	/* destroy the display */
    	free (display);
    }
}

/**
 * Give display routines access to the game data.
 * @param ingame The game being played.
 */
static void setgame (Game *ingame)
{
    game = ingame;
}

/**
 * Update the screen from the buffer.
 */
static void update (void)
{
    screen->update (screen);
}

/**
 * Display a message in the prompt box.
 * @param format The format of the output.
 * @param ...    The arguments to print with the format.
 */
static void message (char *format, ...)
{
    va_list args; /* argument list */
    char msgstr[61]; /* the output string */

    /* build the output string */
    va_start (args, format);
    vsprintf (msgstr, format, args);
    va_end (args);

    /* scroll the window and print the string */
    screen->transfer
	(screen,
	 screen->buffer,
	 64, 160,
	 64, 168,
	 240, 24,
	 CGALIB_SET);
    screen->ink = 3;
    screen->box (screen, 64, 184, 240, 8, CGALIB_BOX_BLANK);
    screen->font = font;
    screen->print (screen, 64, 184, msgstr);
}

/**
 * Display or clear a "busy" message.
 * @param message The message to display; blank to clear.
 */
static void busy (char *message)
{
    /* restore the previous background */
    if (busybitmap) {
	screen->put
	    (screen, busybitmap,
	     320 - busybitmap->width, 192,
	     CGALIB_SET);
	busybitmap->destroy (busybitmap);
	busybitmap = NULL;
    }

    /* print a message */
    if (*message) {
	busybitmap = screen->get
	    (screen,
	     320 - 4 * (strlen (message)), 192,
	     4 * strlen (message), 8);
	screen->font = highfont;
	screen->print
	    (screen,
	     320 - 4 * (strlen (message)), 192,
	     message);
	screen->font = font;
    }

    /* update the display */
    display->update ();
}

/**
 * Display a brief error message.
 * @param message The message to display.
 */
static void error (char *format, ...)
{
    va_list args; /* argument list */
    char msgstr[61]; /* the output string */

    /* build the output string */
    va_start (args, format);
    vsprintf (msgstr, format, args);
    va_end (args);

    /* display the message */
    display->busy (msgstr);
    controls->release (0);
    controls->wait (1000);
    display->busy ("");
}

/**
 * Display a menu and get an option from it.
 * @param count   The number of options in the menu.
 * @param options The option names in an array.
 * @param initial The initial option selected.
 * @return        The number of the option selected.
 */
static int menu (int count, char **options, int initial)
{
    int top, /* index of top row option */
	option; /* option currently selected */
    Bitmap *behind; /* storage of what's behind the menu */

    /* initialise the menu */
    behind = initmenu ();
    top = initial > 3 ? initial - 3 : 0;
    option = initial;
    displaymenu (count, options, option, top);

    /* allow option selection till fire pressed/released */
    controls->poll ();
    while (controls->fire) {

	/* up key pressed */
	if (controls->up && option > 0) {
	    --option;
	    if (option < top)
		top = option;
	    displaymenu (count, options, option, top);
	    do
		controls->poll ();
	    while (controls->up);
	}

	/* down key pressed */
	else if (controls->down && option < count - 1) {
	    ++option;
	    if (option > top + 3)
		top = option - 3;
	    displaymenu (count, options, option, top);
	    do
		controls->poll ();
	    while (controls->down);
	}

	/* check the keyboard again */
	controls->poll ();
    }

    /* clean up and return */
    screen->put
	(screen,
	 behind,
	 248, 160,
	 CGALIB_SET);
    behind->destroy (behind);
    update ();
    do
	controls->poll ();
    while (controls->fire);
    return option;
}

/**
 * Display a simple yes/no confirmation menu.
 * @param text The message to display (12 chars max).
 * @return     1 if yes, 0 if no.
 */
static int confirm (char *text)
{
    int option; /* option currently selected */
    Bitmap *behind; /* storage of what's behind the menu */

    /* initialise the menu */
    if (! (behind = screen->get (screen, 248, 160, 64, 32)))
	fatal_error (FATAL_MEMORY);
    screen->transfer
	(screen, background,
	 248, 160,
	 56, 160,
	 8, 32,
	 CGALIB_SET);
    screen->transfer
	(screen, background,
	 256, 168,
	 256, 16,
	 48, 8,
	 CGALIB_SET);
    option = 0;
    screen->font = font;
    screen->box (screen, 256, 160, 48, 8, CGALIB_BOX_BLANK);
    screen->box (screen, 256, 176, 48, 16, CGALIB_BOX_BLANK);
    screen->print (screen, 256, 160, text);
    screen->print (screen, 256, 184, "Yes         ");
    screen->font = highfont;
    screen->print (screen, 256, 176, "No          ");
    screen->update (screen);

    /* allow option selection till fire pressed/released */
    controls->poll ();
    while (! (controls->fire ||
	      toupper (controls->ascii) == 'N' ||
	      toupper (controls->ascii) == 'Y')) {

	/* up key pressed */
	if (controls->up && option == 1) {
	    --option;
	    screen->font = highfont;
	    screen->print (screen, 256, 176, "No          ");
	    screen->font = font;
	    screen->print (screen, 256, 184, "Yes         ");
	    screen->update (screen);
	    controls->release (0);
	}

	/* down key pressed */
	else if (controls->down && option == 0) {
	    ++option;
	    screen->font = highfont;
	    screen->print (screen, 256, 184, "Yes         ");
	    screen->font = font;
	    screen->print (screen, 256, 176, "No          ");
	    screen->update (screen);
	    controls->release (0);
	}

	/* check the keyboard again */
	controls->poll ();
    }

    /* check for keyboard shortcut */
    if (toupper (controls->ascii) == 'N')
	option = 0;
    else if (toupper (controls->ascii) == 'Y')
	option = 1;

    /* clean up and return */
    cleanupmenu (behind);
    update ();
    controls->release (0);
    return option;
}

/**
 * Start a delay timer.
 */
static void startdelay (int msecs)
{
    struct timeb now; /* time now */
    ftime (&now);
    delaytime = now;
    delaytime.millitm += msecs;
    while (delaytime.millitm > 1000) {
	++delaytime.time;
	delaytime.millitm -= 1000;
    }
}

/**
 * Wait until a delay timer has finished.
 */
static void enddelay (void)
{
    struct timeb now; /* time now */
    do
	ftime (&now);
    while (now.time < delaytime.time ||
	     (now.time == delaytime.time &&
	      now.millitm <= delaytime.millitm));
}

/**
 * Show the title screen.
 */
static void showtitlescreen (void)
{
    screen->put (screen, titleimage, 88, 20, CGALIB_SET);
    screen->put (screen, titletext, 40, 148, CGALIB_SET);
    screen->update (screen);
}

/**
 * Await the fire key at the title screen.
 */
static void titlekey (void)
{
    int key; /* value of key pressed */

    /* show the"Press FIRE" message */
    screen->font = font;
    screen->ink = 3;
    screen->print (screen, 128, 184, "   Press FIRE   ");
    screen->update (screen);

    /* wait for a key until FIRE is pressed */
    do {
	if (gamesound) {
	    tune->play (tune, controls->getkeyhandler ());
	    while (! controls->getkeyhandler ()->anykey ());
	} else
	    while (! controls->getkeyhandler ()->anykey ());
	key = controls->getkeyhandler ()->scancode ();
    } while (key != KEY_CTRL && key != KEY_ENTER && key != KEY_SPACE);

    /* erase the "Press FIRE" message */
    screen->box (screen, 128, 184, 64, 8, CGALIB_BOX_BLANK);
    screen->update (screen);
    do
	controls->poll ();
    while (controls->fire);
}

/**
 * Make a sound.
 * @param id The ID of the sound.
 */
static void playsound (int id)
{
    if (gamesound)
	noises[id]->play (noises[id]);
}

/**
 * Show the standard screen background.
 */
static void showbackground (void)
{
    screen->put (screen, background, 0, 0, CGALIB_SET);
}

/**
 * Show a unit in the unit window.
 * @param unit The unit to show.
 */
static void showunit (Unit *unit)
{
    char numtext[5]; /* numeric text */
    int c, /* general counter */
	x, /* x coordinate of inventory item */
	y; /* y coordinate of inventory item */

    /* initialise */
    screen->font = font;

    /* unit image, class and name */
    screen->put
	(screen,
	 tiles[unitbitmapids[unit->class]],
	 8, 8,
	 CGALIB_SET);
    screen->box (screen, 24, 12, 32, 8, CGALIB_BOX_BLANK);
    screen->print (screen, 24, 12, getclassname (unit->class));
    screen->box (screen, 8, 28, 48, 8, CGALIB_BOX_BLANK);
    screen->print (screen, 8, 28, unit->name);

    /* basic stats */
    sprintf (numtext, "%02d", unit->strength);
    screen->print
	(screen,
	 12, 56,
	 unit->class == UNIT_ENEMY ? "--" : numtext);
    sprintf (numtext, "%02d", unit->agility);
    screen->print
	(screen,
	 28, 56,
	 unit->class == UNIT_ENEMY ? "--" : numtext);
    sprintf (numtext, "%02d", unit->endurance);
    screen->print
	(screen,
	 44, 56,
	 unit->class == UNIT_ENEMY ? "--" : numtext);

    /* action points and health */
    sprintf (numtext, "%02d", unit->action);
    screen->print
	(screen,
	 24, 68,
	 unit->class == UNIT_ENEMY ? "--" : numtext);
    sprintf (numtext, "%02d", unit->health);
    screen->print
	(screen,
	 48, 68,
	 unit->class == UNIT_ENEMY ? "--" : numtext);
    screen->font = fracfont;
    sprintf (numtext, "%02d", unit->agility + 14);
    screen->print
	(screen,
	 24, 76,
	 unit->class == UNIT_ENEMY ? "::" : numtext);
    sprintf (numtext, "%02d", unit->strength + unit->endurance + 7);
    screen->print
	(screen,
	 48, 76,
	 unit->class == UNIT_ENEMY ? "::" : numtext);

    /* skills */
    screen->box (screen, 8, 88, 48, 32, CGALIB_BOX_BLANK);
    screen->font = font;
    if (unit->class == UNIT_ENEMY) {
	screen->ink = 1;
	screen->box (screen, 8, 88, 48, 32, CGALIB_BOX_DITHERED);
	screen->ink = 3;
    } else
	for (c = 0; c < 4; ++c)
	    if (unit->skills[c]) {
		screen->print
		    (screen, 8, 88 + 8 * c,
		     unit->skills[c]->name (unit->skills[c]));
		sprintf (numtext, "%1d", unit->skills[c]->level);
		screen->print (screen, 52, 88 + 8 * c, numtext);
	    }

    /* inventory */
    for (c = 0; c < 10 - 6 * (unit->class == UNIT_ENEMY); ++c) {
	getinvposition (c, DISPLAY_INVENTORY_SINGLE, &x, &y);
	if (unit->inventory[c])
	    screen->put
		(screen,
		 tiles[itembitmapids[unit->inventory[c]]],
		 x, y,
		 CGALIB_SET);
	else
	    screen->box (screen, x, y, 16, 16, CGALIB_BOX_BLANK);
    }
    if (unit->class == UNIT_ENEMY) {
	screen->ink = 1;
	screen->box (screen, 8, 160, 48, 32, CGALIB_BOX_DITHERED);
	screen->ink = 3;
    }
}

/**
 * Show a unit's action points.
 * @param unit The unit to show.
 */
static void showunitaction (Unit *unit)
{
    char numtext[3]; /* text representation of numeric text */
    sprintf (numtext, "%02d", unit->action);
    screen->ink = 3;
    screen->font = font;
    screen->print (screen, 24, 68, numtext);
}

/**
 * Show a unit's health points.
 * @param unit The unit to show.
 */
static void showunithealth (Unit *unit)
{
    char numtext[3]; /* text representation of numeric text */
    sprintf (numtext, "%02d", unit->health);
    screen->ink = 3;
    screen->font = font;
    screen->print (screen, 48, 68, numtext);
}

/**
 * Show a title above the text window.
 * @param text The title text.
 */
static void showtitle (char *title)
{
    char buffer[25]; /* buffer for the formatted text */
    sprintf (buffer, "%-24.24s", title);
    screen->font = font;
    screen->print (screen, 216, 8, buffer);
}

/**
 * Typeset a piece of text and display it.
 * @param text The text to typeset.
 */
static void typeset (char *text)
{
    int i, /* index counter */
	gap, /* last gap location */
	start, /* start of sentence */
	y; /* y location of text */
    char line[25]; /* one line of text */
    Bitmap *bitmap; /* the page of typeset text */

    /* initialise */
    if (! (bitmap = new_Bitmap (96, 128)))
	fatal_error (FATAL_MEMORY);
    bitmap->box (bitmap, 0, 0, 96, 128, CGALIB_BOX_BLANK);
    bitmap->ink = 3;
    bitmap->font = font;

    /* format the text and print onto the bitmap */
    for (y = start = gap = i = 0; i < strlen (text); ++i) {

	/* register where we last found a gap */
	if (text[i] == ' ')
	    gap = i;

	/* perform a word wrap if the line is too long. */
	if (i - start >= 24) {
	    if (y < 128) {
		strcpy (line, "                        ");
		strncpy (line, &text[start], gap - start);
		bitmap->print (bitmap, 0, y, line);
		y += 8;
	    }
	    start = gap + 1;
	}

    }

    /* typeset the last line */
    if (y < 128) {
	strcpy (line, "                        ");
	strncpy (line, &text[start], i - start);
	bitmap->print (bitmap, 0, y, line);
    }

    /* display the typeset text */
    screen->put (screen, bitmap, 216, 24, CGALIB_SET);
    bitmap->destroy (bitmap);
}

/**
 * Prepare a scrolling mission map.
 * @param game The game object.
 */
static void preparemap (Game *game)
{
    int x, /* x coordinate counter */
	y, /* y coordinate counter */
	c, /* general counter */
	loc; /* location of a player unit */

    /* destroy an existing map bitmap and create the new one */
    if (mapbitmap)
	mapbitmap->destroy (mapbitmap);
    if (! (mapbitmap = new_Bitmap
	   (16 * game->map->width, 16 * game->map->height)))
	fatal_error (FATAL_MEMORY);

    /* generate mask for what squares are seen by any PC */
    game->visibility = 0;
    for (c = 0; c < game->unitcount; ++c)
	if (game->units[c] &&
	    game->units[c]->health &&
	    ! game->units[c]->unconscious &&
	    (game->units[c]->x ||
	     game->units[c]->y))
	{
	    loc = game->units[c]->x
		+ game->map->width
		* game->units[c]->y;
	    game->visibility |= game->sightlines[loc];
	}

    /* update each individual map square */
    for (x = 0; x < game->map->width; ++x)
	for (y = 0; y < game->map->height; ++y)
	    display->updatemapsquare (game, x, y);
}

/**
 * Update a single map square.
 * @param game The game object.
 * @param x       X coordinate.
 * @param y       Y coordinate.
 */
static void updatemapsquare (Game *game, int x, int y)
{
    int blockvalue, /* original block value */
	tile, /* unit tile */
	mask, /* unit mask */
	loc, /* location for unit/item stack */
	item, /* item at the top of a stack */
	los, /* unit's line of sight mask */
	visible; /* 1 if a unit is visible, 0 otherwise */
    Unit *unit; /* unit at this square */
    ItemStack *stack; /* stack at this location */

    /* see what's there */
    blockvalue = game->map->blocks[x + game->map->width * y];
    switch (blockvalue & 0x70) {
    case LEVELMAP_VOID:
	mapbitmap->box
	    (mapbitmap,
	     16 * x, 16 * y,
	     16, 16,
	     CGALIB_BOX_BLANK);
	break;
    case LEVELMAP_OPEN:
	mapbitmap->put
	    (mapbitmap,
	     tiles[TILE_FLOOR],
	     16 * x, 16 * y,
	     CGALIB_SET);
	break;
    case LEVELMAP_WALL:
	mapbitmap->put
	    (mapbitmap,
	     tiles[TILE_WALL + (blockvalue & 0xf)],
	     16 * x, 16 * y,
	     CGALIB_SET);
	break;
    case LEVELMAP_DOOR:
	mapbitmap->put
	    (mapbitmap,
	     (blockvalue & 0x80)
	     ? tiles[TILE_DOOR_WE_OPEN
		     + ((blockvalue & 0xf) / 5 - 1)]
	     : tiles[TILE_DOOR_WE_CLOSED
		       + ((blockvalue & 0xf) / 5 - 1)],
	     16 * x, 16 * y,
	     CGALIB_SET);
	break;
    case MISSION_FURNITURE:
	mapbitmap->put
	    (mapbitmap,
	     tiles[TILE_CABINET + (blockvalue & 0xf)],
	     16 * x, 16 * y,
	     CGALIB_SET);
	break;
    }

    /* check for a stack present */
    loc = x + game->map->width * y;
    if (game->stackcache[loc] & 0x80) {
	stack = game->stacks[game->stackcache[loc] & 0x7f];
	item = stack->top (stack);
	mapbitmap->put
	    (mapbitmap,
	     tiles[itemmaskids[item]],
	     16 * stack->x, 16 * stack->y,
	     CGALIB_AND);
	mapbitmap->put
	    (mapbitmap,
	     tiles[itembitmapids[item]],
	     16 * stack->x, 16 * stack->y,
	     CGALIB_OR);
    }

    /* check for a unit present */
    if (game->unitcache[loc] & MISSION_CACHE_UNIT)
	unit = game->units[game->unitcache[loc] & 0xf];
    else if (game->unitcache[loc] & MISSION_CACHE_ENEMY) {
	unit = game->enemies[game->unitcache[loc] & 0xf];
	los = 1 << unit->id;
    } else if (game->unitcache[loc] & MISSION_CACHE_HOSTAGE) {
	unit = game->hostages[game->unitcache[loc] & 0xf];
	los = 1 << (game->enemycount + unit->id);
    } else
	return;

    /* determine if the unit is visible */
    if (game->state == STATE_MAP)
	visible
	    = (unit->class < UNIT_ENEMY ||
	       (unit->class == UNIT_HOSTAGE && ! unit->hostage) ||
	       (game->visibility & los));
    else if (game->state == STATE_COMPUTER)
	visible = (unit->class < UNIT_ENEMY ||
		   (unit->class == UNIT_HOSTAGE && ! unit->hostage) ||
		   (game->sightlines[loc] & game->conscious));
    else
	visible = 0;

    /* show the unit */
    if (visible) {
	if (unit->health == 0 || unit->unconscious) {
	    tile = deadbitmapids[unit->class];
	    mask = deadmaskids[unit->class];
	} else {
	    tile = unitbitmapids[unit->class];
	    mask = unitmaskids[unit->class];
	}
	mapbitmap->put
	    (mapbitmap,
	     tiles[mask],
	     16 * unit->x, 16 * unit->y,
	     CGALIB_AND);
	mapbitmap->put
	    (mapbitmap,
	     tiles[tile],
	     16 * unit->x, 16 * unit->y,
	     CGALIB_OR);
	if (unit->health == 0) {
	    mapbitmap->put
		(mapbitmap,
		 minitiles[SKULL_MASK],
		 4 + 16 * unit->x, 16 * unit->y,
		 CGALIB_AND);
	    mapbitmap->put
		(mapbitmap,
		 minitiles[SKULL],
		 4 + 16 * unit->x, 16 * unit->y,
		 CGALIB_OR);
	}
    }
}

/**
 * Show the map.
 * @param x X coordinate of top left corner.
 * @param y Y coordinate of top left corner.
 */
static void showmap (int x, int y)
{
    screen->transfer
	(screen, mapbitmap,
	 64, 8,
	 16 * x, 16 * y,
	 144, 144,
	 CGALIB_SET);
}

/**
 * Show the map cursor.
 * @param cursorx X coordinate of the map cursor.
 * @param cursory Y coordinate of the map cursor.
 * @param viewx   X coordinate of the map view.
 * @param viewy   Y coordinate of the map view.
 */
static void showmapcursor
(int cursorx, int cursory, int viewx, int viewy)
{
    /* do nothing if the cursor is outside the window */
    if (cursorx < viewx ||
	cursorx > viewx + 8 ||
	cursory < viewy ||
	cursory > viewy + 8)
	return;

    /* show the map cursor */
    screen->put
	(screen,
	 tiles[TILE_CURSOR_MASK],
	 64 + 16 * (cursorx - viewx), 8 + 16 * (cursory - viewy),
	 CGALIB_AND);
    screen->put
	(screen,
	 tiles[TILE_CURSOR],
	 64 + 16 * (cursorx - viewx), 8 + 16 * (cursory - viewy),
	 CGALIB_OR);
}

/**
 * Hide the map cursor.
 * @param cursorx X coordinate of the map cursor.
 * @param cursory Y coordinate of the map cursor.
 * @param viewx   X coordinate of the map view.
 * @param viewy   Y coordinate of the map view.
 */
static void hidemapcursor
(int cursorx, int cursory, int viewx, int viewy)
{
    /* do nothing if the cursor is outside the window */
    if (cursorx < viewx ||
	cursorx > viewx + 8 ||
	cursory < viewy ||
	cursory > viewy + 8)
	return;

    /* show the map cursor */
    screen->transfer
	(screen, mapbitmap,
	 64 + 16 * (cursorx - viewx), 8 + 16 * (cursory - viewy),
	 16 * cursorx, 16 * cursory,
	 16, 16,
	 CGALIB_SET);
}

/**
 * Show a battle animation.
 * @param attacker The attacking unit.
 * @param defender The defender unit.
 * @param game  The game object.
 * @param viewx    View X coordinate.
 * @param viewy    View Y coordinate.
 */
static void battleanimation
(Unit *attacker,
 Unit *defender,
 Game *game,
 int viewx,
 int viewy)
{
    int atttile, /* sprite to superimpose on attacker */
	attmask, /* mask to superimpose on attacker */
	deftile, /* sprite to superimpose on defender */
	defmask, /* mask to superimpose on defender */
	noiseid, /* id of the noise to play */
	loc, /* location of the attacker */
	visible; /* 1 if the attacker is visible to the player */

    /* what to show at the attacker location */
    if (attacker->combattype == SKILL_LIGHT ||
	attacker->combattype == SKILL_HEAVY)
    {
	atttile = TILE_FLASH;
	attmask = TILE_FLASH_MASK;
	noiseid = DISPLAY_PEWPEW;
    } else if (attacker->combattype == SKILL_BLADE) {
	atttile = TILE_BLADE;
	attmask = TILE_BLADE_MASK;
	noiseid = DISPLAY_STAB;
    } else {
	atttile = TILE_PUNCH;
	attmask = TILE_PUNCH_MASK;
	noiseid = DISPLAY_HIT;
    }

    /* what to show at the defender location */
    if (attacker->feedbackid == UNIT_FEEDBACK_HIT) {
	deftile = TILE_HIT;
	defmask = TILE_HIT_MASK;
    } else
	deftile = defmask = 0;

    /* should the attack sprite be visible? */
    loc = attacker->x + game->map->width * attacker->y;
    if (game->state == STATE_MAP)
	visible = 1;
    else if (game->state == STATE_COMPUTER)
	visible = (attacker->class < UNIT_ENEMY ||
		   (attacker->class == UNIT_HOSTAGE &&
		    ! attacker->hostage) ||
		   (game->sightlines[loc] & game->conscious));
    else
	visible = 0;

    /* show the attack sprite */
    if (attacker->x >= viewx && attacker->x < viewx + 9 &&
	attacker->y >= viewy && attacker->y < viewy + 9 &&
	visible)
    {
	screen->put
	    (screen,
	     tiles[attmask],
	     64 + 16 * (attacker->x - viewx),
	     8 + 16 * (attacker->y - viewy),
	     CGALIB_AND);
	screen->put
	    (screen,
	     tiles[atttile],
	     64 + 16 * (attacker->x - viewx),
	     8 + 16 * (attacker->y - viewy),
	     CGALIB_OR);
    }

    /* show the defence sprite */
    if (deftile &&
	defender->x >= viewx && defender->x < viewx + 9 &&
	defender->y >= viewy && defender->y < viewy + 9) {
	screen->put
	    (screen,
	     tiles[defmask],
	     64 + 16 * (defender->x - viewx),
	     8 + 16 * (defender->y - viewy),
	     CGALIB_AND);
	screen->put
	    (screen,
	     tiles[deftile],
	     64 + 16 * (defender->x - viewx),
	     8 + 16 * (defender->y - viewy),
	     CGALIB_OR);
    }
    display->update ();

    /* pause, then restore the screen */
    display->startdelay (250);
    display->playsound (noiseid);
    display->enddelay ();
    display->showmapsquare (attacker->x, attacker->y, viewx, viewy);
    display->showmapsquare (defender->x, defender->y, viewx, viewy);
    display->update ();
}

/**
 * Show a list of bodies in a pile and allow selection.
 * @param unit The unit at the top of the pile.
 * @return     The unit selected.
 */
static Unit *selectbody (Unit *unit)
{
    Bitmap *behind; /* what was behind the body list */
    Unit *bodies[8]; /* list of bodies in the pile */
    int bodycount, /* counter */
	c, /* general counter */
	cursor; /* cursor bar */
    char textline[13]; /* formatted unit name */

    /* check the stack of bodies */
    bodycount = 0;
    bodies[bodycount++] = unit;
    while (unit->prone && bodycount < 8) {
	if (unit->prone & MISSION_CACHE_UNIT)
	    unit = game->units[unit->prone & 0xf];
	else if (unit->prone & MISSION_CACHE_ENEMY)
	    unit = game->enemies[unit->prone & 0xf];
	else if (unit->prone & MISSION_CACHE_HOSTAGE)
	    unit = game->hostages[unit->prone & 0xf];
	else
	    break; /* shouldn't happen */
	bodies[bodycount++] = unit;
    }

    /* display the list */
    if (! (behind = screen->get (screen, 64, 120, 56, 80)))
	fatal_error (FATAL_MEMORY);
    screen->transfer
	(screen, background, 64, 120, 8, 120, 56, 80, CGALIB_SET);
    screen->box (screen, 64, 124, 48, 36, CGALIB_BOX_BLANK);
    screen->transfer
	(screen, background, 64, 124, 8, 120, 48, 4, CGALIB_SET);
    screen->ink = 3;
    screen->font = font;
    for (c = 0; c < bodycount; ++c)
	screen->print (screen, 64, 128 + 8 * c, bodies[c]->name);

    /* allow selection of the unit */
    cursor = 0;
    do {

	/* highlight the unit and wait for input */
	sprintf (textline, "%-12.12s", bodies[cursor]->name);
	screen->font = highfont;
	screen->print
	    (screen,
	     64, 128 + 8 * cursor,
	     textline);
	screen->update (screen);
	controls->release (0);
	controls->wait (0);
	controls->poll ();

	/* move cursor up */
	if (controls->up && cursor > 0) {
	    screen->font = font;
	    screen->print
		(screen,
		 64, 128 + 8 * cursor,
		 textline);
	    --cursor;
	}

	/* move cursor down */
	else if (controls->down && cursor < bodycount - 1) {
	    screen->font = font;
	    screen->print
		(screen,
		 64, 128 + 8 * cursor,
		 textline);
	    ++cursor;
	}
	
    } while (! controls->fire);

    /* restore the background */
    screen->put (screen, behind, 64, 120, CGALIB_SET);
    screen->update (screen);
    behind->destroy (behind);

    /* wait for control release, and return */
    controls->release (0);
    return bodies[cursor];
}

/**
 * Show a second inventory.
 * @param invtype The inventory type (unit or cabinet).
 * @param unit    The unit, for unit inventories.
 * @param cabinet The cabinet, for cabinet inventories.
 */
static void showinventory (int invtype, Unit *unit, Cabinet *cabinet)
{
    int c, /* item counter */
	x, /* screen X coordinate */
	y; /* screen Y coordinate */

    /* unnecessary for a single inventory */
    if (invtype == DISPLAY_INVENTORY_SINGLE)
	return;

    /* store what was behind the inventory */
    if (! behindinventory &&
	! (behindinventory = screen->get
	   (screen, 64, 120, 56, 80)))
	fatal_error (FATAL_MEMORY);

    /* display the background */
    screen->transfer
	(screen, background, 64, 120, 8, 120, 56, 80, CGALIB_SET);
    if (invtype == DISPLAY_INVENTORY_CABINET) {
	screen->box (screen, 64, 124, 48, 36, CGALIB_BOX_BLANK);
	screen->transfer
	    (screen, background, 64, 124, 8, 120, 48, 4, CGALIB_SET);
    }

    /* display a unit inventory */
    if (invtype == DISPLAY_INVENTORY_DUAL) {
	for (c = 0; c < 10; ++c) {
	    getinvposition (10 + c, invtype, &x, &y);
	    if (unit->inventory[c])
		screen->put
		    (screen,
		     tiles[itembitmapids[unit->inventory[c]]],
		     x, y,
		     CGALIB_SET);
	    else
		screen->box (screen, x, y, 16, 16, CGALIB_BOX_BLANK);
	}
    }

    /* display a cabinet inventory */
    else if (invtype == DISPLAY_INVENTORY_CABINET) {
	for (c = 0; c < 12; ++c) {
	    getinvposition (10 + c, invtype, &x, &y);
	    if (cabinet->items[c])
		screen->put
		    (screen,
		     tiles[itembitmapids[cabinet->items[c]]],
		     x, y,
		     CGALIB_SET);
	    else
		screen->box (screen, x, y, 16, 16, CGALIB_BOX_BLANK);
	}
    }
}

/**
 * Hide the second inventory.
 */
static void hideinventory (void)
{
    if (! behindinventory)
	return;
    screen->put (screen, behindinventory, 64, 120, CGALIB_SET);
    behindinventory->destroy (behindinventory);
    behindinventory = 0;
}

/**
 * Show the inventory cursor.
 * @param cursor  The cursor.
 * @param invtype The inventory type.
 */
static void showinvcursor (int cursor, int invtype)
{
    int x, /* X coordinate of cursor */
	y; /* Y coordinate of cursor */
    getinvposition (cursor, invtype, &x, &y);
    screen->put (screen, tiles[TILE_CURSOR_MASK], x, y, CGALIB_AND);
    screen->put (screen, tiles[TILE_CURSOR], x, y, CGALIB_OR);
}

/**
 * Hide the inventory cursor.
 * @param cursor  The cursor position.
 * @param invtype The inventory type.
 * @param item    The item to show.
 */
static void hideinvcursor (int cursor, int invtype, int item)
{
    int x, /* X coordinate of cursor */
	y; /* Y coordinate of cursor */
    getinvposition (cursor, invtype, &x, &y);
    if (item)
	screen->put
	    (screen,
	     tiles[itembitmapids[item]],
	     x, y,
	     CGALIB_SET);
    else
	screen->box (screen, x, y, 16, 16, CGALIB_BOX_BLANK);
}

/**
 * Show computer turn progress.
 * @param percent Percentage points.
 */
static void showcomputerprogress (int percent)
{
    char numstr[5]; /* number as string */
    screen->ink = 3;
    screen->font = lowfont;
    screen->print (screen, 248, 80, "Progress");
    screen->font = font;
    sprintf (numstr, "%3d%%", percent);
    screen->print (screen, 256, 88, numstr);
}

/**
 * Show the scores for the current mission in the map window.
 * @param game    The game object.
 * @param scores  The array of scores.
 * @param victory 1 if the game was won, 0 if lost.
 */
static void showmissionscores
(Game *game,
 int *scores,
 int *performance,
 int victory)
{
    int c; /* general counter */
    char numstr[19]; /* number as string */

    /* mission scores template */
    screen->ink = 3;
    screen->box (screen, 64, 8, 144, 144, CGALIB_BOX_BLANK);
    screen->font = font;
    screen->print
	(screen, 120, 16,
	 victory ? "VICTORY!" : " DEFEAT ");
    screen->font = lowfont;
    screen->print (screen, 108, 40, "Health:");
    screen->print (screen, 112, 56, "Kills:");
    screen->print (screen, 100, 72, "Hostages:");
    screen->print (screen, 112, 88, "Cards:");
    screen->print (screen, 112, 104, "Speed:");
    screen->print (screen, 104, 128, "Overall:");

    /* print the scores template */
    for (c = 0; c < 5; ++c) {
	if (victory && game->getmultiplier (game, c)) {
	    sprintf (numstr, "%03d%%", scores[c]);
	    screen->font = font;
	} else if (victory) {
	    strcpy (numstr, "N/A");
	    screen->font = lowfont;
	} else {
	    sprintf
		(numstr,
		 "%d/%d",
		 performance[c],
		 performance[5 + c]);
	    screen->font = lowfont;
	}
	screen->print (screen, 140, 40 + 16 * c, numstr);
    }
    sprintf (numstr, "%03d%%", scores[SCORE_TOTAL]);
    screen->font = font;
    screen->print (screen, 140, 128, numstr);
}

/**
 * Show the high score table.
 * @param scores The score table object.
 * @param game   The current game.
 */
static void showscoretable (Scores *scores, Game *game)
{
    char numstr[5]; /* number as string */
    int l, /* difficulty level counter */
	t, /* mission type counter */
	avg[3]; /* average of all missions */

    /* display score table template */
    screen->ink = 3;
    screen->box (screen, 64, 8, 144, 144, CGALIB_BOX_BLANK);
    screen->font = lowfont;
    screen->print (screen, 72, 16, "SCORES");
    screen->print (screen, 136, 16, "Easy");
    screen->print (screen, 160, 16, "Fair");
    screen->print (screen, 184, 16, "Hard");
    for (t = 0; t < 6; ++t)
	screen->print (screen, 72, 32 + 16 * t, getmissiontype (t + 1));
    screen->print (screen, 72, 136, "Average");

    /* output scores */
    avg[0] = avg[1] = avg[2] = 0;
    for (l = 0; l < 3; ++l)
	for (t = 0; t < 6; ++t) {
	    if (game->level == l &&
		game->type == t + 1 &&
		game->highest)
		screen->font = highfont;
	    else
		screen->font = font;
	    sprintf (numstr, "%03d%%", scores->scores[l][t]);
	    screen->print
		(screen,
		 136 + 24 * l, 32 + 16 * t,
		 numstr);
	    avg[l] += scores->scores[l][t];
	}

    /* output score totals */
    avg[0] /= 6;
    avg[1] /= 6;
    avg[2] /= 6;
    screen->font = font;
    for (l = 0; l < 3; ++l) {
	sprintf (numstr, "%03d%%", avg[l]);
	screen->print (screen, 136 + 24 * l, 136, numstr);
    }
}

/**
 * Show a line of the configuration.
 * @param config    The configuration object.
 * @param line      The line of the configuration.
 * @param highlight 1 if highlighted, 0 if not.
 */
static void showconfigline (Config *config, int line, int highlight)
{
    char str[14]; /* generic formatted string */
    int unitid; /* unit ID */

    /* set colour and font for label */
    screen->ink = 3;
    screen->font = lowfont;

    /* difficulty level */
    if (line == 0) {
	screen->print (screen, 108, 32, "Level:");
	screen->font = highlight ? highfont : font;
	sprintf (str, "%-13.13s", levelnames[config->level]);
	screen->print (screen, 136, 32, str);
    }

    /* mission type */
    else if (line == 1) {
	screen->print (screen, 112, 48, "Type:");
	sprintf (str, "%-13.13s", getmissiontype (config->type));
	screen->font = highlight ? highfont : font;
	screen->print (screen, 136, 48, str);
    }

    /* unit class */
    else {
	unitid = line - 2;
	sprintf (str, "Unit %d:", unitid + 1);
	screen->print (screen, 104, 72 + 16 * unitid, str);
	screen->font = highlight ? highfont : font;
	sprintf (str, "%-13.13s", getclassname (config->classes[unitid]));
	screen->print (screen, 136, 72 + 16 * unitid, str);
    }
}

/**
 * Show a tutorial message if it has not been seen before.
 * @param id ID of the tutorial message.
 */
static void showtutorial (int id)
{
    Config *config; /* config object */
    config = getconfig ();
    if (config->tutorial & (1 << id))
	return;
    display->message (tutorialtext[id]);
    config->tutorial |= (1 << id);
}

/*----------------------------------------------------------------------
 * Constructor Functions.
 */

/**
 * Display constructor.
 * @param colourset = 0 for mono, 1 for colour, 2 for nice colour.
 * @param quiet = 0 for sound and music, 1 for silence.
 * @return the new display.
 */
Display *new_Display (int colourset, int quiet)
{
    /* local variables */
    int mode; /* the screen mode */

    /* allocate memory */
    if (display)
	return display;
    if (! (display = malloc (sizeof (Display))))
	return NULL;

    /* initialise methods */
    display->destroy = destroy;
    display->setgame = setgame;
    display->update = update;
    display->message = message;
    display->busy = busy;
    display->error = error;
    display->menu = menu;
    display->confirm = confirm;
    display->startdelay = startdelay;
    display->enddelay = enddelay;
    display->showtitlescreen = showtitlescreen;
    display->titlekey = titlekey;
    display->playsound = playsound;
    display->showbackground = showbackground;
    display->showunit = showunit;
    display->showunitaction = showunitaction;
    display->showunithealth = showunithealth;
    display->showtitle = showtitle;
    display->typeset = typeset;
    display->preparemap = preparemap;
    display->updatemapsquare = updatemapsquare;
    display->showmap = showmap;
    display->showmapcursor = showmapcursor;
    display->hidemapcursor = hidemapcursor;
    display->showmapsquare = hidemapcursor;
    display->battleanimation = battleanimation;
    display->selectbody = selectbody;
    display->showinventory = showinventory;
    display->hideinventory = hideinventory;
    display->showinvcursor = showinvcursor;
    display->hideinvcursor = hideinvcursor;
    display->showinvitem = hideinvcursor;
    display->showcomputerprogress = showcomputerprogress;
    display->showmissionscores = showmissionscores;
    display->showscoretable = showscoretable;
    display->showconfigline = showconfigline;
    display->showtutorial = showtutorial;

    /* initialise class variables */
    controls = getcontrols ();

    /* prepare to initialise the CGALIB screen */
    if (colourset == 0)
	mode = 6;
    else if (colourset == 1)
	mode = 4;
    else if (colourset == 2)
	mode = 5;

    /* initialise the CGALIB screen */
    if (! (screen = new_Screen (mode, CGALIB_SHOWN))) {
	free (display);
	return NULL;
    }
    fatal_display (display);
    if (colourset == 2)
	screen->palette
	    (screen, CGALIB_LIGHT_CYAN_RED_WHITE, CGALIB_BLUE);
    screen->updates = 1;

    /* initialise the assets */
    loadassets ();

    /* set sound */
    gamesound = ! quiet;

    /* return the screen */
    return display;
}
