/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 10-Nov-2023..
 *
 * Utility Functions Header.
 */

#ifndef __UTILS_H__
#define __UTILS_H__

/*----------------------------------------------------------------------
 * Public Level Routines.
 */

/**
 * Roll a number of six-sided dice.
 * If min and max are equal, there is no range checking.
 * @param dice The number of dice to throw.
 * @param dm   Dice modifier.
 * @param min  Minimum throw allowed.
 * @param max  Maximum throw allowed.
 * @return     The dice throw.
 */
int throwdice (int dice, int dm, int min, int max);

/**
 * Roll a number of dice and return the highest value.
 * @param dice The number of dice to throw.
 * @return     The highest of the dice thrown.
 */
int highestdice (int dice);

/**
 * Roll 3 dice and return the total of the highest 2.
 * @return The highest 2 dice.
 */
int highest2of3 (void);

/**
 * Roll a number of dice and return the lowest value.
 * @param dice The number of dice to throw.
 * @return     The lowest of the dice thrown.
 */
int lowestdice (int dice);

/**
 * Take a skill roll.
 * @param  level THe skill level.
 * @return       0 = fail, 1 = pass, 2 = critical.
 */
int skillroll (int level);

/**
 * Write an integer as 2 bytes to an already open output file.
 * @param  value  A pointer to the integer variable to write.
 * @param  output The output file handle.
 * @return        1 if successful, 0 if not.
 */
int writeint (int *value, FILE *output);

/**
 * Write an integer as a byte to an already open output file.
 * @param  value  A pointer to the integer variable to write.
 * @param  output The output file handle.
 * @return        1 if successful, 0 if not.
 */
int writebyte (int *value, FILE *output);

/**
 * Write a string to an already open output file.
 * @param  value  A pointer to the string buffer to write.
 * @param  output The output file handle.
 * @return        1 if successful, 0 if not.
 */
int writestring (char *value, FILE *output);

/**
 * Read a 2-byte integer from an already open input file
 * and store it in an integer variable.
 * @param  value A pointer to the integer variable to store into.
 * @param  input The input file handle.
 * @return       1 if successful, 0 if not.
 */
int readint (int *value, FILE *input);

/**
 * Read a byte from an already open input file
 * and store it in an integer variable.
 * @param  value A pointer to the integer variable to store into.
 * @param  input The input file handle.
 * @return       1 if successful, 0 if not.
 */
int readbyte (int *value, FILE *input);

/**
 * Read a string from an already open input file
 * and store it in a string buffer.
 * @param  value A pointer to the string buffer to store into.
 * @param  input The input file handle.
 * @return       1 if successful, 0 if not.
 */
int readstring (char *value, FILE *input);

/**
 * Calculate the distance between two points.
 * @param firstx  First point X coordinate.
 * @param firsty  First point Y coordinate.
 * @param secondx Second point X coordinate.
 * @param secondy Second point Y coordinate.
 */
int distance (int firstx, int firsty, int secondx, int secondy);

#endif
