/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 16-Aug-2020.
 *
 * Game Header.
 */

/* structures defined in this header */
typedef struct game Game;

#ifndef __GAME_H__
#define __GAME_H__

/* included headers */
#include "levelmap.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/* external types */
typedef struct unit Unit;
typedef struct cabinet Cabinet;
typedef struct itemstack ItemStack;
typedef struct config Config;
typedef struct ai AI;

/**
 * @enum State
 * The game state, i.e. which screen is being shown.
 */
typedef enum {
    STATE_NONE, /* no state (quit signal) */
    STATE_MAP, /* player turn map screen */
    STATE_COMPUTER, /* computer is taking its turn */
    STATE_VICTORY, /* the game has been won */
    STATE_DEFEAT, /* the game has been lost */
    STATE_SCORES, /* viewing the high scores */
    STATE_NEWGAME, /* setting up a new game */
    STATE_COUNT /* count of the number of states */
} State;

/**
 * @enum MissionType is the type of mission.
 */
typedef enum {
    MISSION_RANDOM, /* no mission type set, choose at random */
    MISSION_ASSASSINATE, /* kill a particular enemy */
    MISSION_ANNIHILATE, /* kill as many enemies as possible */
    MISSION_HOSTAGE, /* rescue a single hostage */
    MISSION_RESCUE, /* rescue as many hostages as possible */
    MISSION_RETRIEVE, /* retrieve a particular data card */
    MISSION_GATHER, /* gather as many data cards as possible */
    MISSION_TYPES /* count of mission types */
} MissionType;

/**
 * @enum MissionLevel is the difficulty of a mission.
 */
typedef enum {
    MISSION_EASY, /* easy mission */
    MISSION_FAIR, /* medium level mission */
    MISSION_HARD, /* hard level mission */
    MISSION_LEVELS /* count of mission levels */
} MissionLevel;

/* Bit masks for elements on the map. */
#define MISSION_FURNITURE (0x40 | LEVELMAP_OPEN)
#define MISSION_UNIT 0x80

/* Cache values */
#define MISSION_CACHE_UNIT 0x10
#define MISSION_CACHE_ENEMY 0x20
#define MISSION_CACHE_HOSTAGE 0x40
#define MISSION_CACHE_CORPSE 0x80

/**
 * @enum FurnitureType The types of furniture.
 */
typedef enum {
    FURNITURE_CABINET,
    FURNITURE_CHAIR,
    FURNITURE_TABLE,
    FURNITURE_LAST /* placeholder */
} FurnitureType;

/** @enum ScoreID ID of the different score elements. */
typedef enum {
    SCORE_HEALTH, /* element for health score */
    SCORE_KILLS, /* element for kills score */
    SCORE_FREED, /* element for hostages score */
    SCORE_CARDS, /* element for cards collected score */
    SCORE_SPEED, /* element for speed score */
    SCORE_TOTAL /* element for total score */
} ScoreID;

/** @enum SideID ID of the side who is playing. */
typedef enum {
    SIDE_PLAYER, /* the player is playing */
    SIDE_COMPUTER /* the computer is playing */
} SideID;

/* display hooks */
typedef void (*DisplayHook) (int);

/**
 * @struct game is the structure of a full game.
 */
struct game {

    /*
     * Attributes
     */

    /** @var state The current game state. */
    int state;

    /** @var score For this game. */
    int score;

    /** @var highest 1 if this score is a high score. */
    int highest;

    /** @var turns The number of turns taken. */
    int turns;

    /** @var type The mission type */
    int type;

    /** @var level The difficulty level */
    int level;

    /** @var map The level map */
    LevelMap *map;

    /** @var entrance Location of the level entrance. */
    int entrance;

    /** @var units is a pointer to the team units. */
    Unit **units;

    /** @var unitcount is the size of the team. */
    int unitcount;

    /** @var unit The currently selected unit. */
    Unit *unit;

    /** @var enemycount The number of enemies. */
    int enemycount;

    /** @var enemies The enemy units. */
    Unit **enemies;

    /** @var ai AI for the enemy units. */
    AI **ai;

    /** @var hostagecount The number of hostages. */
    int hostagecount;

    /** @var hostages The hostage units. */
    Unit **hostages;

    /** @var cabinetcount The number of cabinets. */
    int cabinetcount;

    /** @var cabinets The cabinets on the level. */
    Cabinet **cabinets;

    /** @var stackcount The number of item stacks. */
    int stackcount;

    /** @var cabinets The item stacks in the level. */
    ItemStack **stacks;

    /** @var cardcount The number of cards to collect. */
    int cardcount;

    /** @var visibility A mask indicating units visible. */
    int visibility;

    /** @var attack A mask indicating who can attack. */
    int attack;

    /** @var conscious A mask indicating who is conscious. */
    int conscious;

    /** @var enemysight Lines of sight from each enemy. */
    int *sightlines;

    /** @var unitcache Cache for units on map. */
    char *unitcache;

    /** @var stackcache Cache for item stacks. */
    char *stackcache;

    /*
     * Methods
     */

    /**
     * Destroy a game when it is finished with.
     * @param game is the game to destroy.
     */
    void (*destroy) (Game *game);

    /**
     * Generate a new game at random.
     * @param game   The game object.
     * @param config The configuration.
     * @returns      1 on success, 0 on failure.
     */
    int (*generate) (Game *game, Config *config);

    /**
     * Load a game from a file.
     * @param game is the game to load.
     * @returns 1 on success, 0 on failure.
     */
    int (*load) (Game *game);

    /**
     * Save a game to a file.
     * @param game is the game to save.
     * @returns 1 on success, 0 on failure.
     */
    int (*save) (Game *game);

    /**
     * Get the number of unitcount in the team.
     * @param game is the game to examine.
     * @returns the number of units in the team.
     */
    int (*getunitcount) (Game *game);

    /**
     * Make the next available unit the current one.
     * @param game The game object.
     * @return     1 if another unit found, 0 otherwise.
     */
    int (*nextunit) (Game *game);

    /**
     * Prepare a new turn.
     * @param game The game object.
     */
    void (*initturn) (Game *game);

    /**
     * Check for line of sight between two points.
     * @param game    The game object.
     * @param origin  The point of origin.
     * @param dest    The destination points.
     * @return        1 if origin can see destination, 0 otherwise.
     */
    int (*lineofsight) (Game *game, int origin, int dest);

    /**
     * Clear the sight line grid.
     * @param game The game object.
     */
    void (*clearsightlines) (Game *game);

    /**
     * Check the whole level for lines of sight.
     * @param game    The game object.
     * @param origin  The point of origin.
     * @param mask    The bitmask.
     */
    void (*calcsightlines) (Game *game, int origin, int mask);

    /**
     * Check the sightlines to non-player units.
     * @param game The game object.
     * @param side Who is playing?
     */
    void (*calcunitsightlines) (Game *game, DisplayHook hook);

    /**
     * Check for change in unit visibility.
     * @param game       The game object.
     * @param origin     The origin square.
     * @param visibility The present visibility.
     * @return           The bits that were changed.
     */
    int (*changedvisibility)
    (Game *game, int origin, int visibility);

    /**
     * Return the briefing text for this game.
     * @param game    The game object.
     * @param buffer  The buffer for the game text.
     * @return        A pointer to the buffer.
     */
    char *(*briefing) (Game *game, char *buffer);

    /**
     * Return the unit at a particular location.
     * @param game    The game object.
     * @param x       The X coordinate.
     * @param y       The Y coordinate.
     * @return        The unit at that location, or NULL.
     */
    Unit *(*getunit) (Game *game, int x, int y);

    /**
     * Return the stack at the specified location.
     * @param game   The game object.
     * @param x      The X coordinate.
     * @param y      The Y coordinate.
     * @param create 1 to create a new stack if none exists.
     * @return       An item stack, or NULL if no room.
     */
    ItemStack *(*getstack) (Game *game, int x, int y, int create);

    /**
     * Remove a stack from the map.
     * @param game    The game object.
     * @param stack   The stack to remove.
     */
    void (*removestack) (Game *game, ItemStack *stack);

    /**
     * Return a bitfield for units that can attack or can see. Used as
     * a quick reference for opportunity fire, and for sighting enemy
     * units.
     * @param game   The game object.
     * @param action Minimum action points to qualify.
     * @return       A bitfield of units that can attack.
     */
    int (*activeunits) (Game *game, int action);

    /**
     * Check for opportunity attacks.
     * @param game    The game object.
     * @return        Pointer to the unit that attacked, or NULL.
     */
    Unit *(*opportunity) (Game *game);

    /**
     * Detect the end of a game.
     * @param game The game object.
     * @return     1 if the game is ended.
     */
    int (*detectend) (Game *game);

    /**
     * Ascertain if a mission is successful or not.
     * @param game The game object.
     * @return     1 on victory, 0 on defeat.
     */
    int (*ascertainvictory) (Game *game);

    /**
     * Calculate mission score after victory.
     * @param game   The game object.
     * @param scores An array of integers to store the scores in.
     */
    void (*calcscore) (Game *game, int *scores, int *performance);

    /**
     * Get score multiplier for the current mission.
     * @param game    The game object.
     * @param scoreid The mission score ID.
     * @return        The score multiplier.
     */
    int (*getmultiplier) (Game *game, int scoreid);

};

/*----------------------------------------------------------------------
 * Function Prototypes: Constructors.
 */

/**
 * Create a new game.
 * @returns the newly initialised game.
 */
Game *new_Game (void);

/**
 * Return the title of a particular mission type.
 * @param type The type ID of the mission.
 * @return     A pointer to the mission name.
 */
char *getmissiontype (int type);

#endif
