/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker, 2020.
 * Created: 29-Dec-2020.
 *
 * Graphical Display Handler Header.
 */

/* types defined here */
typedef struct display Display;

#ifndef __DISPLAY_H__
#define __DISPLAY_H__

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/* external types */
typedef struct game Game;
typedef struct unit Unit;
typedef struct scores Scores;
typedef struct config Config;
typedef struct cabinet Cabinet;

/**
 * @enum display_control is a control code.
 */
typedef enum display_control {
    DISPLAY_UP,
    DISPLAY_DOWN,
    DISPLAY_LEFT,
    DISPLAY_RIGHT,
    DISPLAY_FIRE
} DisplayControl;

/**
 * @enum display_sound Enumerates the sound effects.
 */
typedef enum display_sound {
    DISPLAY_PLOD,
    DISPLAY_PEWPEW,
    DISPLAY_STAB,
    DISPLAY_HIT,
    DISPLAY_UNCONSCIOUS,
    DISPLAY_DEATH,
    DISPLAY_COMMS
} DisplaySound;

/** @enum InvType Types of inventory display. */
typedef enum {
    DISPLAY_INVENTORY_SINGLE, /* unit inventory only */
    DISPLAY_INVENTORY_DUAL, /* unit inventory and adjacent friend */
    DISPLAY_INVENTORY_CABINET, /* unit inventory and cabinet */
    DISPLAY_INVENTORY_CHECK /* dual inventory requiring validation */
} InvType;

/**
 * @struct display is the display handler.
 */
typedef struct display_data DisplayData;
struct display {

    /** @var data is the private display data */
    DisplayData *data;

    /**
     * Destroy the display when no longer needed.
     */
    void (*destroy) ();

    /**
     * Give display routines access to the game data.
     * @param ingame The game being played.
     */
    void (*setgame) (Game *ingame);

    /**
     * Update the screen from the buffer.
     */
    void (*update) (void);

    /**
     * Display a message in the prompt box.
     * @param format The format of the output.
     * @param ...    The arguments to print with the format.
     */
    void (*message) (char *format, ...);

    /**
     * Display or clear a "busy" message.
     * @param message The message to display; blank to clear.
     */
    void (*busy) (char *message);

    /**
     * Display a brief error message.
     * @param format The format of the output.
     * @param ...    The arguments to print with the format.
     */
    void (*error) (char *format, ...);

    /**
     * Display a menu and get an option from it.
     * @param count   The number of options in the menu.
     * @param options The option names in an array.
     * @param initial The initial option selected.
     * @return        The number of the option selected.
     */
    int (*menu) (int count, char **options, int initial);

    /**
     * Display a simple yes/no confirmation menu.
     * @param text The message to display (12 chars max).
     * @return     1 if yes, 0 if no.
     */
    int (*confirm) (char *text);

    /**
     * Start a delay timer.
     * @param msecs Number of milliseconds to wait.
     */
    void (*startdelay) (int msecs);

    /**
     * Wait until a delay timer has finished.
     */
    void (*enddelay) (void);

    /**
     * Show the title screen.
     * @param display is the display to affect.
     */
    void (*showtitlescreen) (void);

    /**
     * Await the fire key at the title screen.
     * @param display is the display to affect.
     */
    void (*titlekey) (void);

    /**
     * Make a sound.
     * @param id The ID of the sound.
     */
    void (*playsound) (int id);

    /**
     * Show the standard screen background.
     */
    void (*showbackground) (void);

    /**
     * Show a unit in the unit window.
     * @param unit The unit to show.
     */
    void (*showunit) (Unit *unit);

    /**
     * Show a unit's action points.
     * @param unit The unit to show.
     */
    void (*showunitaction) (Unit *unit);

    /**
     * Show a unit's health points.
     * @param unit The unit to show.
     */
    void (*showunithealth) (Unit *unit);

    /**
     * Show a title above the text window.
     * @param text The title text.
     */
    void (*showtitle) (char *title);

    /**
     * Typeset a piece of text and display it.
     * @param text The text to typeset.
     */
    void (*typeset) (char *text);

    /**
     * Prepare a scrolling mission map.
     * @param game The game object.
     */
    void (*preparemap) (Game *game);

    /**
     * Update a single map square.
     * @param game    The game object.
     * @param x       X coordinate.
     * @param y       Y coordinate.
     */
    void (*updatemapsquare) (Game *game, int x, int y);

    /**
     * Show the map.
     * @param x X coordinate of top left corner.
     * @param y Y coordinate of top left corner.
     */
    void (*showmap) (int x, int y);

    /**
     * Show the map cursor.
     * @param cursorx X coordinate of the map cursor.
     * @param cursory Y coordinate of the map cursor.
     * @param viewx   X coordinate of the map view.
     * @param viewy   Y coordinate of the map view.
     */
    void (*showmapcursor)
    (int cursorx, int cursory, int viewx, int viewy);

    /**
     * Hide the map cursor.
     * @param cursorx X coordinate of the map cursor.
     * @param cursory Y coordinate of the map cursor.
     * @param viewx   X coordinate of the map view.
     * @param viewy   Y coordinate of the map view.
     */
    void (*hidemapcursor)
    (int cursorx, int cursory, int viewx, int viewy);

    /**
     * Show a map square.
     * @param cursorx X coordinate of the map cursor.
     * @param cursory Y coordinate of the map cursor.
     * @param viewx   X coordinate of the map view.
     * @param viewy   Y coordinate of the map view.
     */
    void (*showmapsquare)
    (int cursorx, int cursory, int viewx, int viewy);

    /**
     * Show a battle animation.
     * @param attacker The attacking unit.
     * @param defender The defender unit.
     * @param game     The game object.
     * @param viewx    View X coordinate.
     * @param viewy    View Y coordinate.
     */
    void (*battleanimation)
    (Unit *attacker,
     Unit *defender,
     Game *game,
     int viewx,
     int viewy);

    /**
     * Show a list of bodies in a pile and allow selection.
     * @param unit The unit at the top of the pile.
     * @return     The unit selected.
     */
    Unit *(*selectbody) (Unit *unit);

    /**
     * Show a second inventory.
     * @param invtype The inventory type (unit or cabinet).
     * @param unit    The unit, for unit inventories.
     * @param cabinet The cabinet, for cabinet inventories.
     */
    void (*showinventory) (int invtype, Unit *unit, Cabinet *cabinet);

    /**
     * Hide the second inventory.
     */
    void (*hideinventory) (void);

    /**
     * Show the inventory cursor.
     * @param cursor  The cursor.
     * @param invtype The inventory type.
     */
    void (*showinvcursor) (int cursor, int invtype);

    /**
     * Hide the inventory cursor.
     * @param cursor  The cursor position.
     * @param invtype The inventory type.
     * @param item    The item to show.
     */
    void (*hideinvcursor) (int cursor, int invtype, int item);

    /**
     * Hide the inventory cursor.
     * @param cursor  The cursor position.
     * @param invtype The inventory type.
     * @param item    The item to show.
     */
    void (*showinvitem) (int cursor, int invtype, int item);

    /**
     * Show computer turn progress.
     * @param percent Percentage points.
     */
    void (*showcomputerprogress) (int percent);

    /**
     * Show the scores for the current mission.
     * @param game         The game object.
     * @param scores       The array of scores.
     * @param performmance Detailed performance stats
     * @param victory      1 if the game was won, 0 if lost.
     */
    void (*showmissionscores)
    (Game *game,
     int *scores,
     int *performance,
     int victory);

    /**
     * Show the high score table.
     * @param scores The score table object.
     * @param game   The current game.
     */
    void (*showscoretable) (Scores *scores, Game *game);

    /**
     * Show a line of the configuration.
     * @param config    The configuration object.
     * @param line      The line of the configuration.
     * @param highlight 1 if highlighted, 0 if not.
     */
    void (*showconfigline) (Config *config, int line, int highlight);

    /**
     * Show a tutorial message if it has not been seen before.
     * @param id ID of the tutorial message.
     */
    void (*showtutorial) (int id);

};

/*----------------------------------------------------------------------
 * Constructor Declarations.
 */

/**
 * Display constructor.
 * @param colourset = 0 for mono, 1 for colour, 2 for nice colour.
 * @param quiet = 0 for sound and music, 1 for silence.
 * @return the new display.
 */
Display *new_Display (int colourset, int quiet);

#endif
