/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 31-May-2024.
 *
 * AI Header.
 */

/* types defined by this header */
typedef struct ai AI;

#ifndef __AI_H__
#define __AI_H__

/* required headers */
#include <stdio.h>

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/* external types */
typedef struct game Game;
typedef struct unit Unit;

/**
 * @enum OrderID
 * AI order identifiers.
 */
typedef enum {
    AI_ORDER_NONE, /* doing nothing */
    AI_ORDER_GUARD, /* guarding a location */
    AI_ORDER_PATROL /* patrolling the area */
} OrderID;

/**
 * @enum ActionID
 * AI action IDs.
 */
typedef enum {
    AI_ACTION_MOVE, /* move to position */
    AI_ACTION_ATTACK /* attack unit at position */
} ActionID;

/**
 * @struct ai
 * The AI object structure.
 */
struct ai {

    /*
     * Attributes
     */

    /** @var game The game object. */
    Game *game;

    /** @var unit The unit this AI controls. */
    Unit *unit;

    /** @var order The main order. */
    int order;

    /** @var orderx X coordinate of order. */
    int orderx;

    /** @var ordery Y coordinate of order. */
    int ordery;

    /**
     * @var investigating
     * 1 if investigating an incident,
     * 0 otherwise.
     */
    int investigating;

    /** @var investx X coordinate of investigation. */
    int investx;

    /** @var investy Y coordinate of investigation. */
    int investy;

    /*
     * Methods
     */

    /**
     * Destroy the AI when no longer needed.
     * @param ai The AI to destroy.
     */
    void (*destroy) (AI *ai);

    /**
     * Write the AI to an already open file.
     * @param ai     The AI to write.
     * @param output The output file.
     * @return       1 if successful, 0 otherwise.
     */
    int (*write) (AI *ai, FILE *output);

    /**
     * Read the AI to an already open file.
     * @param ai    The AI to read.
     * @param input The input file.
     * @return      1 if successful, 0 otherwise.
     */
    int (*read) (AI *ai, FILE *input);

    /**
     * Get a single action.
     * @param ai     The AI object.
     * @param option Pointer to The option.
     * @param x      Pointer to the target X coordinate.
     * @param y      Pointer to the target Y coordinate.
     * @return       1 if action decided, 0 otherwise.
     */
    int (*getaction) (AI *ai, int *action, int *x, int *y);

    /**
     * Find a new patrol location for the AI.
     * @param ai The AI object.
     */
    void (*newpatrollocation) (AI *ai);

};

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Alert nearby units to a noise.
 * @param x X coordinate of the noise.
 * @param y Y coordinate of the noise.
 */
void ai_alert (int x, int y);

/**
 * AI Constructor.
 * @return The new AI.
 */
AI *new_AI (void);

#endif
