/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 17-Aug-2020.
 *
 * Unit Module.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* project headers */
#include "sccc.h"
#include "fatal.h"
#include "game.h"
#include "unit.h"
#include "skill.h"
#include "item.h"
#include "cabinet.h"
#include "stack.h"
#include "ai.h"
#include "utils.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var initalstats Initial stats for each class. */
static int initialstats[6][3] = {
    {2, 2, 7}, /* Medic */
    {2, 7, 2}, /* Scout */
    {5, 5, 5}, /* Auxiliary */
    {7, 2, 2}, /* Infantry */
    {2, 2, 2}, /* Enemy */
    {2, 2, 2} /* Hostage */
};

/**
 * @var primaryskills
 * The primary skill for each character class. One third of a
 * character's skill pointe will go straight on this skill.
 */
static int primaryskills[] = {
    SKILL_MEDICINE, /* medic has medicine */
    SKILL_STEALTH, /* scout has stealth */
    SKILL_LIGHT, /* auxiliary has light firearms */
    SKILL_HEAVY /* infantry has heavy firearms */
};

/**
 * @var secondary skills
 * The secondary skills for each character class. One third of a
 * character's skill points will be chosen from this table. The
 * entries are ordered as per the rolls 2-12 on 2D6.
 */
static int secondaryskills[5][11] = {

    /* Medic skills */
    {SKILL_MEDICINE, SKILL_MEDICINE, SKILL_MEDICINE,
     SKILL_LIGHT, SKILL_LIGHT,
     SKILL_BLADE,
     SKILL_UNARMED, SKILL_UNARMED,
     SKILL_BLADE,
     SKILL_MEDICINE, SKILL_MEDICINE},

    /* Scout skills */
    {SKILL_STEALTH, SKILL_STEALTH, SKILL_STEALTH, SKILL_STEALTH,
     SKILL_BLADE, SKILL_BLADE,
     SKILL_UNARMED, SKILL_UNARMED, SKILL_UNARMED,
     SKILL_STEALTH,
     SKILL_BLADE},

    /* Auxiliary skills */
    {SKILL_LIGHT, SKILL_LIGHT, SKILL_LIGHT, SKILL_LIGHT,
     SKILL_BLADE, SKILL_BLADE,
     SKILL_UNARMED, SKILL_UNARMED, SKILL_UNARMED,
     SKILL_LIGHT,
     SKILL_BLADE},

    /* Infantry Skills */
    {SKILL_HEAVY, SKILL_HEAVY, SKILL_HEAVY,
     SKILL_LIGHT, SKILL_LIGHT,
     SKILL_BLADE,
     SKILL_UNARMED, SKILL_UNARMED,
     SKILL_BLADE,
     SKILL_HEAVY, SKILL_HEAVY},

    /* Enemy Skills */
    {SKILL_HEAVY, SKILL_HEAVY, SKILL_HEAVY,
     SKILL_LIGHT, SKILL_LIGHT,
     SKILL_BLADE,
     SKILL_UNARMED, SKILL_UNARMED,
     SKILL_BLADE,
     SKILL_HEAVY, SKILL_HEAVY}

};

/**
 * @var tertiaryskills
 * The tertiary skills for any character class. One third of a
 * character's skill points will be chosen from this table. The
 * entries are ordered as per the rolls 2-12 on 2D6.
 */
static int tertiaryskills[] = {
    SKILL_STEALTH,
    SKILL_UNARMED,
    SKILL_BLADE,
    SKILL_UNARMED,
    SKILL_STEALTH,
    SKILL_MEDICINE,
    SKILL_LIGHT,
    SKILL_HEAVY,
    SKILL_BLADE,
    SKILL_HEAVY,
    SKILL_LIGHT
};

/** @var classnames An array of class names. */
static char *classnames[] = {
    "Random",
    "Medic",
    "Scout",
    "Auxil'ry",
    "Soldier",
    "Enemy",
    "Hostage"
};

/** @enum ErrorCode */
typedef enum {
    ERROR_NONE, /* no error */
    ERROR_ACTIONS, /* insufficient action points */
    ERROR_BLOCKED, /* movement blocked */
    ERROR_DEPLOYED, /* unit already deployed */
    ERROR_NOTDEPLOYED, /* unit not deployed */
    ERROR_NOTEXIT, /* unit not at exit */
    ERROR_RIFLE, /* rifle needs both hands */
    ERROR_CLOSE, /* too close for laser rifle */
    ERROR_FAR, /* unit cannot step this far */
    ERROR_HANDS, /* unit's hands are full */
    ERROR_COMBAT, /* combat fails for unknown reason */
    ERROR_NOITEM, /* no item is selected */
    ERROR_NOROOM, /* no room for the item */
    ERROR_ENEMY, /* enemies do not take orders from you */
    ERROR_HOSTAGE, /* hostage must be freed first */
    ERROR_EQUIPPED, /* the item is already equipped */
    ERROR_BACKPACK, /* the item is not in the backpack */
    ERROR_UNEQUIPPED, /* item is not equipped */
    ERROR_UNCONSCIOUS, /* the unit is unconscious */
    ERROR_DEAD, /* the unit is dead */
    ERROR_FREE, /* the unit is already free */
    ERROR_BLADE, /* a blade is required to free a hostage */
    ERROR_HEALTHY, /* the unit is already healthy */
    ERROR_MEDIKIT, /* a medikit is needed to heal */
    ERROR_HEALENEMY /* healing enemies would be unwise */
} ErrorCode;

/** @var errortext The text for the various errors. */
static char *errortext[] = {
    "",
    "Not enough action points",
    "The way is blocked",
    "This unit is already deployed",
    "This unit is not deployed",
    "This unit is not at the exit",
    "The Laser Rifle requires both hands",
    "The Laser Rifle cannot fire at point blank range",
    "Target must be adjacent",
    "Hands are full",
    "Cannot attack that target",
    "No item is selected",
    "No room for the item",
    "This unit doesn't take orders from you",
    "This hostage must be freed first",
    "This item is already equipped",
    "This item is not in the unit's backpack",
    "This item is not currently equipped",
    "This unit is unconscious",
    "This unit is dead",
    "This unit is already free",
    "A blade is required to cut the bonds",
    "This unit is already healthy",
    "A medikit is needed to heal",
    "Healing an enemy would be unwise"
};

/** @var error The last error that occurred. */
static int error;

/*----------------------------------------------------------------------
 * Level 3 Functions.
 */

/**
 * Identify what phase of skill generation this skill belongs.
 * @param  class The unit class concerned.
 * @param  id    The skill id.
 * @return phase The phase of skill generation.
 */
static int skillphase (int class, int id)
{
    int s, /* skill counter */
	phase = 3; /* phase to which skill belongs */
    if (class < UNIT_ENEMY && id == primaryskills[class - 1])
	phase = 1;
    else if (class < UNIT_HOSTAGE)
	for (s = 0; s < 11; ++s)
	    if (id == secondaryskills[class - 1][s])
		phase = 2;
    return phase;
}

/*----------------------------------------------------------------------
 * Level 2 Functions.
 */

/**
 * Increment a statistic if it is not at maximum.
 * @param stat Pointer to the stat to increment.
 * @return 1 if the stat was incremented, 0 if at max already.
 */
static int incstat (int *stat)
{
    if (*stat >= 12)
	return 0;
    ++*stat;
    return 1;
}

/**
 * Increment a skill level, adding the skill if necessary.
 * @param player The player player to receive the skill point.
 * @param id The ID of the skill to receive the skill point.
 * @return 1 if skill added, 0 if not.
 */
static int inclevel (Unit *unit, SkillId id)
{
    int s, /* skill slot counter */
	phase, /* phase to which skill belongs */
	ephase, /* phase of existing skill */
	limit; /* limit for this skill */

    /* ensure that the skill is valid */
    if (id <= SKILL_NONE || id > SKILL_UNARMED)
	return 0;

    /* determine the highest level of this skill allowed */
    limit = 6;
    phase = skillphase (unit->class, id);
    for (s = 0; s < 4; ++s)
	if (unit->skills[s]) {
	    ephase = skillphase (unit->class, unit->skills[s]->id);
	    if (ephase == phase - 1 && unit->skills[s]->level < limit)
		limit = unit->skills[s]->level;
	}

    /* check each existing skill slot */
    for (s = 0; s < 4; ++s) {

	/* if we reach an empty skill slot, add new skill */
	if (! unit->skills[s]) {
	    unit->skills[s] = new_Skill ();
	    unit->skills[s]->id = id;
	    unit->skills[s]->level = 1;
	    return 1;
	}

	/* if the unit has the skill slot already, level up */
	else if (unit->skills[s]->id == id) {
	    if (unit->skills[s]->level >= limit)
		return 0;
	    ++unit->skills[s]->level;
	    return 1;
	}
    }

    /* we didn't manage to add the skill point */
    return 0;
}

/**
 * Sort the skills in descending order of level.
 * @param unit The unit to affect.
 */
static void sortskills (Unit *unit)
{
    int s, /* skill counter */
	sorted; /* 1 if the list is sorted */
    Skill *swap; /* temporary swap value */
    do {
	sorted = 1;
	for (s = 0; s < 3; ++s)
	    if ((unit->skills[s] &&
		 unit->skills[s + 1] &&
		 unit->skills[s]->level <
		 unit->skills[s + 1]->level) ||
		(unit->skills[s + 1] && ! unit->skills[s]))
	    {
		swap = unit->skills[s];
		unit->skills[s] = unit->skills[s + 1];
		unit->skills[s + 1] = swap;
		sorted = 0;
	    }
    } while (! sorted);
}

/**
 * Generate the defensive equipment for a unit.
 * @param unit The unit to affect.
 */
static void generatedefence (Unit *unit)
{
    switch (unit->class) {
    case UNIT_MEDIC:
	if (rand () % 6 >= 4)
	    unit->inventory[UNIT_HEAD] = ITEM_HELMET;
	if (rand () % 6 >= 4)
	    unit->inventory[UNIT_BODY] = ITEM_ARMOUR;
	break;
    case UNIT_SCOUT:
	break;
    case UNIT_AUXILIARY:
	if (rand () % 6 >= 2)
	    unit->inventory[UNIT_HEAD] = ITEM_HELMET;
	if (rand () % 6 >= 2)
	    unit->inventory[UNIT_BODY] = ITEM_ARMOUR;
	break;
    case UNIT_INFANTRY:
	unit->inventory[UNIT_HEAD] = ITEM_HELMET;
	unit->inventory[UNIT_BODY] = ITEM_ARMOUR;
	break;
    case UNIT_ENEMY:
	if (rand () % 6 >= 3)
	    unit->inventory[UNIT_HEAD] = ITEM_HELMET;
	if (rand () % 6 >= 3)
	    unit->inventory[UNIT_BODY] = ITEM_ARMOUR;
	break;
    }
}

/**
 * Generate a unit's weaponry.
 * @param unit The unit affected.
 */
static void generateweapon (Unit *unit)
{
    int max = 0, /* maximum weapon skill found so far */
	id = 0, /* id of highest weapon skill */
	c; /* skill counter */

    /* ascertain the highest weapon skill */
    for (c = 0; c < 4; ++c)
	if (unit->skills[c] && unit->skills[c]->level > max &&
	   (unit->skills[c]->id == SKILL_HEAVY ||
	    unit->skills[c]->id == SKILL_LIGHT ||
	    unit->skills[c]->id == SKILL_BLADE)) {
	    max = unit->skills[c]->level;
	    id = unit->skills[c]->id;
	}

    /* give the character a weapon */
    switch (id) {
    case SKILL_HEAVY:
	unit->inventory[UNIT_HAND] = ITEM_RIFLE;
	break;
    case SKILL_LIGHT:
	unit->inventory[UNIT_HAND] = ITEM_PISTOL;
	break;
    case SKILL_BLADE:
	unit->inventory[UNIT_HAND] = ITEM_BLADE;
	break;
    }
}

/**
 * Generate a medikit for a unit.
 * @param unit The unit affected.
 */
static void generatemedikit (Unit *unit)
{
    int c; /* skill counter */
    for (c = 0; c < 4; ++c)
	if (unit->skills[c] &&
	    unit->skills[c]->id == SKILL_MEDICINE &&
	    unit->skills[c]->level > 3) {
	    if (unit->inventory[UNIT_HAND])
		unit->inventory[UNIT_OFFHAND] = ITEM_MEDIKIT;
	    else
		unit->inventory[UNIT_HAND] = ITEM_MEDIKIT;
	}
}

/**
 * Find a free slot for an item.
 * @param unit The unit object.
 * @param item The item.
 * @return     The inventory slot to put the item, or 10 if no room.
 */
static int freeslot (Unit *unit, int item, int start)
{
    int c; /* counter */
    for (c = start; c < 10; ++c)
	if (unit->inventory[c] == ITEM_NONE) {
	    if (c == 0 && item == ITEM_HELMET)
		return c;
	    else if (c == 1 &&
		     item == ITEM_ARMOUR &&
		     unit->inventory[2] == ITEM_NONE)
		continue;
	    else if (c == 2 && item == ITEM_ARMOUR)
		return c;
	    else if (c != 0 && c != 2)
		return c;
	}
    return 10;
}

/**
 * Calculate a unit's encumbrance.
 * @param unit The unit object.
 */
static void calcencumbrance (Unit *unit)
{
    int c, /* item count */
	carried = 0; /* number of items carried */
    for (c = 0; c < 10; ++c)
	if (unit->inventory[c])
	    ++carried;
    unit->encumbrance
	= (carried > unit->strength - 2)
	? carried - (unit->strength - 2)
	: 0;
}

/*----------------------------------------------------------------------
 * Level 1 Functions.
 */

/**
 * Clear the old unit data.
 * @param unit The unit object.
 */
static void clearunit (Unit *unit)
{
    int c; /* general counter */
    for (c = 0; c < 4; ++c)
	if (unit->skills[c]) {
	    unit->skills[c]->destroy (unit->skills[c]);
	    unit->skills[c] = NULL;
	}
}

/**
 * Generate a character name.
 * @param name Pointer to buffer for player name.
 * @return     The player name.
 */
static char *generatename (char *name)
{
    Names *names; /* pointer to name generator */
    names = getnamegenerator ();
    names->minlength = 4;
    names->maxlength = 12;
    names->generate (names, name);
    *name = toupper (*name);
    return name;
}

/**
 * Generate a unit's statistics.
 * @param unit The unit object.
 */
static void generatestats (Unit *unit)
{
    int c, /* counter variable */
	s; /* stat to increase */

    /* generate main attributes */
    unit->strength = initialstats[unit->class - 1][0];
    unit->agility = initialstats[unit->class - 1][1];
    unit->endurance = initialstats[unit->class - 1][2];
    c = unit->strength + unit->agility + unit->endurance;
    while (c < 21) {
	s = rand () % 3;
	switch (s) {
	case 0:
	    c += incstat (&unit->strength);
	    break;
	case 1:
	    c += incstat (&unit->agility);
	    break;
	case 2:
	    c += incstat (&unit->endurance);
	    break;
	}
    }
    unit->action = 14 + unit->agility;
    unit->health = 7 + unit->endurance + unit->strength;
}

/**
 * Generate the skills.
 * @param unit The unit object.
 */
static void generateskills (Unit *unit)
{
    int s, /* skill id */
	c; /* counter */

    /* primary skills */
    c = 0;
    if (unit->class < UNIT_ENEMY)
	s = primaryskills[unit->class - 1];
    else if (unit->class < UNIT_HOSTAGE)
	s = secondaryskills[unit->class - 1][throwdice (2, -2, 0, 0)];
    else
	s = tertiaryskills[throwdice (2, -2, 0, 0)];
    while (c < 2 + unit->level)
	if (inclevel (unit, s))
	    ++c;

    /* secondary skills */
    while (c < 4 + 2 * unit->level) {
	if (unit->class < UNIT_HOSTAGE)
	    s = secondaryskills[unit->class - 1][throwdice (2, -2, 0, 0)];
	else
	    s = tertiaryskills[throwdice (2, -2, 0, 0)];
	if (inclevel (unit, s))
	    ++c;
    }

    /* tertiary skills */
    while (c < 6 + 3 * unit->level) {
	s = tertiaryskills[throwdice (2, -2, 0, 0)];
	if (inclevel (unit, s))
	    ++c;
    }

    /* sort skills */
    sortskills (unit);
}

/**
 * Generate the equipment.
 * @param unit The unit object.
 */
static void generateitems (Unit *unit)
{
    generatedefence (unit);
    generateweapon (unit);
    generatemedikit (unit);
    calcencumbrance (unit);
}

/**
 * Determine the combat skill to be used.
 * @param unit The unit object.
 * @param dist The distance at which combat occurs.
 * @return     The weapon to be used.
 */
static int combattype (Unit *unit, int dist)
{
    /* laser rifle fire */
    if (dist > 1 &&
	((unit->inventory[1] == ITEM_RIFLE &&
	  unit->inventory[3] == ITEM_NONE) ||
	 (unit->inventory[3] == ITEM_RIFLE &&
	  unit->inventory[1] == ITEM_NONE)))
	return SKILL_HEAVY;

    /* laser pistol fire */
    else if (unit->inventory[1] == ITEM_PISTOL ||
	     unit->inventory[3] == ITEM_PISTOL)
	return SKILL_LIGHT;

    /* knife attack */
    else if (dist == 1 &&
	     (unit->inventory[1] == ITEM_BLADE ||
	      unit->inventory[3] == ITEM_BLADE))
	return SKILL_BLADE;

    /* unarmed attack */
    else if (dist == 1 &&
	     (unit->inventory[1] == ITEM_NONE ||
	      unit->inventory[3] == ITEM_NONE))
	return SKILL_UNARMED;

    /* unit needs both hands for the laser rifle */
    else if ((unit->inventory[1] == ITEM_RIFLE &&
	       unit->inventory[3] != ITEM_NONE) ||
	      (unit->inventory[3] == ITEM_RIFLE &&
	       unit->inventory[1] != ITEM_NONE))
	error = ERROR_RIFLE;

    /* unit too close to attack with laser rifle */
    else if (dist == 1 &&
	((unit->inventory[1] == ITEM_RIFLE &&
	  unit->inventory[3] == ITEM_NONE) ||
	 (unit->inventory[3] == ITEM_RIFLE &&
	  unit->inventory[1] == ITEM_NONE)))
	error = ERROR_CLOSE;

    /* unit too far for blade combat */
    else if (dist > 1 &&
	     (unit->inventory[1] == ITEM_BLADE ||
	      unit->inventory[3] == ITEM_BLADE))
	error = ERROR_FAR;

    /* unit too far for unarmed combat */
    else if (dist > 1 &&
	     (unit->inventory[1] == ITEM_NONE ||
	      unit->inventory[3] == ITEM_NONE))
	error = ERROR_HANDS;

    /* unknown error */
    else
	error = ERROR_COMBAT;

    /* return attack failure */
    return SKILL_NONE;
}

/**
 * Process a hit in combat.
 * @param unit   The unit that is attacking.
 * @param target The target of the attack.
 * @param game   The game object.
 * @return       1 if attack made, 0 if not.
 */
static int combat (Unit *unit, Unit *target, Game *game)
{
    int attskill, /* the skill level used for attacking */
	attdamage, /* potential attack damage */
	attroll, /* the attack roll */
	defdamage, /* amount of damage stopped by defence */
	conscious; /* health level for guaranteed consciousness */
    char *template, /* template for feedback */
	*attverb; /* shot, stabbed, hit */

    /* determine the skill to be used for attack */
    attskill = unit->getskill (unit, unit->combattype);
    switch (unit->combattype) {
    case SKILL_HEAVY:
	attdamage = throwdice (3, 0, 0, 0);
	attverb = "shot";
	break;
    case SKILL_LIGHT:
	attdamage = throwdice (2, 0, 0, 0);
	attverb = "shot";
	break;
    case SKILL_BLADE:
	attdamage = highest2of3 ();
	attverb = "stabbed";
	break;
    case SKILL_UNARMED:
	attdamage = highestdice (2);
	attverb = "hit";
	break;
    default:
	unit->feedbackid = UNIT_FEEDBACK_MISS;
	sprintf
	    (unit->feedback,
	     "%s missed %s",
	     unit->name,
	     target->name);
	return ERROR_NONE;
    }

    /* calculate the unit's defence */
    defdamage = 0;
    if (target->inventory[0] == ITEM_HELMET)
	defdamage += lowestdice (2);
    if (target->inventory[2] == ITEM_ARMOUR)
	defdamage += highestdice (2);

    /* make the skill roll */
    attroll = skillroll (attskill);
    if (attroll == 0)
	attdamage = 0;
    else if (attroll == 1);
    else if (attroll == 2)
	attdamage = target->health + defdamage;
    if (defdamage > attdamage)
	defdamage = attdamage;

    /* spend action points and apply the damage to the target */
    unit->action -= 6 + unit->encumbrance + unit->creeping;
    conscious = (7 + target->strength + target->endurance) / 2;
    target->health -= attdamage - defdamage;
    if (target->health < 0)
	target->health = 0;
    target->surprise = UNIT_NOTSURPRISED;

    /* determine feedback template */
    if (target->health == 0) {
	unit->feedbackid = UNIT_FEEDBACK_HIT;
	template = "%s fatally %s %s";
	game->map->blocks
	    [target->x + game->map->width * target->y]
	    &= 0x7f;
	if (game->state == STATE_COMPUTER)
	    game->conscious &= ~(1 << target->id);
    } else if (! target->unconscious &&
	       target->health < conscious &&
	       throwdice (2, 0, 0, 0) >= target->endurance)
    {
	unit->feedbackid = UNIT_FEEDBACK_HIT;
	template = "%s %s %s, who falls unconscious";
	target->unconscious = 1;
	game->map->blocks
	    [target->x + game->map->width * target->y]
	    &= 0x7f;
	if (game->state == STATE_COMPUTER)
	    game->conscious &= ~(1 << target->id);
    } else if (attdamage && attdamage == defdamage) {
	unit->feedbackid = UNIT_FEEDBACK_HIT;
	template = "%s %s %s, but caused no injury";
    } else if (attdamage) {
	unit->feedbackid = UNIT_FEEDBACK_HIT;
	template = "%s %s %s";
    } else {
	unit->feedbackid = UNIT_FEEDBACK_MISS;
	template = "%s %s at %s, and missed";
    }

    /* alert nearby enemy units to the noise */
    if (unit->class != UNIT_ENEMY &&
	(unit->combattype == SKILL_HEAVY ||
	 unit->combattype == SKILL_LIGHT ||
	 (target->health == 0 &&
	 ! unit->creeping)))
	ai_alert (unit->x, unit->y);

    /* create feedback and reterun */
    sprintf
	(unit->feedback,
	 template,
	 unit->name,
	 attverb,
	 target->name);
    return ERROR_NONE;
}

/**
 * Transfer an item from the unit to a cabinet.
 * @param unit     The unit.
 * @param target   The target unit.
 * @param fromslot The slot from which the item is taken.
 * @param toslot   Pointer to destination slot.
 * @return         1 if successful, 0 on failure.
 */
static int transfertounit
(Unit *unit, Unit *target, int fromslot, int *toslot)
{
    int empty, /* empty item slot */
	item; /* the item to transfer */

    /* check that an item exists */
    item = unit->inventory[fromslot];
    if (! item) {
	error = ERROR_NOITEM;
	return 0;
    }

    /* check that there's room for the item */
    empty = freeslot (target, item, 0);
    if (empty == 10) {
	error = ERROR_NOROOM;
	return 0;
    }

    /* transfer the item */
    --unit->action;
    target->inventory[empty] = item;
    unit->inventory[fromslot] = ITEM_NONE;
    *toslot = 10 + empty;
    calcencumbrance (unit);
    calcencumbrance (target);
    return 1;
}

/**
 * Transfer an item to the unit from another unit.
 * @param unit     The unit.
 * @param target   The target unit.
 * @param fromslot The slot from which the item is taken.
 * @param toslot   Pointer to destination slot.
 * @return         1 if successful, 0 on failure.
 */
static int transferfromunit
(Unit *unit, Unit *target, int fromslot, int *toslot)
{
    int empty, /* empty slot */
	item; /* item being transferred */

    /* check that an item exists */
    item = target->inventory[fromslot - 10];
    if (item == ITEM_NONE) {
	error = ERROR_NOITEM;
	return 0;
    }

    /* check that there's room for the item */
    empty = freeslot (unit, item, 0);
    if (empty == 10) {
	error = ERROR_NOROOM;
	return 0;
    }

    /* check for action points */
    if (unit->action < 1) {
	error = ERROR_ACTIONS;
	return 0;
    }

    /* transfer the item */
    --unit->action;
    unit->inventory[empty] = item;
    target->inventory[fromslot - 10] = ITEM_NONE;
    *toslot = empty;
    calcencumbrance (unit);
    calcencumbrance (target);
    return 1;
}

/**
 * Transfer an item from the unit to a cabinet.
 * @param unit The unit.
 * @param cabinet  The cabinet.
 * @param fromslot The slot from which the item is taken.
 * @param toslot   Pointer to destination slot.
 * @return         1 if successful, 0 on failure.
 */
static int transfertocabinet
(Unit *unit, Cabinet *cabinet, int fromslot, int *toslot)
{
    int c; /* counter */

    /* check that an item exists */
    if (! unit->inventory[fromslot]) {
	error = ERROR_NOITEM;
	return 0;
    }

    /* check that there's room for the item */
    for (c = 0; c < 12; ++c)
	if (cabinet->items[c] == ITEM_NONE)
	    break;
    if (c == 12) {
	error = ERROR_NOROOM;
	return 0;
    }

    /* check for action points */
    if (unit->action < 1) {
	error = ERROR_ACTIONS;
	return 0;
    }

    /* transfer the item */
    --unit->action;
    cabinet->items[c] = unit->inventory[fromslot];
    unit->inventory[fromslot] = ITEM_NONE;
    *toslot = 10 + c;
    calcencumbrance (unit);
    return 1;
}

/**
 * Transfer an item to the unit from a cabinet.
 * @param unit The unit.
 * @param cabinet  The cabinet.
 * @param fromslot The slot from which the item is taken.
 * @param toslot   Pointer to destination slot.
 * @return         1 if successful, 0 on failure.
 */
static int transferfromcabinet
(Unit *unit, Cabinet *cabinet, int fromslot, int *toslot)
{
    int empty, /* empty slot */
	item; /* item being transferred */

    /* check that an item exists */
    item = cabinet->items[fromslot - 10];
    if (item == ITEM_NONE) {
	error = ERROR_NOITEM;
	return 0;
    }

    /* check that there's room for the item */
    empty = freeslot (unit, item, 0);
    if (empty == 10) {
	error = ERROR_NOROOM;
	return 0;
    }

    /* check for action points */
    if (unit->action < 1) {
	error = ERROR_ACTIONS;
	return 0;
    }

    /* transfer the item */
    --unit->action;
    unit->inventory[empty] = cabinet->items[fromslot - 10];
    cabinet->items[fromslot - 10] = ITEM_NONE;
    *toslot = empty;
    calcencumbrance (unit);
    return 1;
}

/*----------------------------------------------------------------------
 * Public Level Functions.
 */

/**
 * Destroy the unit when it is no longer needed.
 * @param unit is the unit to destroy.
 */
static void destroy (Unit *unit)
{
    clearunit (unit);
    free (unit);
}

/**
 * Write the unit to an open file.
 * @param unit   The unit object.
 * @param output The output file handle.
 * @return       1 on success, 0 on failure.
 */
static int write (Unit *unit, FILE *output)
{
    int skillcount, /* number of skills */
	c; /* general counter */

    /* write basic character details */
    if (! writebyte (&unit->id, output))
	return 0;
    if (! writestring (unit->name, output))
	return 0;
    if (! writebyte (&unit->level, output))
	return 0;
    if (! writebyte (&unit->class, output))
	return 0;
    if (! writebyte (&unit->hostage, output))
	return 0;
    if (! writebyte (&unit->unconscious, output))
	return 0;
    if (! writebyte (&unit->encumbrance, output))
	return 0;
    if (! writebyte (&unit->surprise, output))
	return 0;
    if (! writebyte (&unit->creeping, output))
	return 0;

    /* write the statistics */
    if (! writebyte (&unit->strength, output))
	return 0;
    if (! writebyte (&unit->agility, output))
	return 0;
    if (! writebyte (&unit->endurance, output))
	return 0;
    if (! writebyte (&unit->action, output))
	return 0;
    if (! writebyte (&unit->health, output))
	return 0;

    /* write the skills */
    skillcount
	= (unit->skills[0] != NULL)
	+ (unit->skills[1] != NULL)
	+ (unit->skills[2] != NULL)
	+ (unit->skills[3] != NULL);
    if (! writebyte (&skillcount, output))
	return 0;
    for (c = 0; c < skillcount; ++c)
	if (unit->skills[c] &&
	    ! unit->skills[c]->write (unit->skills[c], output))
	    return 0;

    /* write the inventory */
    for (c = 0; c < 10; ++c)
	if (! writebyte (&unit->inventory[c], output))
	    return 0;

    /* write the unit location */
    if (! writebyte (&unit->x, output))
	return 0;
    if (! writebyte (&unit->y, output))
	return 0;
    if (! writebyte (&unit->prone, output))
	return 0;

    /* success! */
    return 1;
}

/**
 * Read the unit from an open file.
 * @param unit  The unit object.
 * @param input The input file handle.
 * @return      1 on the success, 0 on failure.
 */
static int read (Unit *unit, FILE *input)
{
    int skillcount, /* number of skills */
	c; /* general counter */

    /* clear the old unit data */
    clearunit (unit);

    /* read basic character details */
    if (! readbyte (&unit->id, input))
	return 0;
    if (! readstring (unit->name, input))
	return 0;
    if (! readbyte (&unit->level, input))
	return 0;
    if (! readbyte (&unit->class, input))
	return 0;
    if (! readbyte (&unit->hostage, input))
	return 0;
    if (! readbyte (&unit->unconscious, input))
	return 0;
    if (! readbyte (&unit->encumbrance, input))
	return 0;
    if (! readbyte (&unit->surprise, input))
	return 0;
    if (! readbyte (&unit->creeping, input))
	return 0;

    /* read the statistics */
    if (! readbyte (&unit->strength, input))
	return 0;
    if (! readbyte (&unit->agility, input))
	return 0;
    if (! readbyte (&unit->endurance, input))
	return 0;
    if (! readbyte (&unit->action, input))
	return 0;
    if (! readbyte (&unit->health, input))
	return 0;

    /* read the skills */
    if (! readbyte (&skillcount, input))
	return 0;
    for (c = 0; c < skillcount; ++c) {
	if (! (unit->skills[c] = new_Skill ()))
	    fatal_error (FATAL_MEMORY);
	if (unit->skills[c] &&
	    ! unit->skills[c]->read (unit->skills[c], input))
	    return 0;
    }

    /* read the inventory */
    for (c = 0; c < 10; ++c)
	if (! readbyte (&unit->inventory[c], input))
	    return 0;

    /* read the unit location */
    if (! readbyte (&unit->x, input))
	return 0;
    if (! readbyte (&unit->y, input))
	return 0;
    if (! readbyte (&unit->prone, input))
	return 0;

    /* success! */
    return 1;
}

/**
 * Generate a unit at random.
 * @param unit is the unit to generate.
 */
static int generate (Unit *unit)
{
    if (unit->class == UNIT_NONE)
	unit->class = 1 + rand () % 4;
    generatename (unit->name);
    generatestats (unit);
    generateskills (unit);
    if (! unit->hostage)
	generateitems (unit);
    unit->reset (unit);
    return 1;
}

/**
 * Deploy the unit on the map.
 * @param unit The unit object.
 * @param game The game object.
 * @param x    The new X coordinate.
 * @param y    The new Y coordinate.
 * @return     1 if move successful, 0 if not.
 */
static int deploy (Unit *unit, Game *game, int x, int y)
{
    /* validate the move */
    if (unit->action < 2 + unit->encumbrance + unit->creeping) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    } else if (game->map->blocks[x + game->map->width * y] & 0x80) {
	error = ERROR_BLOCKED;
	return 0;
    } else if (unit->x || unit->y) {
	error = ERROR_DEPLOYED;
	return 0;
    }

    /* deploy the unit */
    unit->action -= 2 + unit->encumbrance + unit->creeping;
    unit->x = x;
    unit->y = y;
    game->map->blocks[x + game->map->width * y] |= 0x80;
    unit->prone = game->unitcache[x + game->map->width * y];
    game->unitcache[x + game->map->width * y]
	= (unit->class != UNIT_ENEMY && ! unit->hostage)
	? MISSION_CACHE_UNIT + unit->id
	: (unit->class == UNIT_ENEMY)
	? MISSION_CACHE_ENEMY + unit->id
	: MISSION_CACHE_HOSTAGE | unit->id;
    return 1;
}

/**
 * Remove the unit from the map.
 * @param unit The unit object.
 * @param game The game.
 * @return     1 if move successful, 0 if not.
 */
static int undeploy (Unit *unit, Game *game)
{
    /* validate the move */
    if (unit->action < 2 + unit->encumbrance + unit->creeping) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    } else if (unit->x + game->map->width * unit->y
	       != game->entrance)
    {
	error = ERROR_NOTEXIT;
	return 0;
    } else if (! unit->x && ! unit->y) {
	error = ERROR_NOTDEPLOYED;
	return 0;
    }

    /* undeploy the unit */
    unit->action -= 2 + unit->encumbrance + unit->creeping;
    unit->x = 0;
    unit->y = 0;
    game->map->blocks[game->entrance] &= 0x7f;
    game->unitcache[game->entrance] = unit->prone;
    unit->prone = 0;
    return 1;
}

/**
 * Move the unit on the map.
 * @param unit The unit object.
 * @param game The game.
 * @param x    The new X coordinate.
 * @param y    The new Y coordinate.
 * @return     1 if move successful, 0 if not.
 */
static int move (Unit *unit, Game *game, int x, int y)
{
    int cost, /* movement cost */
	block; /* block at the location */

    /* validate the move */
    if (unit->x == 0 && unit->y == 0) {
	error = ERROR_NOTDEPLOYED;
	return 0;
    } else if (abs (unit->x - x) > 1 ||
	abs (unit->y - y) > 1)
    {
	error = ERROR_FAR;
	return 0;
    }
    block = game->map->blocks[x + game->map->width * y];
    if (block & 0x80) {
	error = ERROR_BLOCKED;
	return 0;
    } else if (! (block & 0x10)) {
	error = ERROR_BLOCKED;
	return 0;
    } else if (block == (MISSION_FURNITURE | FURNITURE_TABLE) ||
	       block == (MISSION_FURNITURE | FURNITURE_CABINET))
    {
	error = ERROR_BLOCKED;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    }
    cost = 2
	+ (unit->x != x && unit->y != y)
	+ ((block & 0x30) == LEVELMAP_DOOR)
	+ (block == (MISSION_FURNITURE | FURNITURE_CHAIR))
	+ (game->unitcache[x + game->map->width * y] != 0)
	+ unit->encumbrance
	+ unit->creeping;
    if (cost > unit->action) {
	error = ERROR_ACTIONS;
	return 0;
    }

    /* move the unit */
    unit->action -= cost;
    game->map->blocks[unit->x + game->map->width * unit->y]
	&= 0x7f;
    game->unitcache[unit->x + game->map->width * unit->y]
	= unit->prone;
    unit->x = x;
    unit->y = y;
    game->map->blocks[x + game->map->width * y] |= 0x80;
    unit->prone = game->unitcache[x + game->map->width * y];
    game->unitcache[x + game->map->width * y]
	= (unit->class != UNIT_ENEMY && ! unit->hostage)
	? MISSION_CACHE_UNIT + unit->id
	: (unit->class == UNIT_ENEMY)
	? MISSION_CACHE_ENEMY + unit->id
	: MISSION_CACHE_HOSTAGE | unit->id;

    /* return */
    return 1;
}

/**
 * Attack another unit.
 * @param unit   The unit that is attacking.
 * @param target The target of the attack.
 * @param game   The game map.
 * @return       1 if attack made, 0 if not.
 */
static int attack (Unit *unit, Unit *target, Game *game)
{
    /* validate attack */
    if (unit->x == 0 && unit->y == 0) {
	error = ERROR_NOTDEPLOYED;
	return 0;
    } else if (! game->lineofsight
	(game,
	 unit->x + game->map->width * unit->y,
	 target->x + game->map->width * target->y))
    {
	error = ERROR_BLOCKED;
	return 0;
    } else if (unit->action < 6 + unit->encumbrance + unit->creeping) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    }

    /* determine weapon and range of combat */
    unit->combattype = combattype
	(unit,
	 distance (unit->x, unit->y, target->x, target->y));

    /* resolve combat */
    if (unit->combattype != SKILL_NONE) {
	combat (unit, target, game);
	return 1;
    } else
	return 0;
}

/**
 * Reset a unit ready for the next turn.
 * @param unit The unit object.
 */
static void reset (Unit *unit)
{
    unit->action = 14 + unit->agility;
}

/**
 * Return the unit's level in a particular skill.
 * @param unit    The unit object.
 * @param skillid The skill to check.
 * @return        The level of skill.
 */
static int getskill (Unit *unit, int skillid)
{
    int c; /* counter */
    for (c = 0; c < 4; ++c)
	if (unit->skills[c] &&
	    unit->skills[c]->id == skillid)
	    return unit->skills[c]->level;
    return 0;
}

/**
 * Transfer an item between the unit and another unit.
 * @param unit     The unit.
 * @param target   The target.
 * @param fromslot The slot from which the item is taken.
 * @param toslot   Pointer to destination slot.
 * @return         1 if successful, 0 on failure.
 */
static int transferitemunit
(Unit *unit, Unit *target, int fromslot, int *toslot)
{
    /* check for action points */
    if (unit->action < 1) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    }

    /* transfer the item */
    if (fromslot < 10)
	return transfertounit (unit, target, fromslot, toslot);
    else
	return transferfromunit
	    (unit, target, fromslot, toslot);
}

/**
 * Transfer an item between the unit and a cabinet.
 * @param unit The unit.
 * @param cabinet  The cabinet.
 * @param fromslot The slot from which the item is taken.
 * @param toslot   Pointer to destination slot.
 * @return         1 if successful, 0 on failure.
 */
static int transferitemcabinet
(Unit *unit, Cabinet *cabinet, int fromslot, int *toslot)
{
    /* check for action points */
    if (unit->action < 1) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    }

    /* transfer the item */
    if (fromslot < 10)
	return transfertocabinet (unit, cabinet, fromslot, toslot);
    else
	return transferfromcabinet
	    (unit, cabinet, fromslot, toslot);
}

/**
 * Unequip an item from the unit's body.
 * @param unit   The unit object.
 * @param origin The origin slot.
 * @param dest   Pointer to the destination slot number.
 * @return       1 if successful, 0 if not.
 */
static int unequip (Unit *unit, int origin, int *dest)
{
    int empty; /* empty slot */

    /* validate */
    if (unit->class == UNIT_ENEMY) {
	error = ERROR_ENEMY;
	return 0;
    } else if (unit->class == UNIT_HOSTAGE && unit->hostage) {
	error = ERROR_HOSTAGE;
	return 0;
    } else if (origin > 3) {
	error = ERROR_UNEQUIPPED;
	return 0;
    } else if (! unit->inventory[origin]) {
	error = ERROR_NOITEM;
	return 0;
    } else if (unit->action < 1) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    }

    /* check to see if there's room for the item */
    empty = freeslot (unit, unit->inventory[origin], 4);
    if (empty == 10) {
	error = ERROR_NOROOM;
	return 0;
    }

    /* unequip the item */
    --unit->action;
    unit->inventory[empty] = unit->inventory[origin];
    unit->inventory[origin] = ITEM_NONE;
    *dest = empty;
    return 1;
}

/**
 * Unequip an item from the unit's body.
 * @param unit   The unit object.
 * @param origin The origin slot.
 * @param dest   Pointer to the destination slot number.
 * @return       1 if successful, 0 if not.
 */
static int equip (Unit *unit, int origin, int *dest)
{
    int empty, /* empty slot */
	item; /* item to equip */

    /* validate */
    if (unit->class == UNIT_ENEMY) {
	error = ERROR_ENEMY;
	return 0;
    } else if (unit->class == UNIT_HOSTAGE && unit->hostage) {
	error = ERROR_HOSTAGE;
	return 0;
    } else if (origin > 9) {
	error = ERROR_BACKPACK;
	return 0;
    } else if (! unit->inventory[origin]) {
	error = ERROR_NOITEM;
	return 0;
    } else if (origin < 4) {
	error = ERROR_EQUIPPED;
	return 0;
    } else if (unit->action < 1) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    }

    /* check to see if there's room for the item */
    item = unit->inventory[origin];
    empty = freeslot (unit, item, 0);
    if (empty >= 4) {
	error = ERROR_NOROOM;
	return 0;
    }

    /* unequip the item */
    --unit->action;
    unit->inventory[empty] = unit->inventory[origin];
    unit->inventory[origin] = ITEM_NONE;
    *dest = empty;
    return 1;
}

/**
 * Drop an item on the ground.
 * @param unit The unit object.
 * @param slot The inventory slot.
 * @param game The game object.
 * @return     1 if successful, 0 if not.
 */
static int drop (Unit *unit, int slot, Game *game)
{
    int loc, /* location as an integer */
	block; /* block at location */
    ItemStack *stack; /* item stack to drop the item on */

    /* validate */
    if (unit->class == UNIT_ENEMY) {
	error = ERROR_ENEMY;
	return 0;
    } else if (unit->class == UNIT_HOSTAGE && unit->hostage) {
	error = ERROR_HOSTAGE;
	return 0;
    } else if (slot > 9) {
	error = ERROR_BACKPACK;
	return 0;
    } else if (! unit->inventory[slot]) {
	error = ERROR_NOITEM;
	return 0;
    } else if (unit->x == 0 && unit->y == 0) {
	error = ERROR_NOTDEPLOYED;
	return 0;
    } else if (unit->action < 1) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    }

    /* check to see if there's room for the item */
    loc = unit->x + game->map->width * unit->y;
    block = game->map->blocks[loc];
    if ((block & 0x70) != LEVELMAP_OPEN) {
	error = ERROR_NOROOM;
	return 0;
    }
    if (! (stack = game->getstack (game, unit->x, unit->y, 1))) {
	error = ERROR_NOROOM;
	return 0;
    }
    
    /* unequip the item */
    --unit->action;
    stack->drop (stack, unit->inventory[slot]);
    unit->inventory[slot] = ITEM_NONE;
    calcencumbrance (unit);
    return 1;
}

/**
 * Pick up a unit from the ground.
 * @param unit The unit object.
 * @param slot Pointer to the inventory slot.
 * @param game The game object.
 * @return     1 if successful, 0 if not.
 */
static int pickup (Unit *unit, int *slot, Game *game)
{
    int item, /* item picked up */
	empty; /* free inventory slot */
    ItemStack *stack; /* item stack to drop the item on */

    /* validate */
    if (unit->class == UNIT_ENEMY) {
	error = ERROR_ENEMY;
	return 0;
    } else if (unit->class == UNIT_HOSTAGE && unit->hostage) {
	error = ERROR_HOSTAGE;
	return 0;
    } else if (unit->x == 0 && unit->y == 0) {
	error = ERROR_NOTDEPLOYED;
	return 0;
    } else if (unit->action < 1) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    }

    /* see what the item is */
    if (! (stack = game->getstack (game, unit->x, unit->y, 0))) {
	error = ERROR_NOITEM;
	return 0;
    } else if (! (item = stack->top (stack))) {
	error = ERROR_NOITEM;
	return 0;
    }

    /* check to see if there's room for the item */
    empty = freeslot (unit, item, 0);
    if (empty == 10) {
	error = ERROR_NOROOM;
	return 0;
    }
    
    /* unequip the item */
    --unit->action;
    unit->inventory[empty] = item;
    stack->take (stack);
    if (stack->itemcount == 0)
	game->removestack (game, stack);
    *slot = empty;
    calcencumbrance (unit);
    return 1;
}
    
/**
 * Free a hostage.
 * @param unit The unit object.
 * @param target The target unit.
 */
static int freehostage (Unit *unit, Unit *target)
{
    Game *game; /* pointer to the game object */
    int loc; /* hostage location */

    /* validate */
    if (unit->class == UNIT_ENEMY) {
	error = ERROR_ENEMY;
	return 0;
    } else if (unit->class == UNIT_HOSTAGE && unit->hostage) {
	error = ERROR_HOSTAGE;
	return 0;
    } else if (unit->action < 6 + unit->encumbrance + unit->creeping) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (abs (unit->x - target->x) > 1 ||
	       abs (unit->y - target->y) > 1) {
	error = ERROR_FAR;
	return 0;
    } else if (unit->x == 0 && unit->y == 0) {
	error = ERROR_NOTDEPLOYED;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    } else if (! target->hostage) {
	error = ERROR_FREE;
	return 0;
    } else if (unit->inventory[1] != ITEM_BLADE &&
	       unit->inventory[3] != ITEM_BLADE) {
	error = ERROR_BLADE;
	return 0;
    }

    /* remove the hostage line of sight detail */
    unit->action -= 6 + unit->encumbrance + unit->creeping;
    game = getgame ();
    game->units = realloc
	(game->units,
	 (game->unitcount + 1) * sizeof (Unit *));
    game->units[game->unitcount] = target;
    game->hostages[target->id] = NULL;
    target->id = game->unitcount;
    loc = target->x + game->map->width * target->y;
    game->unitcache[loc] = MISSION_CACHE_UNIT + target->id;
    target->hostage = 0;
    ++game->unitcount;
    return 1;
}

/**
 * Heal a unit.
 * @param unit The unit object.
 * @param target The target unit.
 */
static int heal (Unit *unit, Unit *target)
{
    int healroll, /* healing dice roll */
	maxhealth, /* maximum health of target */
	healdamage, /* amount of damage healed */
	healcost, /* cost of healing */
	conscious; /* amount of health required for consciousness */
    char *template; /* template to use for feedback */

    /* validate */
    maxhealth = 7 + target->strength + target->endurance;
    if (unit->class == UNIT_ENEMY) {
	error = ERROR_ENEMY;
	return 0;
    } else if (unit->class == UNIT_HOSTAGE && unit->hostage) {
	error = ERROR_HOSTAGE;
	return 0;
    } else if (unit->x == 0 && unit->y == 0) {
	error = ERROR_NOTDEPLOYED;
	return 0;
    } else if (unit->action < 6 + unit->encumbrance + unit->creeping) {
	error = ERROR_ACTIONS;
	return 0;
    } else if (abs (unit->x - target->x) > 1 ||
	       abs (unit->y - target->y) > 1)
    {
	error = ERROR_FAR;
	return 0;
    } else if (unit->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (unit->unconscious) {
	error = ERROR_UNCONSCIOUS;
	return 0;
    } else if (target->class == UNIT_ENEMY) {
	error = ERROR_HEALENEMY;
	return 0;
    } else if (target->health == 0) {
	error = ERROR_DEAD;
	return 0;
    } else if (target->health == maxhealth)
    {
	error = ERROR_HEALTHY;
	return 0;
    } else if (unit->inventory[1] != ITEM_MEDIKIT &&
	       unit->inventory[3] != ITEM_MEDIKIT) {
	error = ERROR_MEDIKIT;
	return 0;
    }

    /* do a skill roll, determine heal damage and cost */
    healroll = skillroll (unit->getskill (unit, SKILL_MEDICINE));
    if (healroll == 0)
	healdamage = 0;
    else
	healdamage = throwdice (1, 0, 0, 0);
    if (target->health + healdamage > maxhealth)
	healdamage = maxhealth - target->health;
    if (healroll == 2)
	healcost = 0;
    else
	healcost = 6 + unit->encumbrance + unit->creeping;

    /* apply healing */
    unit->action -= healcost;
    target->health += healdamage;
    conscious = (7 + target->strength + target->endurance) / 2;

    /* determine feedback template */
    if (! healdamage) {
	unit->feedbackid = UNIT_FEEDBACK_NOHEALED;
	template = "%s failed to heal %s";
    } else if (target->unconscious &&
	       (target->health
		== 7 + target->strength + target->endurance ||
		(target->health >= conscious &&
		 throwdice (2, 0, 0, 0) < target->endurance)))
    {
	unit->feedbackid = UNIT_FEEDBACK_CONSCIOUS;
	template = "%s restored %s to consciousness";
	target->unconscious = 0;
    } else if (target->health
	       == 7 + target->strength + target->endurance)
    {
	unit->feedbackid = UNIT_FEEDBACK_RESTORED;
	template = "%s restored %s to full health";
    } else {
	unit->feedbackid = UNIT_FEEDBACK_HEALED;
	sprintf (unit->feedback,
		 "%s healed %s (%d%%)",
		 unit->name,
		 target == unit ? "themself" : target->name,
		 100 * target->health / maxhealth);
	return 1;
    }

    /* create feedback and reterun */
    sprintf
	(unit->feedback,
	 template,
	 unit->name,
	 target == unit ? "themself" : target->name);
    return 1;
}

/*----------------------------------------------------------------------
 * Top Level Functions
 */

/**
 * Get the name of a particular class.
 * @returns A pointer to the class name.
 */
char *getclassname (int class)
{
    return classnames[class];
}

/**
 * Get the text of an error message.
 * @return       A pointer to the buffer.
 */
char *getuniterror (void)
{
    return errortext[error];
}

/*----------------------------------------------------------------------
 * Top Level Functions: Constructors.
 */

/**
 * Construct a unit.
 * @returns a pointer to the new unit.
 */
Unit *new_Unit (void)
{
    Unit *unit; /* the new unit */
    int c; /* counter */

    /* attempt to allocate memory */
    if (! (unit = malloc (sizeof (Unit))))
	return NULL;

    /* initialise the attributes */
    unit->id = 0;
    *unit->name = '\0';
    unit->class = UNIT_NONE;
    unit->level = UNIT_HARD;
    unit->hostage = 0;
    unit->unconscious = 0;
    unit->encumbrance = 0;
    unit->surprise = 0;
    unit->creeping = 0;
    unit->strength = 0;
    unit->agility = 0;
    unit->endurance = 0;
    unit->action = 0;
    unit->health = 0;
    unit->x = 0;
    unit->y = 0;
    for (c = 0; c < 4; ++c)
	unit->skills[c] = NULL;
    for (c = 0; c < 10; ++c)
	unit->inventory[c] = 0;
    unit->feedbackid = UNIT_FEEDBACK_NONE;
    *unit->feedback = '\0';
    unit->prone = 0;

    /* initialise the methods */
    unit->destroy = destroy;
    unit->write = write;
    unit->read = read;
    unit->generate = generate;
    unit->deploy = deploy;
    unit->undeploy = undeploy;
    unit->move = move;
    unit->attack = attack;
    unit->reset = reset;
    unit->getskill = getskill;
    unit->transferitemcabinet = transferitemcabinet;
    unit->transferitemunit = transferitemunit;
    unit->unequip = unequip;
    unit->equip = equip;
    unit->drop = drop;
    unit->pickup = pickup;
    unit->freehostage = freehostage;
    unit->heal = heal;

    /* return the new unit */
    return unit;
}
