/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 30-May-2024.
 *
 * Scores UI Module.
 */

/*
 * Required Headers
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* project headers */
#include "sccc.h"
#include "display.h"
#include "controls.h"
#include "game.h"
#include "scores.h"
#include "ui.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var ui A pointer to the UI structure. */
static UI *ui;

/** @var display The display object. */
static Display *display;

/** @var controls The controls object. */
static Controls *controls;

/** @var game The game object. */
static Game *game;

/** @var scoresmenu The scores screen menu. */
static char *scoresmenu[] = {
    "Cancel",
    "New game",
    "Quit game"
};

/** @var scoreskeys Shortcut keys for scores screen. */
static char *scoreskeys = " NQ";

/** @enum ScoresMenuID Identifiers for scores menu options. */
typedef enum {
    SCORESMENU_CANCEL,
    SCORESMENU_NEWGAME,
    SCORESMENU_QUITGAME,
    SCORESMENU_COUNT
} ScoresMenuID;

/** @var scoretext Text description for the score table. */
static char *scoretext =
    "The object of the game is to achieve the highest average score"
    " you can, starting at Easy level and progressing through Fair to"
    " Hard levels. Uncompleted missions are counted towards the"
    " average, so you need to attempt all six mission types at each"
    " level for the best score.";

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Initialise the UI.
 * @return   1 on success, 0 on failure.
 */
static int init (void)
{
    Scores *scores; /* pointer to scores */

    /* initialise module-wide variables */
    display = getdisplay ();
    controls = getcontrols ();
    game = getgame ();
    scores = getscores ();

    /* show the initial display */
    display->showbackground ();
    display->showscoretable (scores, game);
    display->showtitle ("Performance");
    display->typeset (scoretext);
    display->update ();

    /* success */
    return 1;
}

/**
 * Operate the UI.
 * @return   1 on success, 0 on failure.
 */
static int operate (void)
{
    int option; /* menu option chosen */
    do {

	/* wait for input */
	do {
	    controls->poll ();
	    if (controls->fire)
		option = display->menu
		    (SCORESMENU_COUNT,
		     scoresmenu,
		     1);
	    else if (strchr (scoreskeys, toupper (controls->ascii)))
		option
		    = strchr (scoreskeys, toupper (controls->ascii))
		    - scoreskeys;
	    else
		option = 0;
	} while (! option);

	/* act upon input */
	switch (option) {
	case SCORESMENU_NEWGAME:
	    return (game->state = STATE_NEWGAME);
	    break;
	case SCORESMENU_QUITGAME:
	    if (display->confirm ("Quit game?"))
		return 0;
	    break;
	}
    } while (1);
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Construct Scores Screen UI.
 * @return The new UI.
 */
UI *new_ScoresUI (void)
{
    /* reserve memory for the UI */
    if (! (ui = new_UI ()))
	return NULL;

    /* initialise methods */
    ui->init = init;
    ui->operate = operate;

    /* return the UI */
    return ui;
}
