/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 09-May-2024.
 *
 * Map UI Module.
 */

/*----------------------------------------------------------------------
 * Included Headers.
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* project headers */
#include "sccc.h"
#include "config.h"
#include "display.h"
#include "controls.h"
#include "game.h"
#include "unit.h"
#include "item.h"
#include "cabinet.h"
#include "ui.h"
#include "utils.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var ui A pointer to the UI structure. */
static UI *ui;

/** @var display The display object. */
static Display *display;

/** @var controls The controls object. */
static Controls *controls;

/** @var game The game object. */
static Game *game;

/** @var cursorx The X coordinate of the cursor. */
static int cursorx;

/** @var cursory The Y coordinate of the cursor. */
static int cursory;

/** @var cursorinv The inventory cursor (item #). */
static int cursorinv;

/** @var viewx The X coordinate of the view. */
static int viewx;

/** @var viewy The Y coordinate of the view. */
static int viewy;

/** @var mapmenu The main map screen menu. */
static char *mapmenu[] = {
    "Cancel",
    "next Unit",
    "Move",
    "Select",
    "Attack",
    "cReep",
    "Heal",
    "Free",
    "Pick up",
    "Inventory",
    "End turn",
    "New game",
    "Quit game"
};

/** @var mapkeys Shortcut keys for map screen. */
static char *mapkeys = " UMSARHFPIENQ";

/** @enum MapMenuID Identifiers for menu options. */
typedef enum {
    MAPMENU_CANCEL,
    MAPMENU_NEXTUNIT,
    MAPMENU_MOVE,
    MAPMENU_SELECT,
    MAPMENU_ATTACK,
    MAPMENU_CREEP,
    MAPMENU_HEAL,
    MAPMENU_FREE,
    MAPMENU_PICKUP,
    MAPMENU_INVENTORY,
    MAPMENU_ENDTURN,
    MAPMENU_NEWGAME,
    MAPMENU_QUITGAME,
    MAPMENU_COUNT
} MapMenuID;

/** @var invmenu The map screen inventory menu. */
static char *invmenu[] = {
    "Cancel",
    "Equip",
    "Unequip",
    "Transfer",
    "Close"
};

/** @enum InvMenuID Identifiers for inventory menu options. */
typedef enum {
    INVMENU_CANCEL,
    INVMENU_EQUIP,
    INVMENU_UNEQUIP,
    INVMENU_TRANSFER,
    INVMENU_CLOSE,
    INVMENU_COUNT
} InvMenuID;

/** @var invkeys Shortcut keys for the inventory menu. */
static char invkeys[] = " EUTC";

/**
 * @var invmatrices
 * A table controlling movement around the inventories.
 */
static char invmatrices[3][4][22] = {
    /* single inventory */
    {
	/* cursor left */
	{
	    1,
	    1, 1, 2,
	    4, 4, 5,
	    7, 7, 8,
	    10, 11, 12,
	    13, 14, 15,
	    16, 17, 18,
	    19, 20, 21
	},
	/* cursor right */
	{
	    3,
	    0, 3, 3,
	    5, 6, 6,
	    8, 9, 9,
	    10, 11, 12,
	    13, 14, 15,
	    16, 17, 18,
	    19, 20, 21
	},
	/* cursor up */
	{
	    0,
	    0, 0, 0,
	    1, 2, 3,
	    4, 5, 6,
	    10, 11, 12,
	    13, 14, 15,
	    16, 17, 18,
	    19, 20, 21
	},
	/* cursor down */
	{
	    2,
	    4, 5, 6,
	    7, 8, 9,
	    7, 8, 9,
	    10, 11, 12,
	    13, 14, 15,
	    16, 17, 18,
	    19, 20, 21
	}
    },
    /* dual inventory */
    {
	/* cursor left */
	{
	    1,
	    1, 1, 2,
	    4, 4, 5,
	    7, 7, 8,
	    11,
	    3, 11, 12,
	    6, 14, 15,
	    9, 17, 18,
	    20, 21
	},
	/* cursor right */
	{
	    3,
	    0, 3, 11,
	    5, 6, 14,
	    8, 9, 17,
	    13,
	    10, 13, 13,
	    15, 16, 16,
	    18, 19, 19,
	    20, 21
	},
	/* cursor up */
	{
	    0,
	    0, 0, 0,
	    1, 2, 3,
	    4, 5, 6,
	    10,
	    10,10,10,
	    11,12,13,
	    14,15,16,
	    20, 21
	},
	/* cursor down */
	{
	    2,
	    4, 5, 6,
	    7, 8, 9,
	    7, 8, 9,
	    12,
	    14,15,16,
	    17,18,19,
	    17,18,19,
	    20, 21
	}
    },
    /* cabinet inventory */
    {
	/* cursor left */
	{
	    1,
	    1, 1, 2,
	    4, 4, 5,
	    7, 7, 8,
	    3, 10, 11,
	    3, 13, 14,
	    6, 16, 17,
	    9, 19, 20
	    
	},
	/* cursor right */
	{
	    3,
	    0, 3, 10,
	    5, 6, 16,
	    8, 9, 19,
	    11, 12, 12,
	    14, 15, 15,
	    17, 18, 18,
	    20, 21, 21
	},
	/* cursor up */
	{
	    0,
	    0, 0, 0,
	    1, 2, 3,
	    4, 5, 6,
	    10, 11, 12,
	    10, 11, 12,
	    13, 14, 15,
	    16, 17, 18
	},
	/* cursor down */
	{
	    2,
	    4, 5, 6,
	    7, 8, 9,
	    7, 8, 9,
	    13, 14, 15,
	    16, 17, 18,
	    19, 20, 21,
	    19, 20, 21
	}
    }
};

/*----------------------------------------------------------------------
 * Level 3 Function Definitions.
 */

/**
 * Update the map according to changed enemy visibility.
 * @param changes Changed bits in the visibility
 */
static void updatevisibility (int changes)
{
    int c, /* counter */
	seen = 0; /* total number of units seen */
    Unit *unit = NULL; /* unit seen */

    /* update the visibility first */
    game->visibility |= changes;

    /* check for newly discovered enemies */
    for (c = 0; c < game->enemycount; ++c)
	if (changes & (1 << c)) {
	    display->updatemapsquare
		(game,
		 game->enemies[c]->x,
		 game->enemies[c]->y);
	    display->showmapsquare
		(game->enemies[c]->x,
		 game->enemies[c]->y,
		 viewx, viewy);
	    ++seen;
	    unit = game->enemies[c];
	}

    /* check for newly discovered hostages */
    for (c = 0; c < game->hostagecount; ++c)
	if (changes & (1 << (game->enemycount + c))) {
	    display->updatemapsquare
		(game,
		 game->hostages[c]->x,
		 game->hostages[c]->y);
	    display->showmapsquare
		(game->hostages[c]->x,
		 game->hostages[c]->y,
		 viewx, viewy);
	    ++seen;
	    unit = game->hostages[c];
	}

    /* add a message */
    if (unit && seen == 1)
	display->message ("%s spotted", unit->name);
    else
	display->message ("%d units spotted", seen);
}

/**
 * Process the opportunity attacks.
 */
static void opportunityattacks (void)
{
    Unit *attacker; /* pointer to attacking unit */
    int unconscious, /* unconscious before attack */
	health; /* health before attack */
    unconscious = game->unit->unconscious;
    health = game->unit->health;
    while ((attacker = game->opportunity (game))) {
	display->startdelay (500);
	display->updatemapsquare
	    (game,
	     game->unit->x,
	     game->unit->y);
	display->showmapsquare (cursorx, cursory, viewx, viewy);
	display->showunithealth (game->unit);
	display->message ("%s", attacker->feedback);
	display->battleanimation
	    (attacker, game->unit, game, viewx, viewy);
	display->update ();
	if (health && ! game->unit->health)
	    display->playsound (DISPLAY_DEATH);
	else if (! unconscious && game->unit->unconscious)
	    display->playsound (DISPLAY_UNCONSCIOUS);
	display->enddelay ();
	unconscious = game->unit->unconscious;
	health = game->unit->health;
    }
}

/**
 * Move the inventory cursor according to last control.
 * @param invtype The inventory type.
 * @param item    The item under the cursor.
 */
static void moveinventorycursor (int invtype, int item)
{
    int newcursor; /* new cursor position */

    /* initialise, assume no movement */
    newcursor = cursorinv;

    /* check controls for new cursor position */
    if (controls->left)
	newcursor = invmatrices[invtype][0][cursorinv];
    else if (controls->right)
	newcursor = invmatrices[invtype][1][cursorinv];
    else if (controls->up)
	newcursor = invmatrices[invtype][2][cursorinv];
    else if (controls->down)
	newcursor = invmatrices[invtype][3][cursorinv];

    /* if cursor has moved, redraw it */
    if (newcursor != cursorinv) {
	display->hideinvcursor (cursorinv, invtype, item);
	cursorinv = newcursor;
    }
}

/*----------------------------------------------------------------------
 * Level 2 Function Definitions.
 */

/**
 * Centre the view on a set of coordinates.
 * @param x The x coordinate.
 * @param y The y coordinate.
 * @return  1 if the view moved, 0 if not.
 */
static int centreview (int x, int y)
{
    int oldx, /* previous view X coordinate */
	oldy; /* previous view Y coordinate */

    /* record old view location */
    oldx = viewx;
    oldy = viewy;

    /* calculate extactly centred position */
    viewx = x - 4;
    viewy = y - 4;

    /* ensure we haven't gone off the map */
    if (viewx < 0)
	viewx = 0;
    else if (viewx > game->map->width - 9)
	viewx = game->map->width - 9;
    if (viewy < 0)
	viewy = 0;
    else if (viewy > game->map->height - 9)
	viewy = game->map->height - 9;

    /* tell the calling process if the view has moved */
    return (viewx != oldx || viewy != oldy);
}

/**
 * Ensure a coordinate position is within the view.
 * @param x The x coordinate.
 * @param y The y coordinate.
 * @return  1 if the view moved, 0 if not.
 */
static int checkview (int x, int y)
{
    int oldx, /* previous view X coordinate */
	oldy; /* previous view Y coordinate */

    /* record old view location */
    oldx = viewx;
    oldy = viewy;

    /* check if we're off the map */
    if (x < viewx)
	viewx = x;
    else if (x > viewx + 8)
	viewx = x - 8;
    if (y < viewy)
	viewy = y;
    else if (y > viewy + 8)
	viewy = y - 8;

    /* tell the calling process if the view has moved */
    return (viewx != oldx || viewy != oldy);
}

/**
 * Enter a unit onto the map.
 * @param unit The unit to enter.
 */
static void entermap (Unit *unit)
{
    int changedvisibility; /* ... of enemies and hostages */

    /* attempt to deploy */
    if (unit->deploy (unit, game, cursorx, cursory)) {

	/* update the screen */
	display->showunitaction (unit);
	display->updatemapsquare
	    (game, cursorx, cursory);
	display->showmapsquare
	    (cursorx, cursory, viewx, viewy);
	display->message ("%s entered the area", unit->name);
	display->showtutorial (TUTORIAL_MOVE);
	display->update ();
	display->playsound (DISPLAY_PLOD);

	/* check for new units visible */
	if ((changedvisibility
	     = game->changedvisibility
	     (game,
	      unit->x + game->map->width * unit->y,
	      game->visibility)))
	    updatevisibility (changedvisibility);

	/* opportunity attacks */
	opportunityattacks ();

    }

    /* show an error message */
    else
	display->error (getuniterror ());
}

/**
 * Remove a unit from the map.
 * @param unit The unit to remove.
 */
static void leavemap (Unit *unit)
{
    /* attempt to deploy */
    if (unit->undeploy (unit, game)) {
	display->showunitaction (unit);
	display->updatemapsquare
	    (game, cursorx, cursory);
	display->showmapsquare
	    (cursorx, cursory, viewx, viewy);
	display->message ("%s left the area", unit->name);
	display->update ();
	display->playsound (DISPLAY_PLOD);
    }

    /* show an error message */
    else
	display->error (getuniterror ());
}

/**
 * Move a unit within the map.
 * @param unit The unit to move.
 */
static void moveunitwithinmap (Unit *unit)
{
    int prevx, /* previous square X coordinate */
	prevy, /* previous square Y coordinate */
	changedvisibility; /* ... of enemies and hostages */

    /* initialise */
    prevx = unit->x;
    prevy = unit->y;

    /* attempt to move */
    if (unit->move (unit, game, cursorx, cursory)) {

	/* update the screen */
	display->showunitaction (unit);
	display->updatemapsquare
	    (game, prevx, prevy);
	display->updatemapsquare
	    (game, cursorx, cursory);
	display->showmapsquare
	    (prevx, prevy, viewx, viewy);
	display->showmapsquare
	    (unit->x, unit->y, viewx, viewy);
	display->update ();
	display->playsound (DISPLAY_PLOD);

	/* advance the cursor */
	if (cursorx + game->map->width * cursory
	    != game->entrance)
	{
	    cursorx += (unit->x - prevx);
	    cursory += (unit->y - prevy);
	    if (cursorx < 0)
		cursorx = 0;
	    else if (cursorx >= game->map->width)
		cursorx = game->map->width - 1;
	    if (cursory < 0)
		cursory = 0;
	    else if (cursory >= game->map->height)
		cursory = game->map->height - 1;
	    display->showtutorial (TUTORIAL_NEXTUNIT);
	} else
	    display->showtutorial (TUTORIAL_LEAVE);

	/* check for new units visible */
	if ((changedvisibility
	     = game->changedvisibility
	     (game,
	      unit->x + game->map->width * unit->y,
	      game->visibility)))
	    updatevisibility (changedvisibility);

	/* opportunity attacks */
	opportunityattacks ();

    }

    /* show an error message */
    else
	display->error (getuniterror ());
}

/**
 * Initialise the inventory view.
 * @param invtype The type of inventory.
 * @param target The target unit for dual inventory.
 * @param cabinet The cabinet for cabinet inventory.
 */
static void initinventory (int invtype, Unit *target, Cabinet *cabinet)
{
    /* set the menu option to drop/transfer */
    if (invtype == DISPLAY_INVENTORY_SINGLE) {
	invmenu[INVMENU_TRANSFER] = "Drop";
	invkeys[INVMENU_TRANSFER] = 'D';
    } else {
	invmenu[INVMENU_TRANSFER] = "Transfer";
	invkeys[INVMENU_TRANSFER] = 'T';
    }

    /* show the inventory and set the cursor position */
    display->showinventory (invtype, target, cabinet);
    cursorinv = 0;
}

/**
 * Move the inventory cursor
 * @param invtype The type of inventory.
 * @param target The target unit for dual inventory.
 * @param cabinet The cabinet for cabinet inventory.
 */
static void inventorycursor
(int invtype, Unit *target, Cabinet *cabinet)
{
    int item; /* item */

    /* loop until menu is requested */
    do {

	/* show the cursor and wait for input */
	display->showinvcursor (cursorinv, invtype);
	display->update ();
	controls->release (250);
	controls->wait (0);
	controls->poll ();

	/* move the cursor */
	if (controls->left ||
	    controls->right ||
	    controls->up ||
	    controls->down)
	{
	    if (cursorinv < 10)
		item = game->unit->inventory[cursorinv];
	    else if (invtype == DISPLAY_INVENTORY_DUAL)
		item = target->inventory[cursorinv - 10];
	    else if (invtype == DISPLAY_INVENTORY_CABINET)
		item = cabinet->items[cursorinv - 10];
	    moveinventorycursor (invtype, item);
	}

    } while (! controls->fire &&
	     ! strchr (invkeys, toupper (controls->ascii)));
}

/**
 * Work out the default inventory menu option.
 */
static int defaultinvoption
(int invtype, Unit *target, Cabinet *cabinet)
{
    if (cursorinv >= 10 &&
	     invtype == DISPLAY_INVENTORY_DUAL &&
	     target->inventory[cursorinv - 10])
	return INVMENU_TRANSFER;
    else if (cursorinv >= 10 &&
	     invtype == DISPLAY_INVENTORY_CABINET &&
	     cabinet->items[cursorinv - 10])
	return INVMENU_TRANSFER;
    else if (cursorinv < 10 &&
	     invtype != DISPLAY_INVENTORY_SINGLE &&
	     game->unit->inventory[cursorinv])
	return INVMENU_TRANSFER;
    else if (cursorinv < 4 && game->unit->inventory[cursorinv])
	return INVMENU_UNEQUIP;
    else if (cursorinv < 10 && game->unit->inventory[cursorinv])
	return INVMENU_EQUIP;
    else
	return INVMENU_CLOSE;
}

/**
 * Unequip an item.
 */
static void equip (void)
{
    int slot; /* slot into which item was unequipped */

    /* attempt to equip the item */
    if (game->unit->equip (game->unit, cursorinv, &slot)) {
	display->showunitaction (game->unit);
	display->showinvitem
	    (cursorinv,
	     DISPLAY_INVENTORY_SINGLE,
	     game->unit->inventory[cursorinv]);
	display->showinvitem
	    (slot,
	     DISPLAY_INVENTORY_SINGLE,
	     game->unit->inventory[slot]);
    }

    /* show an error message */
    else
	display->error (getuniterror ());
}

/**
 * Unequip an item.
 */
static void unequip (void)
{
    int slot; /* slot into which item was unequipped */

    /* attempt to unequip the item */
    if (game->unit->unequip (game->unit, cursorinv, &slot)) {
	display->showunitaction (game->unit);
	display->showinvitem
	    (cursorinv,
	     DISPLAY_INVENTORY_SINGLE,
	     game->unit->inventory[cursorinv]);
	display->showinvitem
	    (slot,
	     DISPLAY_INVENTORY_SINGLE,
	     game->unit->inventory[slot]);
    }

    /* show an error message */
    else
	display->error (getuniterror ());
}

/**
 * Drop an item on the floor.
 */
static void dropitem (void)
{
    /* attempt to drop the item on the ground */
    if (game->unit->drop (game->unit, cursorinv, game)) {
	display->showunitaction (game->unit);
	display->showinvitem
	    (cursorinv,
	     DISPLAY_INVENTORY_SINGLE,
	     game->unit->inventory[cursorinv]);
	display->updatemapsquare
	    (game,
	     game->unit->x,
	     game->unit->y);
	display->showmapsquare
	    (game->unit->x,
	     game->unit->y,
	     viewx,
	     viewy);
    }

    /* show an error message */
    else
	display->error (getuniterror ());
}

/**
 * Transfer an item between units.
 * @param target The target unit.
 */
static void transferitemunit (Unit *target)
{
    int slot; /* slot the item was put into */

    /* attempt to transfer the item */
    if (game->unit->transferitemunit
	(game->unit, target, cursorinv, &slot))
    {
	display->showunitaction (game->unit);
	if (cursorinv < 10) {
	    display->showinvitem
		(cursorinv,
		 DISPLAY_INVENTORY_DUAL,
		 game->unit->inventory[cursorinv]);
	    display->showinvitem
		(slot,
		 DISPLAY_INVENTORY_DUAL,
		 target->inventory[slot - 10]);
	} else {
	    display->showinvitem
		(slot,
		 DISPLAY_INVENTORY_DUAL,
		 game->unit->inventory[slot]);
	    display->showinvitem
		(cursorinv,
		 DISPLAY_INVENTORY_DUAL,
		 target->inventory[cursorinv - 10]);
	}
    }

    /* show error messages */
    else
	display->error (getuniterror ());
}

/**
 * Transfer an item between unit and cabinet.
 * @param cabinet The cabinet.
 */
static void transferitemcabinet (Cabinet *cabinet)
{
    int slot; /* slot the item was put into */

    /* attempt to transfer the item */
    if (game->unit->transferitemcabinet
	(game->unit, cabinet, cursorinv, &slot))
    {
	display->showunitaction (game->unit);
	if (cursorinv < 10) {
	    display->showinvitem
		(cursorinv,
		 DISPLAY_INVENTORY_CABINET,
		 game->unit->inventory[cursorinv]);
	    display->showinvitem
		(slot,
		 DISPLAY_INVENTORY_CABINET,
		 cabinet->items[slot - 10]);
	} else {
	    display->showinvitem
		(slot,
		 DISPLAY_INVENTORY_CABINET,
		 game->unit->inventory[slot]);
	    display->showinvitem
		(cursorinv,
		 DISPLAY_INVENTORY_CABINET,
		 cabinet->items[cursorinv - 10]);
	}
    }

    /* show error messages */
    else
	display->error (getuniterror ());
}

/**
 * Clean up after the inventory.
 */
static void cleaninventory (void)
{
    if (cursorinv < 10)
	display->hideinvcursor
	    (cursorinv,
	     DISPLAY_INVENTORY_SINGLE,
	     game->unit->inventory[cursorinv]);
    display->hideinventory ();
}

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Allow the user to move the cursor.
 */
static void movecursor (void)
{
    /* ensure that the cursor is on the screen */
    if (checkview (cursorx, cursory))
	display->showmap (viewx, viewy);

    /* loop until menu is requested */
    do {

	/* show the cursor and wait for input */
	display->showmapcursor (cursorx, cursory, viewx, viewy);
	display->update ();
	controls->release (250);
	controls->wait (0);
	controls->poll ();

	/* horizontal movement */
	if (controls->left && cursorx > 0)
	    display->hidemapcursor (cursorx--, cursory, viewx, viewy);
	else if (controls->right &&
		   cursorx < game->map->width - 1)
	    display->hidemapcursor (cursorx++, cursory, viewx, viewy);

	/* vertical movement */
	if (controls->up && cursory > 0)
	    display->hidemapcursor (cursorx, cursory--, viewx, viewy);
	else if (controls->down &&
		 cursory < game->map->height - 1)
	    display->hidemapcursor (cursorx, cursory++, viewx, viewy);

	/* move the view if necessary and update the display */
	if (checkview (cursorx, cursory))
	    display->showmap (viewx, viewy);
	display->update ();

    } while (! controls->fire &&
	     ! strchr(mapkeys, toupper (controls->ascii)));
}

/**
 * Ascertain the default menu option.
 * @return   The default menu option.
 */
static int defaultoption (void)
{
    int cursor, /* cursor location as an integer */
	adjacent, /* cursor is adjacent to the unit */
	self; /* cursor is pointed at current unit */
    Unit *unit = NULL; /* unit under the cursor */

    /* get cursor location and unit details */
    cursor = cursorx + game->map->width * cursory;
    if (game->unitcache[cursor] & MISSION_CACHE_UNIT)
	unit = game->units[game->unitcache[cursor] & 0xf];
    else if (game->unitcache[cursor] & MISSION_CACHE_ENEMY)
	unit = game->enemies
	    [game->unitcache[cursor] & 0xf];
    else if (game->unitcache[cursor] & MISSION_CACHE_HOSTAGE)
	unit = game->hostages
	    [game->unitcache[cursor] & 0xf];
    adjacent =
	game->unit &&
	abs (game->unit->x - cursorx) <= 1 &&
	abs (game->unit->y - cursory) <= 1 &&
	(game->unit->x != cursorx ||
	 game->unit->y != cursory);
    self =
	game->unit &&
	game->unit->x == cursorx &&
	game->unit->y == cursory;

    /* attack a unit */
    if (unit &&
	unit->class == UNIT_ENEMY &&
	unit->health &&
	(! adjacent || ! unit->unconscious) &&
	(game->visibility & 1 << unit->id))
	return MAPMENU_ATTACK;

    /* free a hostage */
    if (unit &&
	unit->class == UNIT_HOSTAGE &&
	unit->hostage &&
	adjacent)
	return MAPMENU_FREE;

    /* open a cabinet */
    if (game->map->blocks[cursor]
	== MISSION_FURNITURE + FURNITURE_CABINET &&
	adjacent)
	return MAPMENU_INVENTORY;

    /* heal a unit */
    if (game->unit &&
	unit &&
	(adjacent || self) &&
	unit->class != UNIT_ENEMY &&
	! unit->hostage &&
	unit->health > 0 &&
	unit->health < 7 + unit->strength + unit->endurance &&
	(game->unit->inventory[1] == ITEM_MEDIKIT ||
	 game->unit->inventory[3] == ITEM_MEDIKIT))
	return MAPMENU_HEAL;

    /* select a controllable unit */
    if (unit &&
	unit != game->unit &&
	unit->class != UNIT_ENEMY &&
	! unit->hostage &&
	! unit->unconscious &&
	unit->health > 0)
	return MAPMENU_SELECT;

    /* deploy a unit */
    if (game->unit &&
	game->unit->x == 0 &&
	game->unit->y == 0 &&
	cursor == game->entrance)
	return game->unit->creeping ? MAPMENU_CREEP : MAPMENU_MOVE;

    /* undeploy a unit */
    if (game->unit &&
	game->unit->x == cursorx &&
	game->unit->y == cursory &&
	cursor == game->entrance)
	return game->unit->creeping ? MAPMENU_CREEP : MAPMENU_MOVE;

    /* move a unit */
    if (game->unit &&
	game->unit->class != UNIT_ENEMY &&
	! game->unit->hostage &&
	adjacent)
	return game->unit->creeping ? MAPMENU_CREEP : MAPMENU_MOVE;

    /* open the inventory */
    if (game->unit &&
	game->unit == unit)
	return MAPMENU_NEXTUNIT;

    /* select a unit */
    if (unit &&
	unit->class == UNIT_ENEMY &&
	(game->visibility & (1 << unit->id)))
	return MAPMENU_SELECT;

    /* select a unit */
    if (unit &&
	unit->class == UNIT_HOSTAGE &&
	unit->hostage &&
	(game->visibility &
	 (1 << (game->enemycount + unit->id))))
	return MAPMENU_SELECT;

    /* no default option */
    return MAPMENU_CANCEL;
}

/**
 * Select the next unit and update the UI.
 */
static void nextunit (void)
{
    if (game->nextunit (game)) {
	if (game->unit->x && game->unit->y) {
	    cursorx = game->unit->x;
	    cursory = game->unit->y;
	} else {
	    cursorx
		= game->entrance
		% game->map->width;
	    cursory
		= game->entrance
		/ game->map->width;
	}
	centreview (cursorx, cursory);
	display->showunit (game->unit);
	display->showmap (viewx, viewy);
	display->update ();
    }
}

/**
 * Move the current unit.
 */
static void moveunit (void)
{
    Unit *unit; /* the unit to move */

    /* validate moving unit */
    if (! (unit = game->unit)) {
	display->error ("No unit selected");
	return;
    }
    if (unit->class == UNIT_ENEMY) {
	display->error ("Cannot move enemy units");
	return;
    }
    if (unit->hostage) {
	display->error ("Hostage must be freed first");
	return;
    }

    /* switch creeping off */
    unit->creeping = 0;

    /* enter the map */
    if (unit->x == 0 && unit->y == 0 &&
	cursorx == game->entrance % game->map->width &&
	cursory == game->entrance / game->map->width)
	entermap (unit);

    /* leave the map */
    else if (unit->x == cursorx && unit->y == cursory &&
	     cursorx == game->entrance % game->map->width &&
	     cursory == game->entrance / game->map->width)
	leavemap (unit);

    /* move within the map */
    else if (unit->x || unit->y)
	moveunitwithinmap (unit);
}

/**
 * Select the unit under the cursor.
 */
static void selectunit (void)
{
    Unit *unit = NULL; /* unit to select */
    int loc, /* location of cursor */
	cache; /* cache content at that location */

    /* determine the unit at the cursor location */
    loc = cursorx + game->map->width * cursory;
    cache = game->unitcache[loc];
    if (cache & MISSION_CACHE_UNIT)
	unit = game->units[cache & 0xf];
    else if (cache & MISSION_CACHE_ENEMY)
	unit = game->enemies[cache & 0xf];
    else if (cache & MISSION_CACHE_HOSTAGE)
	unit = game->hostages[cache & 0xf];
    else
	return;

    /* ensure the player can see the unit */
    if (unit->class == UNIT_ENEMY &&
	! ((1 << unit->id) & game->visibility))
	return;
    if (unit->class == UNIT_HOSTAGE &&
	unit->hostage &&
	! ((1 << (game->enemycount + unit->id) &
	    game->visibility)))
	return;
	
    /* set the unit */
    if (unit) {
	game->unit = unit;
	centreview (cursorx, cursory);
	display->showunit (game->unit);
	display->showmap (viewx, viewy);
	display->update ();
    }
}

/**
 * Attack the unit under the cursor.
 */
static void attackunit (void)
{
    Unit *attacker, /* the attacking unit */
	*defender = NULL; /* the defending unit */
    int cache, /* unit cache value */
	unconscious, /* target unconscious before attack */
	health; /* target health before attack */

    /* validate attacking unit */
    if (! (attacker = game->unit)) {
	display->error ("No unit selected");
	return;
    } else if (attacker->class == UNIT_ENEMY) {
	display->error
	    ("%s doesn't take orders from you",
	     attacker->name);
	return;
    } else if (attacker->hostage) {
	display->error
	    ("%s must be freed first",
	     attacker->name);
	return;
    }

    /* determine defending unit */
    cache = game->unitcache[cursorx + game->map->width * cursory];
    if (cache & MISSION_CACHE_ENEMY)
	defender = game->enemies[cache & 0xf];
    else {
	display->error ("No enemy unit here");
	return;
    }
    unconscious = defender->unconscious;
    health = defender->health;

    /* attack the unit */
    if (attacker->attack (attacker, defender, game)) {
	display->updatemapsquare
	    (game,
	     defender->x,
	     defender->y);
	display->showmapsquare (cursorx, cursory, viewx, viewy);
	display->showunitaction (attacker);
	display->message ("%s", attacker->feedback);
	display->battleanimation
	    (attacker, defender, game, viewx, viewy);
	display->update ();
	if (health && ! defender->health)
	    display->playsound (DISPLAY_DEATH);
	else if (! unconscious && defender->unconscious)
	    display->playsound (DISPLAY_UNCONSCIOUS);
    } else
	display->error (getuniterror ());

    /* allow opportunity fire */
    opportunityattacks ();
}

/**
 * Toggle a unit's creeping status.
 */
static void creepunit (void)
{
    Unit *unit; /* the unit to move */

    /* validate moving unit */
    if (! (unit = game->unit)) {
	display->error ("No unit selected");
	return;
    }
    if (unit->class == UNIT_ENEMY) {
	display->error ("Cannot move enemy units");
	return;
    }
    if (unit->hostage) {
	display->error ("Hostage must be freed first");
	return;
    }

    /* switch creeping on */
    unit->creeping = 1;

    /* enter the map */
    if (unit->x == 0 && unit->y == 0 &&
	cursorx == game->entrance % game->map->width &&
	cursory == game->entrance / game->map->width)
	entermap (unit);

    /* leave the map */
    else if (unit->x == cursorx && unit->y == cursory &&
	     cursorx == game->entrance % game->map->width &&
	     cursory == game->entrance / game->map->width)
	leavemap (unit);

    /* move within the map */
    else if (unit->x || unit->y)
	moveunitwithinmap (unit);
}

/**
 * Heal the unit under the cursor.
 */
static void healunit (void)
{
    Unit *target; /* target unit */
    int wasunconscious, /* 1 if unconscious before healing */
	changedvisibility; /* change in visibility */

    /* acertain target unit */
    if (! (target = game->getunit
	   (game, cursorx, cursory)))
    {
	display->error ("There is no unit here");
	return;
    }
    wasunconscious = target->unconscious;

    /* attempt to heal target unit */
    if (game->unit->heal (game->unit, target)) {

	/* display units */
	display->updatemapsquare
	    (game,
	     target->x,
	     target->y);
	display->showmapsquare
	    (target->x,
	     target->y,
	     viewx,
	     viewy);

	/* show unit info and confirmation */
	display->showunitaction (game->unit);
	if (game->unit == target)
	    display->showunithealth (game->unit);
	display->message ("%s", game->unit->feedback);

 	/* check for new units visible */
	if (wasunconscious &&
	    ! target->unconscious &&
	    (changedvisibility
	     = game->changedvisibility
	     (game,
	      target->x + game->map->width * target->y,
	      game->visibility)))
	    updatevisibility (changedvisibility);

	/* update the screen */
	display->update ();
	display->playsound (DISPLAY_PLOD);
    }

    /* display an error message */
    else
	display->error (getuniterror ());
}

/**
 * Free a hostage unit.
 */
static void freeunit (void)
{
    Unit *target; /* target unit */
    int changedvisibility; /* change in visibility */

    /* acertain target unit */
    if (! (target = game->getunit
	   (game, cursorx, cursory)))
    {
	display->error ("There is no unit here");
	return;
    }

    /* attempt to free target unit */
    if (game->unit->freehostage (game->unit, target)) {

	/* display confirmation */
	display->showunitaction (game->unit);
	display->message
	    ("%s freed %s",
	     game->unit->name,
	     target->name);

 	/* check for new units visible */
	if ((changedvisibility
	     = game->changedvisibility
	     (game,
	      target->x + game->map->width * target->y,
	      game->visibility)))
	    updatevisibility (changedvisibility);


	/* update the screen */
	display->update ();
	display->playsound (DISPLAY_PLOD);
   }

    /* display an error message */
    else
	display->error (getuniterror ());
}

/**
 * Pick up an item from the ground.
 */
static void pickup (void)
{
    int slot; /* slot ID where unit was placed */

    /* attempt to pick up an item */
    if (game->unit->pickup (game->unit, &slot, game)) {
	display->showunitaction (game->unit);
	display->showinvitem
	    (slot,
	     DISPLAY_INVENTORY_SINGLE,
	     game->unit->inventory[slot]);
	display->updatemapsquare
	    (game,
	     game->unit->x,
	     game->unit->y);
	display->showmapsquare
	    (game->unit->x,
	     game->unit->y,
	     viewx,
	     viewy);
	display->update ();
    }

    /* show an error message */
    else
	display->error (getuniterror ());
}

/**
 * Access the unit inventory.
 */
static void inventory (void)
{
    Unit *unit, /* main unit to access */
	*target = NULL; /* target unit */
    int cursor, /* cursor as a single value */
	dist, /* distance from unit to cursor */
	invtype, /* inventory type */
	c, /* counter */
	option; /* menu option */
    Cabinet *cabinet = NULL; /* target cabinet */

    /* check if we can access the inventory */
    if (! (unit = game->unit)) {
	display->error ("No unit selected");
	return;
    } else if (unit->class == UNIT_ENEMY) {
	display->error
	    ("%s doesn't take orders from you",
	     unit->name);
	return;
    } else if (unit->hostage) {
	display->error
	    ("%s must be freed first",
	     unit->name);
	return;
    }

    /* make sure the target is in range */
    cursor = cursorx + game->map->width * cursory;
    dist = distance (unit->x, unit->y, cursorx, cursory);
    if (game->unitcache[cursor] && dist > 1) {
	display->error ("Target is too far away");
	return;
    }

    /* determine what inventory type we're trying to access */
    if (cursorx == unit->x && cursory == unit->y)
	invtype = DISPLAY_INVENTORY_SINGLE;
    else if (game->unitcache[cursor] & MISSION_CACHE_UNIT)
	invtype = DISPLAY_INVENTORY_DUAL;
    else if (game->unitcache[cursor] & MISSION_CACHE_ENEMY)
	invtype = DISPLAY_INVENTORY_CHECK;
    else if (game->unitcache[cursor] & MISSION_CACHE_HOSTAGE)
	invtype = DISPLAY_INVENTORY_CHECK;
    else if (game->map->blocks[cursor]
	     == MISSION_FURNITURE + FURNITURE_CABINET)
	invtype = DISPLAY_INVENTORY_CABINET;
    else
	invtype = DISPLAY_INVENTORY_SINGLE;

    /* check target unit */
    if (invtype == DISPLAY_INVENTORY_DUAL ||
	invtype == DISPLAY_INVENTORY_CHECK)
    {
	target = game->getunit (game, cursorx, cursory);
	if (target &&
	    target->prone &&
	    (target->unconscious || ! target->health))
	    target = display->selectbody (target);
    }
    if (invtype == DISPLAY_INVENTORY_CHECK) {
	if (! (target->unconscious || target->health == 0)) {
	    display->error ("You cannot access their inventory");
	    return;
	}
	invtype = DISPLAY_INVENTORY_DUAL;
    }

    /* check we have enough action points */
    if (invtype == DISPLAY_INVENTORY_SINGLE);
    else if (unit->action < 1) {
	display->error ("Not enough action points");
	return;
    } else {
	--unit->action;
	display->showunitaction (unit);
    }

    /* check for cabinet */
    if (invtype == DISPLAY_INVENTORY_CABINET)
	for (c = 0; c < game->cabinetcount; ++c)
	    if (game->cabinets[c]->x == cursorx &&
		game->cabinets[c]->y == cursory)
		cabinet = game->cabinets[c];

    /* access the appropriate inventory type */
    initinventory (invtype, target, cabinet);
    do {
	inventorycursor (invtype, target, cabinet);
	if (controls->fire)
	    option = display->menu
		(INVMENU_COUNT, invmenu,
		 defaultinvoption (invtype, target, cabinet));
	else if (strchr (invkeys, toupper (controls->ascii)))
	    option
		= strchr (invkeys, toupper (controls->ascii))
		- invkeys;
	switch (option) {
	case INVMENU_EQUIP:
	    equip ();
	    break;
	case INVMENU_UNEQUIP:
	    unequip ();
	    break;
	case INVMENU_TRANSFER:
	    if (invtype == DISPLAY_INVENTORY_SINGLE)
		dropitem ();
	    else if (invtype == DISPLAY_INVENTORY_DUAL)
		transferitemunit (target);
	    else if (invtype == DISPLAY_INVENTORY_CABINET)
		transferitemcabinet (cabinet);
	    break;
	}
    } while (option != INVMENU_CLOSE);
    cleaninventory ();
}

/**
 * End the turn. Store the next game state in the game object.
 */
static void endturn (void)
{
    /* get confirmation first */
    if (! display->confirm ("End turn?"))
	return;

    /* increment the turn counter */
    ++game->turns;

    /* check for end of game */
    if (! game->detectend (game)) {
	game->state = STATE_COMPUTER;
	return;
    }

    /* check for victory */
    game->state = game->ascertainvictory (game)
	? STATE_VICTORY
	: STATE_DEFEAT;
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Initialise the UI.
 * @return   1 on success, 0 on failure.
 */
static int init (void)
{
    int c, /* counter */
	unitsonmap; /* 1 if there are units on the map, otherwise 0 */
    char briefing[384]; /* buffer for the briefing */

    /* initialise module-wide variables */
    game = getgame ();
    display = getdisplay ();
    controls = getcontrols ();

    /* centre map on first active unit */
    cursorx = game->entrance % game->map->width;
    cursory = game->entrance / game->map->width;
    game->unit = NULL;
    unitsonmap = 0;
    for (c = 0; c < game->unitcount; ++c)
	if (game->units[c] &&
	    (! game->unit ||
	     game->unit->health == 0 ||
	     game->unit->unconscious ||
	     ((game->units[c]->x || game->units[c]->y) &&
	      game->unit->x == 0 &&
	      game->unit->y == 0)))
	{
	    game->unit = game->units[c];
	    if (game->unit->x || game->unit->y) {
		unitsonmap = 1;
		cursorx = game->unit->x;
		cursory = game->unit->y;
	    }
	}
    centreview (cursorx, cursory);

    /* show initial display */
    display->showbackground ();
    if (game->unit)
	display->showunit (game->unit);
    display->showtitle (getmissiontype (game->type));
    game->briefing (game, briefing);
    display->typeset (briefing);
    display->showmap (viewx, viewy);
    if (! unitsonmap)
	display->showtutorial (TUTORIAL_DEPLOY);
    display->update ();
    display->playsound (DISPLAY_COMMS);

    /* success! */
    return 1;
}

/**
 * Operate the UI.
 * @return   1 on success, 0 on failure.
 */
static int operate (void)
{
    int option; /* menu option chosen */
    do {
	movecursor ();
	if (controls->fire)
	    option = display->menu
		(MAPMENU_COUNT,
		 mapmenu,
		 defaultoption ());
	else if (strchr (mapkeys, toupper (controls->ascii)))
	    option
		= strchr (mapkeys, toupper (controls->ascii))
		- mapkeys;
	switch (option) {
	case MAPMENU_NEXTUNIT:
	    nextunit ();
	    break;
	case MAPMENU_MOVE:
	    moveunit ();
	    break;
	case MAPMENU_SELECT:
	    selectunit ();
	    break;
	case MAPMENU_ATTACK:
	    attackunit ();
	    break;
	case MAPMENU_CREEP:
	    creepunit ();
	    break;
	case MAPMENU_HEAL:
	    healunit ();
	    break;
	case MAPMENU_FREE:
	    freeunit ();
	    break;
	case MAPMENU_PICKUP:
	    pickup ();
	    break;
	case MAPMENU_INVENTORY:
	    inventory ();
	    break;
	case MAPMENU_ENDTURN:
	    endturn ();
	    if (game->state != STATE_MAP)
		return game->state;
	    break;
	case MAPMENU_NEWGAME:
	    if (display->confirm ("New game?"))
		return (game->state = STATE_NEWGAME);
	    break;
	case MAPMENU_QUITGAME:
	    if (display->confirm ("Quit game?"))
		return 0;
	    break;
	}
    } while (1);
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Construct Generic UI.
 * @return The new AI.
 */
UI *new_MapUI (void)
{
    /* reserve memory for the UI */
    if (! (ui = new_UI ()))
	return NULL;

    /* initialise methods */
    ui->init = init;
    ui->operate = operate;

    /* return the UI */
    return ui;
}
