/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 30-May-2024.
 *
 * Config UI Module.
 */

/*
 * Required Headers
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* project headers */
#include "sccc.h"
#include "display.h"
#include "controls.h"
#include "game.h"
#include "unit.h"
#include "config.h"
#include "ui.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var ui A pointer to the UI structure. */
static UI *ui;

/** @var display The display object. */
static Display *display;

/** @var controls The controls object. */
static Controls *controls;

/** @var game The game object. */
static Game *game;

/** @var confmenu The config screen menu. */
static char *confmenu[] = {
    "Cancel",
    "Start game",
    "View scores",
    "Quit game"
};

/** @var confkeys Shortcut keys for config screen. */
static char *confkeys = " SVQ";

/** @enum ConfmenuID Identifiers for config menu options. */
typedef enum {
    CONFMENU_CANCEL,
    CONFMENU_STARTGAME,
    CONFMENU_VIEWSCORES,
    CONFMENU_QUITGAME,
    CONFMENU_COUNT
} ConfmenuID;

/** @var config A pointer to the configuration. */
static Config *config;

/** @var cursor The configuration cursor. */
static int cursor;

/** @var instructions Instruction Text. */
static char instructions[] = {
    "Select the configuration for the next game. Cursor up/down moves"
    " between the options; cursor left/right changes the options."
    " Everything except Level can be set to Random, for random"
    " adventures and character types."
};

/*----------------------------------------------------------------------
 * Level 2 Funciton Definitions.
 */

/**
 * Change an individual config item.
 * @param offset Direction of change.
 */
static void changeoption (int offset)
{
    /* change difficulty level */
    if (cursor == 0) {
	config->level += offset;
	if (config->level > MISSION_HARD)
	    config->level = MISSION_HARD;
	else if (config->level < MISSION_EASY)
	    config->level = MISSION_EASY;
    }

    /* change mission type */
    else if (cursor == 1) {
	config->type += offset;
	if (config->type < MISSION_RANDOM)
	    config->type = MISSION_RANDOM;
	else if (config->type > MISSION_GATHER)
	    config->type = MISSION_GATHER;
    }

    /* unit classes */
    else {
	config->classes[cursor - 2] += offset;
	if (config->classes[cursor - 2] < UNIT_NONE)
	    config->classes[cursor - 2] = UNIT_NONE;
	else if (config->classes[cursor - 2] > UNIT_INFANTRY)
	    config->classes[cursor - 2] = UNIT_INFANTRY;
    }
}

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Edit the configuration.
 */
static void editconfig (void)
{
    /* highlight the current config line */
    display->showconfigline (config, cursor, 1);

    /* main editing loop */
    do {

	/* wait for key press */
	display->update ();
	controls->release (0);
	controls->wait (0);
	controls->poll ();

	/* up and down controls */
	if (controls->up && cursor > 0) {
	    display->showconfigline (config, cursor, 0);
	    --cursor;
	    display->showconfigline (config, cursor, 1);
	} else if (controls->down && cursor < 5) {
	    display->showconfigline (config, cursor, 0);
	    ++cursor;
	    display->showconfigline (config, cursor, 1);
	}

	/* left and right controls */
	else if (controls->left) {
	    changeoption (-1);
	    display->showconfigline (config, cursor, 1);
	} else if (controls->right) {
	    changeoption (+1);
	    display->showconfigline (config, cursor, 1);
	}

    } while (! controls->fire &&
	     ! strchr (confkeys, toupper (controls->ascii)));
}

/**
 * Start the game.
 */
static void startgame (void)
{
    display->busy ("Initialising game...");
    game->generate (game, config);
    game->calcunitsightlines (game, generateprogress);
    display->preparemap (game);
    display->busy ("");
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Initialise the UI.
 * @return   1 on success, 0 on failure.
 */
static int init (void)
{
    int c; /* counter */

    /* initialise module-wide variables */
    display = getdisplay ();
    controls = getcontrols ();
    game = getgame ();
    config = getconfig ();

    /* show the initial display */
    cursor = 0;
    display->showbackground ();
    for (c = 0; c < 6; ++c)
	display->showconfigline (config, c, 0);
    display->showtitle ("Game Configuration");
    display->typeset (instructions);
    display->update ();

    /* success */
    return 1;
}

/**
 * Operate the UI.
 * @return   1 on success, 0 on failure.
 */
static int operate (void)
{
    int option; /* menu option chosen */
    do {
	editconfig ();
	if (controls->fire)
	    option = display->menu
		(CONFMENU_COUNT,
		 confmenu,
		 1);
	else if (strchr (confkeys, toupper (controls->ascii)))
	    option = strchr (confkeys, toupper (controls->ascii))
		- confkeys;
	switch (option) {
	case CONFMENU_VIEWSCORES:
	    game->state = STATE_SCORES;
	    return game->state;
	case CONFMENU_STARTGAME:
	    startgame ();
	    return game->state;
	case CONFMENU_QUITGAME:
	    if (display->confirm ("Quit game?"))
		return 0;
	    break;
	}
    } while (1);
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Construct Config Screen UI.
 * @return The new UI.
 */
UI *new_ConfigUI (void)
{
    /* reserve memory for the UI */
    if (! (ui = new_UI ()))
	return NULL;

    /* initialise methods */
    ui->init = init;
    ui->operate = operate;

    /* return the UI */
    return ui;
}
