/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 09-May-2024.
 *
 * Map UI Module.
 */

/*----------------------------------------------------------------------
 * Included Headers.
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "sccc.h"
#include "display.h"
#include "controls.h"
#include "game.h"
#include "unit.h"
#include "ai.h"
#include "ui.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var ui A pointer to the UI structure. */
static UI *ui;

/** @var display The display object. */
static Display *display;

/** @var controls The controls object. */
static Controls *controls;

/** @var game The game object. */
static Game *game;

/** @var cursorx The X coordinate of the cursor. */
static int cursorx;

/** @var cursory The Y coordinate of the cursor. */
static int cursory;

/** @var viewx The X coordinate of the view. */
static int viewx;

/** @var viewy The Y coordinate of the view. */
static int viewy;

/*----------------------------------------------------------------------
 * Callback Functions.
 */

/**
 * Progress message for building sight lines for computer turn.
 * @param unitno Unit for which sightlines are being constructed.
 */
static void computerlosprogress (int unitno)
{
    int percent; /* percentage progress */
    percent = 100 * unitno
	/ (2 * game->enemycount + game->unitcount + game->hostagecount);
    display->showcomputerprogress (percent);
    display->update ();
}

/**
 * Progress message for computer unit AI.
 * @param unitno Unit for which sightlines are being constructed.
 */
static void computeraiprogress (int unitno)
{
    int percent; /* percentage progress */
    percent
	= 100 * (unitno + game->unitcount)
	/ (2 * game->enemycount + game->unitcount + game->hostagecount);
    display->showcomputerprogress (percent);
    display->update ();
}

/**
 * Progress message for building sight lines for player turn.
 * @param unitno Unit for which sightlines are being constructed.
 */
static void playerlosprogress (int unitno)
{
    int percent; /* percentage progress */
    percent
	= 100 * (unitno + game->unitcount + game->enemycount)
	/ (2 * game->enemycount + game->unitcount + game->hostagecount);
    display->showcomputerprogress (percent);
    display->update ();
}

/*----------------------------------------------------------------------
 * Level 4 Function Definitions.
 */

/**
 * Ensure that a particular square is visible.
 * @param x The X coordinate of the square.
 * @param y The Y coordinate of the square.
 */
static void makevisible (int x, int y)
{
    if (x < viewx || x > viewx + 8 ||
	y < viewy || y > viewy + 8)
    {
	display->startdelay (500);
	if (x < viewx)
	    viewx = x;
	else if (x > viewx + 8)
	    viewx = x - 8;
	if (y < viewy)
	    viewy = y;
	else if (y > viewy + 8)
	    viewy = y - 8;
	display->showmap (viewx, viewy);
	display->update ();
	display->enddelay ();
    }
}

/*----------------------------------------------------------------------
 * Level 3 Function Definitions.
 */

/**
 * Update the map according to changed enemy visibility.
 * @param changes Changed bits in the visibility
 */
static void updatevisibility (int changes)
{
    game->visibility |= changes;
    if (changes & game->conscious)
	display->message ("%s seen", game->unit->name);
}

/**
 * Process the opportunity attacks.
 */
static void opportunityattacks (void)
{
    Unit *attacker; /* pointer to attacking unit */
    int unconscious, /* unconscious before attack */
	health; /* health before attack */
    unconscious = game->unit->unconscious;
    health = game->unit->health;
    while ((attacker = game->opportunity (game))) {
	makevisible (attacker->x, attacker->y);
	display->startdelay (500);
	display->updatemapsquare
	    (game,
	     game->unit->x,
	     game->unit->y);
	display->showmapsquare (cursorx, cursory, viewx, viewy);
	display->message (attacker->feedback);
	display->battleanimation
	    (attacker, game->unit, game, viewx, viewy);
	display->update ();
	if (health && ! game->unit->health)
	    display->playsound (DISPLAY_DEATH);
	else if (! unconscious && game->unit->unconscious)
	    display->playsound (DISPLAY_UNCONSCIOUS);
	display->enddelay ();
	unconscious = game->unit->unconscious;
	health = game->unit->health;
    }
}

/*----------------------------------------------------------------------
 * Level 2 Function Definitions.
 */

/**
 * Attack a unit.
 * @param x X coordinate of unit to attack.
 * @param y Y coordinate of unit to attack.
 * @return  1 if attack made, 0 if not.
 */
static int attackunit (x, y)
{
    Unit *target; /* the target unit */
    int loc, /* location of target unit */
	unconscious, /* unconscious before attack */
	health; /* health before attack */

    /* ascertain the target unit */
    loc = x + game->map->width * y;
    if (! (game->unitcache[loc] & MISSION_CACHE_UNIT))
	return 0;
    target = game->units[game->unitcache[loc] & 0xf];
    if (! target || target->health == 0)
	return 0;
    unconscious = game->unit->unconscious;
    health = game->unit->health;

    /* attack the target */
    if (game->unit->attack (game->unit, target, game)) {

	/* update the map square */
	display->updatemapsquare (game, target->x, target->y);

	/* do the screen updates */
	makevisible (x, y);
	display->startdelay (500);
	display->showmapsquare (x, y, viewx, viewy);
	display->message (game->unit->feedback);
	display->battleanimation
	    (game->unit, target, game, viewx, viewy);
	display->update ();
	if (health && ! game->unit->health)
	    display->playsound (DISPLAY_DEATH);
	else if (! unconscious && game->unit->unconscious)
	    display->playsound (DISPLAY_UNCONSCIOUS);
	display->enddelay ();

	/* opportunity attacks */
	opportunityattacks ();
	return 1;
    }

    /* no attack was made */
    return 0;
}

/**
 * Move to an adjacent location.
 * @param x X coordinate to move to.
 * @param y Y coordinate of unit to attack.
 * @return  1 if unit moved, 0 if not.
 */
static int moveunit (x, y)
{
    int prevx, /* previous square X coordinate */
	prevy, /* previous square Y coordinate */
	loc, /* location as a single integer */
	changedvisibility; /* ... of enemies and hostages */

    /* initialise */
    prevx = game->unit->x;
    prevy = game->unit->y;

    /* attempt to move */
    if (game->unit->move (game->unit, game, x, y)) {

	/* update the map square */
	display->updatemapsquare (game, prevx, prevy);
	display->updatemapsquare (game, x, y);
	loc = x + game->map->width * y;

	/* check for new units visible */
	if ((changedvisibility
	     = game->changedvisibility
	     (game,
	      x + game->map->width * y,
	      game->visibility)))
	    updatevisibility (changedvisibility);

	/* update the screen */
	if (game->sightlines[loc] & game->conscious) {
	    makevisible (x, y);
	    display->startdelay (500);
	    display->showmapsquare (prevx, prevy, viewx, viewy);
	    display->showmapsquare (x, y, viewx, viewy);
	    display->update ();
	    display->playsound (DISPLAY_PLOD);
	    display->enddelay ();
	}

	/* opportunity attacks */
	opportunityattacks ();

	/* return */
	return 1;

    }

    /* unit failed to move */
    return 0;
}

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Centre the view on a set of coordinates.
 * @param x The x coordinate.
 * @param y The y coordinate.
 * @return  1 if the view moved, 0 if not.
 */
static int centreview (int x, int y)
{
    int oldx, /* previous view X coordinate */
	oldy; /* previous view Y coordinate */

    /* record old view location */
    oldx = viewx;
    oldy = viewy;

    /* calculate extactly centred position */
    viewx = x - 4;
    viewy = y - 4;

    /* ensure we haven't gone off the map */
    if (viewx < 0)
	viewx = 0;
    else if (viewx > game->map->width - 9)
	viewx = game->map->width - 9;
    if (viewy < 0)
	viewy = 0;
    else if (viewy > game->map->height - 9)
	viewy = game->map->height - 9;

    /* tell the calling process if the view has moved */
    return (viewx != oldx || viewy != oldy);
}

/**
 * Perform the action required by the AI.
 * @param action The action ID.
 * @param X      The X coordinate for the action.
 * @param Y      The Y coordinate for the action.
 * @return       1 if the action was successful, 0 if not.
 */
static int performaction (int action, int x, int y)
{
    switch (action) {
    case AI_ACTION_ATTACK:
	return attackunit (x, y);
    case AI_ACTION_MOVE:
	return moveunit (x, y);
    default:
	return 0;
    }
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Initialise the UI.
 * @return   1 on success, 0 on failure.
 */
static int init (void)
{
    /* initialise module-wide variables */
    display = getdisplay ();
    controls = getcontrols ();
    game = getgame ();

    /* centre map on player's last current unit */
    cursorx = game->entrance % game->map->width;
    cursory = game->entrance / game->map->width;
    if (game->unit &&
	game->unit->x &&
	game->unit->y)
    {
	cursorx = game->unit->x;
	cursory = game->unit->y;
    }
    centreview (cursorx, cursory);
    
    /* show the initial display */
    display->showbackground ();
    display->showmap (viewx, viewy);
    display->showtitle ("Computer Turn");
    display->update ();
    display->startdelay (3000);
    game->calcunitsightlines (game, computerlosprogress);
    game->initturn (game);
    display->enddelay ();

    /* success */
    return 1;
}

/**
 * Operate the UI.
 * @param ui The UI object.
 * @return   1 on success, 0 on failure.
 */
static int operate (void)
{
    int c, /* counter */
	action, /* action required by AI */
	x, /* X coordinate of action */
	y; /* Y coordinate of action */
    AI *ai; /* AI for the current unit */

    /* invoke the AI for each of the units */
    for (c = 0; c < game->enemycount; ++c) {
	if (! game->enemies[c])
	    continue;
	game->unit = game->enemies[c];
	computeraiprogress (c);
	if (game->enemies[c]->health &&
	    ! game->enemies[c]->unconscious)
	{
	    ai = game->ai[c];
	    while (ai->getaction (ai, &action, &x, &y) &&
		   performaction (action, x, y));
	}
    }

    /* check for end of game */
    if (! game->detectend (game))
	return (game->state = STATE_MAP);

    /* check for victory */
    return
	(game->state = game->ascertainvictory (game)
	 ? STATE_VICTORY
	 : STATE_DEFEAT);
}

/**
 * Destroy the UI.
 */
static void destroy (void)
{
    display->startdelay (3000);
    game->calcunitsightlines (game, playerlosprogress);
    game->initturn (game);
    display->preparemap (game);
    free (ui);
    display->enddelay ();    
    ui = NULL;
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Construct Generic UI.
 * @return The new AI.
 */
UI *new_ComputerUI (void)
{
    /* reserve memory for the UI */
    if (! (ui = new_UI ()))
	return NULL;

    /* initialise methods */
    ui->init = init;
    ui->operate = operate;
    ui->destroy = destroy;

    /* return the UI */
    return ui;
}
