/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 30-May-2024.
 *
 * Score Table Module.
 */

/*
 * Required Headers
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "scores.h"
#include "game.h"
#include "utils.h"

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Clear the high score table.
 * @param scores The score table.
 */
static void clearscores (Scores *scores)
{
    int l, /* difficulty level counter */
	t; /* mission type counter */
    for (l = 0; l < 3; ++l)
	for (t = 0; t < 6; ++t)
	    scores->scores[l][t] = 0;
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Destroy the score table object.
 * @param scores The score table object.
 */
static void destroy (Scores *scores)
{
    if (scores)
	free (scores);
}

/**
 * Save the high score table.
 * @param scores The score table object.
 * @return       1 if successful, 0 if not.
 */
static int save (Scores *scores)
{
    FILE *output; /* the output file */
    int l, /* difficulty level counter */
	t; /* mission type counter */

    /* open the output file and write the header */
    if (! (output = fopen ("SCCC.SCO", "wb")))
	return 0;
    if (! (fwrite ("SCC001S", 8, 1, output))) {
	fclose (output);
	return 0;
    }

    /* write the scores */
    for (l = 0; l < 3; ++l)
	for (t = 0; t < 6; ++t)
	    if (! writebyte (&scores->scores[l][t], output)) {
		fclose (output);
		return 0;
	    }

    /* success */
    fclose (output);
    return 1;
}

/**
 * Load the high score table. If there is no valid high score
 * table file, initialise all the scores to zero.
 * @param scores The score table object.
 * @return       1 if successfully loaded, 0 if not.
 */
static int load (Scores *scores)
{
    FILE *input; /* the output file */
    char header[8]; /* header text */
    int l, /* difficulty level counter */
	t; /* mission type counter */

    /* open the input file and read and check the header */
    if (! (input = fopen ("SCCC.SCO", "rb")))
	return 0;
    if (! fread (header, 8, 1, input)) {
	fclose (input);
	return 0;
    }
    if (strcmp ("SCC001S", header)) {
	clearscores (scores);
	fclose (input);
	return 0;
    }

    /* write the scores */
    for (l = 0; l < 3; ++l)
	for (t = 0; t < 6; ++t)
	    if (! readbyte (&scores->scores[l][t], input)) {
		fclose (input);
		return 0;
	    }

    /* success */
    fclose (input);
    return 1;
}

/**
 * Submit a score to the high score table.
 * @param scores The score table object.
 * @param game   The game object.
 * @return       1 if the score was added, 0 if not.
 */
static int submit (Scores *scores, Game *game)
{
    /* validate difficulty level and mission type */
    if (game->level < MISSION_EASY ||
	game->level > MISSION_HARD ||
	game->type < MISSION_ASSASSINATE ||
	game->type > MISSION_GATHER)
	return 0;

    /* check the score and add it if it qualifies */
    if (game->score > scores->scores[game->level][game->type - 1]) {
	scores->scores[game->level][game->type - 1] = game->score;
	game->highest = 1;
    }

    /* tell calling process if a high score added */
    return game->highest;
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Create the score table object.
 * @return The new score table object.
 */
Scores *new_Scores (void)
{
    Scores *scores; /* score table */

    /* allocate memory for scores */
    if (! (scores = malloc (sizeof (Scores))))
	return NULL;

    /* initialise attributes */
    clearscores (scores);

    /* initialise methods */
    scores->destroy = destroy;
    scores->save = save;
    scores->load = load;
    scores->submit = submit;

    /* return the new score */
    return scores;
}

