/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 06-Aug-2020.
 *
 * Main Program Module.
 */

/*----------------------------------------------------------------------
 * Headers
 */

/* Standard ANSI C */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/* Project Headers */
#include "cgalib.h"
#include "levelmap.h"
#include "names.h"
#include "sccc.h"
#include "display.h"
#include "controls.h"
#include "fatal.h"
#include "game.h"
#include "scores.h"
#include "config.h"
#include "ui.h"

/*----------------------------------------------------------------------
 * File Level Variables.
 */

/** @var display options */
int displayoption = 2;

/** @var quiet Game is silent if 1, or has sound and music if 1. */
static int quiet = 0;

/** @var controls The controls object. */
static Controls *controls = NULL;

/** @var display is the display object. */
static Display *display = NULL;

/** @var names The name generator. */
static Names *names;

/** @var game The current game. */
static Game *game;

/** @var scores The high score table. */
static Scores *scores;

/** @var config The configuration. */
static Config *config;

/*----------------------------------------------------------------------
 * Callback Functions.
 */

/**
 * Display progress update for game generation.
 * @param unitno Unit number progress.
 */
void generateprogress (int unitno)
{
    char message[81]; /* message buffer */
    sprintf
	(message,
	 "Inititalising game (%d%%)",
	 100 * unitno / game->enemycount);
    display->busy (message);
}

/*----------------------------------------------------------------------
 * Private Level 2 Functions.
 */

/**
 * Work out desired program options from command line.
 * @param argc is the command line argument count.
 * @param argv is the array of command line argumets.
 */
static void initialiseargs (int argc, char **argv)
{
    /* local variables */
    int a; /* argument count */

    /* look for recognised parameters */
    for (a = 1; a < argc; ++a)
	if (! strcmp (argv[a], "-p") || ! strcmp (argv[a], "-P"))
	    displayoption = 1;
	else if (! strcmp (argv[a], "-m") || ! strcmp (argv[a], "-M"))
	    displayoption = 0;
	else if (! strcmp (argv[a], "-q") || ! strcmp (argv[a], "-Q"))
	    quiet = 1;
	else
	    fatal_error (FATAL_COMMAND_LINE);
}

/**
 * Generate the player's first game.
 */
static void generatefirstgame (void)
{
    display->busy ("Initialising game...");
    game->generate (game, config);
    game->calcunitsightlines (game, generateprogress);
    game->initturn (game);
    display->busy ("");
}

/*----------------------------------------------------------------------
 * Private Level 1 Functions.
 */

/**
 * Initialise the program.
 * @param argc is the command line argument count.
 * @param argv is the array of command line argumets.
 */
static void initialiseprogram (int argc, char **argv)
{
    /* check command line */
    initialiseargs (argc, argv);

    /* initialise the random number generator */
    srand (time (NULL));

    /* initialise the name generator */
    if (! (names = new_Names ()))
	fatal_error (FATAL_MEMORY);

    /* initialise controls */
    if (! (controls = new_Controls ()))
	fatal_error (FATAL_MEMORY);

    /* initialise the display and assets */
    if (! (display = new_Display (displayoption, quiet)))
	fatal_error (FATAL_DISPLAY);
    display->showtitlescreen ();

    /* load or create high score table */
    if (! (scores = new_Scores ()))
	fatal_error (FATAL_MEMORY);
    scores->load (scores);

    /* load or create the configuration */
    if (! (config = new_Config ()))
	fatal_error (FATAL_MEMORY);
    config->load (config);

    /* initialise the game */
    if (! (game = new_Game ()))
	fatal_error (FATAL_MEMORY);
    display->setgame (game);
    if (! game->load (game))
	generatefirstgame ();
    else
	display->busy ("");
    display->preparemap (game);

    /* await the fire key */
    display->titlekey ();
}

/**
 * The main game loop selects a screen to activate.
 */
static int playgame (void)
{
    UI *ui; /* the UI for the current game state */
    int retstate; /* state to return */

    /* construct the current game view */
    switch (game->state) {
    case STATE_MAP:
	ui = new_MapUI ();
	break;
    case STATE_COMPUTER:
	ui = new_ComputerUI ();
	break;
    case STATE_VICTORY:
	ui = new_VictoryUI ();
	break;
    case STATE_DEFEAT:
	ui = new_DefeatUI ();
	break;
    case STATE_SCORES:
	ui = new_ScoresUI ();
	break;
    case STATE_NEWGAME:
	ui = new_ConfigUI ();
	break;
    default:
	return STATE_NONE;
    }

    /* initialise the current game view */
    if (! ui->init ())
	fatal_error (FATAL_INITUI);

    /* operate the UI */
    retstate = ui->operate ();

    /* return the next game state or STATE_NONE for exit */
    ui->destroy ();
    return retstate;
}

/**
 * Clean up when the user quits normally.
 */
static void endprogram (void)
{
    game->save (game);
    display->destroy ();
    controls->destroy ();
    names->destroy (names);
    game->destroy (game);
    scores->save (scores);
    scores->destroy (scores);
    config->save (config);
    config->destroy (config);
}

/*----------------------------------------------------------------------
 * Top Level Routine.
 */

/**
 * Share the name generator with other modules.
 * @return A pointer to the name generator.
 */
Names *getnamegenerator (void)
{
    return names;
}

/**
 * Share the control handler with other modules.
 * @return A pointer to the game control handler.
 */
Controls *getcontrols (void)
{
    return controls;
}

/**
 * Share the display handler with other modules.
 * @return A pointer to the display module.
 */
Display *getdisplay (void)
{
    return display;
}

/**
 * Share the game handler with other modules.
 * @return A pointer to the game module.
 */
Game *getgame (void)
{
    return game;
}

/**
 * Share the scores handler with other modules.
 * @return A pointer to the scores module.
 */
Scores *getscores (void)
{
    return scores;
}

/**
 * Share the config handler with other modules.
 * @return A pointer to the config module.
 */
Config *getconfig (void)
{
    return config;
}

/**
 * Main Program.
 * @param argc is the count of arguments from the OS.
 * @param argv is an array of arguments from the OS.
 * @returns 0 on a successful run.
 */
int main (int argc, char **argv)
{
    initialiseprogram (argc, argv);
    while (playgame ());
    endprogram ();
    return 0;
}
