/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 31-May-2024.
 *
 * AI Module.
 */

/*----------------------------------------------------------------------
 * Required Headers.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "sccc.h"
#include "fatal.h"
#include "game.h"
#include "ai.h"
#include "unit.h"
#include "item.h"
#include "utils.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var game A pointer to the game object. */
static Game *game;

/** @var routex X destination of last route found. */
static int routex;

/** @var routey Y destination of last route found. */
static int routey;

/** @var costs Movements costs for squares to destination. */
static int *costs;

/*----------------------------------------------------------------------
 * Level 2 Function Definitions.
 */

/**
 * Ascertain if an attack target is in attack range.
 * @param attacker The attacking unit.
 * @param defender The target unit.
 * @return         1 if in range, 0 if not.
 */
static int inrange (Unit *attacker, Unit *defender) {

    /* when using a gun, always in range */
    if (attacker->inventory[1] == ITEM_PISTOL ||
	attacker->inventory[3] == ITEM_PISTOL)
	return 1;
    if (attacker->inventory[1] == ITEM_RIFLE &&
	attacker->inventory[3] == ITEM_NONE)
	return 1;
    if (attacker->inventory[3] == ITEM_RIFLE &&
	attacker->inventory[1] == ITEM_NONE)
	return 1;

    /* otherwise unit must be adjacent */
    return
	distance (attacker->x, attacker->y, defender->x, defender->y)
	== 1;
}

/**
 * Find a route from anywhere to a particular point on the map.
 * @param dx X destination.
 * @param dy Y destination.
 */
static void findroute (int dx, int dy)
{
    /* local variables */
    int width, /* width of the map */
	height, /* height of the map */
	sx, /* start x coordinate of sweep */
	sy, /* start y coordinate of sweep */
	ex, /* end x coordinate of sweep */
	ey, /* end y coordinate of sweep */
	x, /* current x coordinate of sweep */
	y, /* current y coordinate of sweep */
	dir, /* current direction of sweep */
	xo, /* x offset to examine */
	yo, /* y offset to examine */
	pos, /* 1-dimensional position with offset */
	moves, /* movement cost into a square */
	best, /* best cost met so far */
	modified, /* 1 if costs modified this sweep */
	block, /* block at location under consideration */
	multiplier; /* multiplier for temporary blockages */
    LevelMap *map; /* map of the battlefield */

    /* initialise */
    map = game->map;
    width = map->width;
    height = map->height;
    if (costs)
	free (costs);
    if (! (costs = calloc (width * height, sizeof (int))))
	fatal_error (FATAL_MEMORY);

    /* outer loop - sweep map until all costs finalised */
    costs[dx + width * dy] = 1;
    sx = sy = 0;
    ex = width - 1;
    ey = height - 1;
    dir = 1;
    do {

	/* inner loop - perform a single sweep */
	modified = 0;
	for (x = sx; x != ex + dir; x += dir)
	    for (y = sy; y != ey + dir; y += dir) {
 
		/* don't check the destination or blocked squares */
		pos = x + map->width * y;
		block = map->blocks[pos] & 0x70;
		if (x == dx && y == dy)
		    continue;
		if (block != LEVELMAP_DOOR && block != LEVELMAP_OPEN)
		    continue;
		multiplier
		    = (map->blocks[pos] & 0x80)
		    ? 100
		    : 1;


		/* inner inner loop - look around */
		best = moves = 0;
		for (xo = -1; xo <= 1; ++xo)
		    for (yo = -1; yo <= 1; ++yo) {
			pos = (x + xo) + width * (y + yo);
			if (xo == 0 && yo == 0)
			    continue;
			if (x + xo < 0 || x + xo >= width ||
			    y + yo < 0 || y + yo >= height)
			    continue;
			if ((costs[pos] > 0 ||
			     (x + xo == dx && y + yo == dy)) &&
			    (costs[pos] < best ||
			     best == 0))
			{
			    best = costs[pos];
			    moves = multiplier
				* (1 + abs (xo) + abs (yo)
				   + (block == LEVELMAP_DOOR));
			}
		    }

		/* modify square if better cost found */
		if (costs[x + width * y] != best + moves && best != 0) {
		    costs[x + width * y] = best + moves;
		    modified = 1;
		}
	    }

	/* reverse direction for next sweep */
	sx ^= ex;
	ex ^= sx;
	sx ^= ex;
	sy ^= ey;
	ey ^= sy;
	sy ^= ey;
	dir = -dir;
    } while (modified);
}

/**
 * Find the next square on the previously plotted route.
 * @param dx A pointer to the X coordinate.
 * @param dy A pointer to the Y coordinate.
 * @return   1 if a square found, 0 if not.
 */
static int nextsquare (AI *ai, int *nx, int *ny)
{
    /* local variables */
    int dx, /* destination X coordinate */
	dy, /* destination Y coordinate */
	xo, /* x offset to check */
	yo, /* y offset to check */
	xdir, /* x direction towards map centre */
	ydir, /* y direction towards map centre */
	best, /* best cost of potential next square */
	cost, /* cost of potential next square */
	x, /* current x coordinate of unit */
	y, /* current y coordinate of unit */
	width, /* width of the map */
	height; /* height of the map */

    /* initialise */
    width = game->map->width;
    height = game->map->height;
    dx = ai->investigating ? ai->investx : ai->orderx;
    dy = ai->investigating ? ai->investy : ai->ordery;
    x = game->unit->x;
    y = game->unit->y;
    *nx = *ny = -1;
    best = 0;
    xdir = (x < width / 2) - (x >= width / 2);
    ydir = (y < height / 2) - (y >= height / 2);

    /* validate */
    if (x == dx && y == dy)
	return 0;

    /* find the best adjacent square */
    for (xo = -xdir; xo != 2 * xdir; xo += xdir)
	for (yo = -ydir; yo != 2 * ydir; yo += ydir) {
	    if (x + xo < 0 || x + xo >= width ||
		y + yo < 0 || y + yo >= height ||
		(xo == 0 && yo == 0))
		continue;
	    if (! (cost = costs[(x + xo) + width * (y + yo)]));
	    else if (cost < best || best == 0) {
		best = cost;
		*nx = x + xo;
		*ny = y + yo;
	    } else if (cost <= best && (xo == 0 || yo == 0)) {
		best = cost;
		*nx = x + xo;
		*ny = y + yo;
	    }
	}

    /* return now if we couldn't find a square */
    if (*nx == -1 || *ny == -1)
	return 0;

    /* otherwise set the next square */
    return 1;
}

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Check if enemies are in sight before doing anything else.
 * @param ai The AI object.
 * @return   1 if there are enemies in sight, 0 if not.
 */
static int enemiesinsight (AI *ai)
{
    int loc; /* unit location */
    loc = ai->unit->x + game->map->width * ai->unit->y;
    return game->sightlines[loc] != 0;
}

/**
 * Attempt to attack an enemy unit.
 * @param ai     The AI object.
 * @param option Pointer to The option.
 * @param x      Pointer to the target X coordinate.
 * @param y      Pointer to the target Y coordinate.
 * @return       1 if action decided, 0 otherwise.
 */
static int attack (AI *ai, int *action, int *x, int *y)
{
    Unit *target = NULL; /* target unit */
    int c, /* counter */
	dist, /* distance to unit */
	closest = 0, /* closest distance */
	visible, /* bitfield of visible units */
	origin, /* origin location */
	dest; /* destination location */

    /* determine which unit to attack */
    origin = ai->unit->x + game->map->width * ai->unit->y;
    visible = game->sightlines[origin];
    for (c = 0; c < game->unitcount; ++c) {

	/* skip nonexistent and dead units */
	if (! game->units[c] ||
	    game->units[c]->health == 0)
	    continue;

	/* check unit if in line of sight */
	dest = game->units[c]->x + game->map->width * game->units[c]->y;
	if (visible & (1 << c) &&
	    game->lineofsight (game, origin, dest))
	{
	    dist = distance
		(ai->unit->x, ai->unit->y,
		 game->units[c]->x, game->units[c]->y);
	    if (closest == 0 || dist < closest) {
		dist = closest;
		target = game->units[c];
	    }
	}
    }

    /* no target - return failure */
    if (! target)
	return 0;

    /* switch investigation to this target */
    ai->investigating = 1;
    ai->investx = target->x;
    ai->investy = target->y;

    /* attack the target, if in range */
    if (inrange (ai->unit, target)) {
	*action = AI_ACTION_ATTACK;
	*x = target->x;
	*y = target->y;
	return 1;
    }

    /* approach the target, if not in range */
    if (routex != target->x || routey != target->y)
	findroute (target->x, target->y);
    if (nextsquare (ai, x, y)) {
	*action = AI_ACTION_MOVE;
	return 1;
    }

    /* return failure */
    return 0;
}

/**
 * Continue with an investigation.
 * @param ai     The AI object.
 * @param option Pointer to The option.
 * @param x      Pointer to the target X coordinate.
 * @param y      Pointer to the target Y coordinate.
 * @return       1 if action decided, 0 otherwise.
 */
static int investigate (AI *ai, int *action, int *x, int *y)
{
    /* stop investigating when we arrive */
    if (ai->unit->x == ai->investx && ai->unit->y == ai->investy) {
	ai->investigating = 0;
	ai->investx = 0;
	ai->investy = 0;
	return 0;
    }

    /* otherwise find a route and approach */
    if (routex != ai->investx || routey != ai->investy)
	findroute (ai->investx, ai->investy);
    if (nextsquare (ai, x, y)) {
	*action = AI_ACTION_MOVE;
	return 1;
    }
    return 0;
}

/**
 * Get an action while idle.
 * @param ai     The AI object.
 * @param option Pointer to The option.
 * @param x      Pointer to the target X coordinate.
 * @param y      Pointer to the target Y coordinate.
 * @return       1 if action decided, 0 otherwise.
 */
static int idle (AI *ai, int *action, int *x, int *y)
{
    return 0;
}

/**
 * Get an action while guarding.
 * @param ai     The AI object.
 * @param option Pointer to The option.
 * @param x      Pointer to the target X coordinate.
 * @param y      Pointer to the target Y coordinate.
 * @return       1 if action decided, 0 otherwise.
 */
static int guard (AI *ai, int *action, int *x, int *y)
{
    /* if we can see the objective, stay put */
    if (game->lineofsight
	(game,
	 ai->unit->x + game->map->width * ai->unit->y,
	 ai->orderx + game->map->width * ai->ordery))
	return 0;

    /* otherwise work out a route and move in that direction */
    if (ai->unit->x == ai->orderx && ai->unit->y == ai->ordery)
	ai->newpatrollocation (ai);
    if (routex != ai->orderx || routey != ai->ordery)
	findroute (ai->orderx, ai->ordery);
    if (nextsquare (ai, x, y)) {
	*action = AI_ACTION_MOVE;
	return 1;
    }
    return 0;
}

/**
 * Get an action while patrolling.
 * @param ai     The AI object.
 * @param option Pointer to The option.
 * @param x      Pointer to the target X coordinate.
 * @param y      Pointer to the target Y coordinate.
 * @return       1 if action decided, 0 otherwise.
 */
static int patrol (AI *ai, int *action, int *x, int *y)
{
    if (ai->unit->x == ai->orderx && ai->unit->y == ai->ordery)
	ai->newpatrollocation (ai);
    if (routex != ai->orderx || routey != ai->ordery)
	findroute (ai->orderx, ai->ordery);
    if (nextsquare (ai, x, y)) {
	*action = AI_ACTION_MOVE;
	return 1;
    }
    return 0;
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Destroy the AI when no longer needed.
 * @param ai The AI to destroy.
 */
static void destroy (AI *ai)
{
    if (ai)
	free (ai);
    if (costs) {
	free (costs);
	costs = NULL;
	routex = 0;
	routey = 0;
    }
}

/**
 * Write the AI to an already open file.
 * @param ai     The AI to write.
 * @param output The output file.
 * @return       1 if successful, 0 otherwise.
 */
static int write (AI *ai, FILE *output)
{
    /* write current order */
    if (! writebyte (&ai->order, output))
	return 0;
    if (! writebyte (&ai->orderx, output))
	return 0;
    if (! writebyte (&ai->ordery, output))
	return 0;

    /* write current investigation */
    if (! writebyte (&ai->investigating, output))
	return 0;
    if (! writebyte (&ai->investx, output))
	return 0;
    if (! writebyte (&ai->investy, output))
	return 0;

    /* success! */
    return 1;
}

/**
 * Read the AI to an already open file.
 * @param ai    The AI to read.
 * @param input The input file.
 * @return      1 if successful, 0 otherwise.
 */
static int read (AI *ai, FILE *input)
{
    /* read current order */
    if (! readbyte (&ai->order, input))
	return 0;
    if (! readbyte (&ai->orderx, input))
	return 0;
    if (! readbyte (&ai->ordery, input))
	return 0;

    /* read current investigation */
    if (! readbyte (&ai->investigating, input))
	return 0;
    if (! readbyte (&ai->investx, input))
	return 0;
    if (! readbyte (&ai->investy, input))
	return 0;

    /* success! */
    return 1;
}

/**
 * Get a single action.
 * @param ai     The AI object.
 * @param option Pointer to The option.
 * @param x      Pointer to the target X coordinate.
 * @param y      Pointer to the target Y coordinate.
 * @return       1 if action decided, 0 otherwise.
 */
static int getaction (AI *ai, int *action, int *x, int *y)
{
    /* if within sight of enemies, attack one of them */
    if (enemiesinsight (ai) &&
	attack (ai, action, x, y))
	return 1;

    /* if investigating, continue doing so */
    if (ai->investigating &&
	investigate (ai, action, x, y))
	return 1;

    /* otherwise just continue with order */
    switch (ai->order) {
    case AI_ORDER_NONE:
	return idle (ai, action, x, y);
    case AI_ORDER_GUARD:
	return guard (ai, action, x, y);
    case AI_ORDER_PATROL:
	return patrol (ai, action, x, y);
    default:
	return 0;
    }
}

/**
 * Find a new patrol location for the AI.
 * @param ai The AI object.
 */
static void newpatrollocation (AI *ai)
{
    int loc; /* potential patrol location */
    do {
	ai->orderx = rand () % game->map->width;
	ai->ordery = rand () % game->map->height;
	loc = ai->orderx + game->map->width * ai->ordery;
    } while ((game->map->blocks[loc] & 0x70) != LEVELMAP_OPEN);
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Alert nearby units to a noise.
 * @param x X coordinate of the noise.
 * @param y Y coordinate of the noise.
 */
void ai_alert (int x, int y)
{
    int c, /* counter */
	dist, /* distance betweeen two units */
	nearest = 0, /* distance of nearest unit */
	id = -1; /* id of nearest unit */

    /* find the closest unit to the noise */
    for (c = 0; c < game->enemycount; ++c)

	/* a unit is already investigating this noise */
	if (game->ai[c]->investigating &&
	    game->ai[c]->investx == x &&
	    game->ai[c]->investy == y)
	    return;

	/* check distance */
	else {
	    dist = distance
		(x, y,
		 game->ai[c]->unit->x, game->ai[c]->unit->y);
	    if (nearest == 0 || dist < nearest) {
		nearest = dist;
		id = c;
	    }
	}

    /* alert unit of the noise */
    if (id != -1) {
	game->ai[id]->investigating = 1;
	game->ai[id]->investx = x;
	game->ai[id]->investy = y;
    }
}

/**
 * AI Constructor.
 * @return The new AI.
 */
AI *new_AI (void)
{
    AI *ai; /* new AI object */

    /* reserve memory for AI */
    if (! (ai = malloc (sizeof (AI))))
	return NULL;

    /* initialise class variables */
    game = getgame ();

    /* initialise attributes */
    ai->unit = NULL;
    ai->order = AI_ORDER_NONE;
    ai->orderx = 0;
    ai->ordery = 0;
    ai->investigating = 0;
    ai->investx = 0;
    ai->investy = 0;

    /* initialise methods */
    ai->destroy = destroy;
    ai->write = write;
    ai->read = read;
    ai->getaction = getaction;
    ai->newpatrollocation = newpatrollocation;

    /* return the AI */
    return ai;
}

