/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 16-Aug-2020.
 *
 * Unit Header.
 */

/* types defined in this file */
typedef struct unit Unit;

#ifndef __UNIT_H__
#define __UNIT_H__

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/* external types */
typedef struct game Game;
typedef struct skill Skill;
typedef struct cabinet Cabinet;

/**
 * @enum unitlevel
 * The difficulty levels available for character generation.
 * Player characters are UNIT_HARD.
 */
typedef enum unitlevel {
    UNIT_EASY, /* character has 6 skill points */
    UNIT_FAIR, /* character has 9 skill points */
    UNIT_HARD /* character has 12 skill points */
} UnitLevel;

/** @enum unitclass Unit class values. */
typedef enum unitclass {
    UNIT_NONE,
    UNIT_MEDIC,
    UNIT_SCOUT,
    UNIT_AUXILIARY,
    UNIT_INFANTRY,
    UNIT_ENEMY,
    UNIT_HOSTAGE,
    UNIT_CLASSES
} UnitClass;

/** @enum unitslot Unit inventory slot. */
typedef enum unitslot {
    UNIT_HEAD,
    UNIT_HAND,
    UNIT_BODY,
    UNIT_OFFHAND
} UnitSlot;

/** @enum FeedbackID Simple IDs for action feedback. */
typedef enum {
    UNIT_FEEDBACK_NONE, /* nothing happened */
    UNIT_FEEDBACK_MISS, /* combat miss */
    UNIT_FEEDBACK_HIT, /* combat hit */
    UNIT_FEEDBACK_NOHEALED, /* healing failed */
    UNIT_FEEDBACK_CONSCIOUS, /* healing restored consciousness */
    UNIT_FEEDBACK_RESTORED, /* healing restored health */
    UNIT_FEEDBACK_HEALED, /* healing improved health */
    UNIT_FEEDBACK_CREEPING, /* unit is creeping */
    UNIT_FEEDBACK_NORMAL /* unit is moving normally */
} FeedbackID;

/** @enum SurpriseLevel The levels of surprise. */
typedef enum {
    UNIT_SURPRISED, /* unit is surprised */
    UNIT_NOTSURPRISED /* unit is not surprised */
} SurpriseLevel;

/**
 * @struct unit is the main combat unit structure.
 */
struct unit {

    /*
     * Attributes
     */

    /** @var id The unit ID for map caches. */
    int id;

    /**
     * @var name is the name of the unit.
     * It identifies the unit on the mission map and in lists.
     */
    char name[13];

    /** @var level The level of the unit. */
    int level;

    /** @var class the character class of the unit. */
    int class;

    /** @var hostage 1 if the unit is a hostage, 0 if free. */
    int hostage;

    /** @var unconscious 1 if the unit is unconscious, 0 if not. */
    int unconscious;

    /**
     * @var encumbrance
     * The encumbrance value. 0 for unencumbered, or
     * 1+ for encumbered, being the number of excess items carried.
     */
    int encumbrance;

    /** @var surprise The surprise level. */
    int surprise;

    /** @var creeping 1 if attempting to use stealth, 0 otherwise. */
    int creeping;

    /**
     * @var strength is the unit's strength.
     * It affects physical attacks, health and action points.
     */
    int strength;

    /**
     * @var agility is the unit's agility.
     * It affects action points.
     */
    int agility;

    /**
     * @var endurance is the unit's endurance.
     * It affects health.
     */
    int endurance;

    /**
     * @var action is the unit's action points.
     * It records how much more a unit can do this turn.
     */
    int action;

    /**
     * @var health is the unit's current health.
     * When reduced to 0, the unit is dead.
     */
    int health;

    /**
     * @var skills The list of skills.
     */
    Skill *skills[4];

    /**
     * @var inventory The player inventory.
     * Slots 0..3 are equipped items.
     * Slots 4..9 are the backpack.
     */
    int inventory[10];

    /**
     * @var x is the unit's x location on the map.
     * When outside the bounds of the map, the unit is not present.
     */
    int x;

    /**
     * @var y is the unit's y location on the map.
     * When outside the bounds of the map, the unit is not present.
     */
    int y;

    /** @var combattype The type of combat that last occurred. */
    int combattype;

    /** @var prone ID of unconscious unit under this one. */
    int prone;

    /** @var feedbackid Numeric feedback from actions. */
    int feedbackid;

    /** @var feedback Feedback from actions. */
    char feedback[61];

    /*
     * Methods
     */

    /**
     * Destroy the unit when it is no longer needed.
     * @param unit is the unit to destroy.
     */
    void (*destroy) (Unit *unit);

    /**
     * Write the unit to an open file.
     * @param unit   The unit object.
     * @param output The output file handle.
     * @return       1 on success, 0 on failure.
     */
    int (*write) (Unit *unit, FILE *output);

    /**
     * Read the unit from an open file.
     * @param unit  The unit object.
     * @param input The input file handle.
     * @return      1 on the success, 0 on failure.
     */
    int (*read) (Unit *unit, FILE *input);

    /**
     * Generate a unit at random.
     * @param unit is the unit to generate.
     */
    int (*generate) (Unit *unit);

    /**
     * Deploy the unit on the map.
     * @param unit The unit object.
     * @param game The game object.
     * @param x    The new X coordinate.
     * @param y    The new Y coordinate.
     * @return     1 if move successful, 0 if not.
     */
    int (*deploy) (Unit *unit, Game *game, int x, int y);

    /**
     * Remove the unit from the map.
     * @param unit The unit object.
     * @param game The game object.
     * @return     1 if move successful, 0 if not.
     */
    int (*undeploy) (Unit *unit, Game *game);

    /**
     * Move the unit on the map.
     * @param unit The unit object.
     * @param game The game object.
     * @param x    The new X coordinate.
     * @param y    The new Y coordinate.
     * @return     1 if move successful, 0 if not.
     */
    int (*move) (Unit *unit, Game *game, int x, int y);

    /**
     * Attack another unit.
     * @param unit   The unit that is attacking.
     * @param target The target of the attack.
     * @param game   The game map.
     * @return       1 if attack made, 0 if not.
     */
    int (*attack) (Unit *unit, Unit *target, Game *game);

    /**
     * Reset a unit ready for the next turn.
     * @param unit The unit object.
     */
    void (*reset) (Unit *unit);

    /**
     * Return the unit's level in a particular skill.
     * @param unit    The unit object.
     * @param skillid The skill to check.
     * @return        The level of skill.
     */
    int (*getskill) (Unit *unit, int skillid);

    /**
     * Transfer an item between the unit and another unit.
     * @param unit     The unit.
     * @param target   The target.
     * @param fromslot The slot from which the item is taken.
     * @param toslot   Pointer to destination slot.
     * @return         1 if successful, 0 on failure.
     */
    int (*transferitemunit)
    (Unit *unit, Unit *target, int fromslot, int *toslot);

    /**
     * Transfer an item between the unit and a cabinet.
     * @param unit The unit.
     * @param cabinet  The cabinet.
     * @param fromslot The slot from which the item is taken.
     * @param toslot   Pointer to destination slot.
     * @return         1 if successful, 0 on failure.
     */
    int (*transferitemcabinet)
    (Unit *unit, Cabinet *cabinet, int fromslot, int *toslot);

    /**
     * Unequip an item from the unit's body.
     * @param unit   The unit object.
     * @param origin The origin slot.
     * @param dest   Pointer to the destination slot number.
     * @return       1 if successful, 0 if not.
     */
    int (*unequip) (Unit *unit, int origin, int *dest);
    
    /**
     * Equip an item on the unit's body.
     * @param unit   The unit object.
     * @param origin The origin slot.
     * @param dest   Pointer to the destination slot number.
     * @return       1 if successful, 0 if not.
     */
    int (*equip) (Unit *unit, int origin, int *dest);

    /**
     * Drop an item on the ground.
     * @param unit The unit object.
     * @param slot The inventory slot.
     * @param game The game object.
     * @return     1 if successful, 0 if not.
     */
    int (*drop) (Unit *unit, int slot, Game *game);
    
    /**
     * Pick up a unit from the ground.
     * @param unit The unit object.
     * @param slot Pointer to the inventory slot.
     * @param game The game object.
     * @return     1 if successful, 0 if not.
     */
    int (*pickup) (Unit *unit, int *slot, Game *game);

    /**
     * Free a hostage.
     * @param unit   The unit object.
     * @param target The target unit.
     * @return       1 if successful, 0 if not.
     */
    int (*freehostage) (Unit *unit, Unit *target);

    /**
     * Heal a unit.
     * @param unit   The unit object.
     * @param target The target unit.
     * @return       1 if successful, 0 if not.
     */
    int (*heal) (Unit *unit, Unit *target);

};

/*----------------------------------------------------------------------
 * Function Prototypes: Constructors.
 */

/**
 * Construct a unit.
 * @returns a pointer to the new unit.
 */
Unit *new_Unit (void);

/**
 * Get the name of a particular class.
 * @returns A pointer to the class name.
 */
char *getclassname (int class);

/**
 * Get the text of an error message.
 * @return       A pointer to the buffer.
 */
char *getuniterror (void);

#endif
