/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 10-Nov-2023..
 *
 * Utility Functions Header.
 */

/*----------------------------------------------------------------------
 * Required Headers.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "utils.h"

/*----------------------------------------------------------------------
 * Public Level Function Definitions.
 */

/**
 * Roll a number of six-sided dice.
 * If min and max are equal, there is no range checking.
 * @param dice The number of dice to throw.
 * @param dm   Dice modifier.
 * @param min  Minimum throw allowed.
 * @param max  Maximum throw allowed.
 * @return     The dice throw.
 */
int throwdice (int dice, int dm, int min, int max)
{
    int throw; /* the final throw */

    /* throw the dice */
    throw = 0;
    while (dice--)
	throw += rand () % 6 + 1;

    /* apply the DM */
    throw += dm;

    /* range check */
    if (min == max)
	; /* no min or max */
    else if (throw < min)
	throw = min;
    else if (throw > max)
	throw = max;

    /* return the throw */
    return throw;
}

/**
 * Roll a number of dice and return the highest value.
 * @param dice The number of dice to throw.
 * @return     The highest of the dice thrown.
 */
int highestdice (int dice)
{
    int count, /* dice counter */
	roll, /* roll of each dice */
	highest; /* highest roll */
    highest = 1;
    for (count = 1; count <= dice; ++count) {
	roll = 1 + rand () % 6;
	if (roll > highest)
	    highest = roll;
    }
    return highest;
}

/**
 * Roll 3 dice and return the total of the highest 2.
 * @return The highest 2 dice.
 */
int highest2of3 (void)
{
    int roll[3],
	count,
	lowest;
    roll[0] = 1 + rand () % 6;
    roll[1] = 1 + rand () % 6;
    roll[2] = 1 + rand () % 6;
    lowest = 6;
    for (count = 0; count < 3; ++count)
	if (roll[count] <= lowest)
	    lowest = roll[count];
    return roll[0] + roll[1] + roll[2] - lowest;
}

/**
 * Roll a number of dice and return the lowest value.
 * @param dice The number of dice to throw.
 * @return     The lowest of the dice thrown.
 */
int lowestdice (int dice)
{
    int count, /* dice counter */
	roll, /* roll of each dice */
	lowest; /* lowest roll */
    lowest = 6;
    for (count = 1; count <= dice; ++count) {
	roll = 1 + rand () % 6;
	if (roll < lowest)
	    lowest = roll;
    }
    return lowest;
}

/**
 * Take a skill roll.
 * @param level The skill level.
 * @return      0 = fail, 1 = pass, 2 = critical.
 */
int skillroll (int level)
{
    int roll1, /* first roll */
	roll2; /* second roll */
    roll1 = 1 + rand () % 6;
    roll2 = 1 + rand () % 6;
    if (roll1 == 6 && roll2 == 6)
	return 0;
    else if (level == 6 && (roll1 == 1 || roll2 == 1))
	return 2;
    else if (roll1 == 1 && roll2 == 1)
	return 1;
    else
	return roll1 <= level || roll2 <= level;
}

/**
 * Write an integer as 2 bytes to an already open output file.
 * @param  value  A pointer to the integer variable to write.
 * @param  output The output file handle.
 * @return        1 if successful, 0 if not.
 */
int writeint (int *value, FILE *output)
{
    unsigned char c; /* character to write */
    c = (char) (*value & 0xff);
    if (! fwrite (&c, 1, 1, output))
	return 0;
    c = (char) (*value / 0x100);
    return fwrite (&c, 1, 1, output);
}

/**
 * Write an integer as a byte to an already open output file.
 * @param  value  A pointer to the integer variable to write.
 * @param  output The output file handle.
 * @return        1 if successful, 0 if not.
 */
int writebyte (int *value, FILE *output)
{
    unsigned char c; /* character to write */
    c = (char) (*value & 0xff);
    return fwrite (&c, 1, 1, output);
}

/**
 * Write a string to an already open output file.
 * @param  value  A pointer to the string buffer to write.
 * @param  output The output file handle.
 * @return        1 if successful, 0 if not.
 */
int writestring (char *value, FILE *output)
{
    unsigned char length; /* length byte */
    length = (char) strlen (value);
    if (! (fwrite (&length, 1, 1, output)))
	return 0;
    if (length)
	return fwrite (value, (int) length, 1, output);
    return 1;
}

/**
 * Read a 2-byte integer from an already open input file
 * and store it in an integer variable.
 * @param  value A pointer to the integer variable to store into.
 * @param  input The input file handle.
 * @return       1 if successful, 0 if not.
 */
int readint (int *value, FILE *input)
{
    unsigned char c; /* character to read */
    if (! (fread (&c, 1, 1, input)))
	return 0;
    *value = (int) c;
    if (! (fread (&c, 1, 1, input)))
	return 0;
    *value += ((int) c) << 8;
    return 1;
}

/**
 * Read a byte from an already open input file
 * and store it in an integer variable.
 * @param  value A pointer to the integer variable to store into.
 * @param  input The input file handle.
 * @return       1 if successful, 0 if not.
 */
int readbyte (int *value, FILE *input)
{
    unsigned char c; /* character to read */
    if (! (fread (&c, 1, 1, input)))
	return 0;
    *value = (int) c;
    return 1;
}

/**
 * Read a string from an already open input file
 * and store it in a string buffer.
 * @param  value A pointer to the string buffer to store into.
 * @param  input The input file handle.
 * @return       1 if successful, 0 if not.
 */
int readstring (char *value, FILE *input)
{
    unsigned char length; /* length byte */
    if (! (fread (&length, 1, 1, input)))
	return 0;
    if (length &&
	! (fread (value, length, 1, input)))
	return 0;
    value[(int) length] = '\0';
    return 1;
}

/**
 * Calculate the distance between two points.
 * @param firstx  First point X coordinate.
 * @param firsty  First point Y coordinate.
 * @param secondx Second point X coordinate.
 * @param secondy Second point Y coordinate.
 */
int distance (int firstx, int firsty, int secondx, int secondy)
{
    int distx, /* x distance between the two points */
	disty; /* y distance between the two points */
    distx = abs (firstx - secondx);
    disty = abs (firsty - secondy);
    return (distx > disty)
	? distx + disty / 2
	: disty + distx / 2;
}

