/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 17-Aug-2020.
 *
 * Skill handling module
 */

/*----------------------------------------------------------------------
 * Included Headers.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "skill.h"
#include "utils.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/**
 * @var skillnames
 * An array of skill names.
 */
static char *skillnames[] = {
    "-",
    "Medicine",
    "Stealth",
    "Hv Firearm",
    "Lt Firearm",
    "Blade",
    "Unarmed Cb"
};

/*----------------------------------------------------------------------
 * Public Methods.
 */

/**
 * Destroy a skill when no longer needed.
 * @param skill The skill to destroy.
 */
static void destroy (Skill *skill)
{
    if (skill)
	free (skill);
}

/**
 * Write the skill to an open file.
 * @param skill  The skill object.
 * @param output The output file handle.
 * @return       1 on success, 0 on failure.
 */
static int write (Skill *skill, FILE *output)
{
    if (! writebyte (&skill->id, output))
	return 0;
    if (! writebyte (&skill->level, output))
	return 0;
    return 1;
}

/**
 * Read the skill from an open file.
 * @param skill The skill object.
 * @param input The input file handle.
 * @return      1 on the success, 0 on failure.
 */
static int read (Skill *skill, FILE *input)
{
    if (! readbyte (&skill->id, input))
	return 0;
    if (! readbyte (&skill->level, input))
	return 0;
    return 1;
}

/**
 * Return the name of a skill.
 * @param skill The skill to name.
 */
static char *name (Skill *skill)
{
    return skillnames[skill->id];
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Constructor for a skill.
 * @return The new skill.
 */
Skill *new_Skill (void)
{
    Skill *skill; /* skill to return */

    /* allocate memory for the new skill */
    if (! (skill = malloc (sizeof (Skill))))
	return NULL;

    /* initialise methods */
    skill->destroy = destroy;
    skill->write = write;
    skill->read = read;
    skill->name = name;

    /* initialise attributes */
    skill->id = SKILL_NONE;
    skill->level = 0;

    /* return the new skill */
    return skill;
}
