/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 16-Aug-2020.
 *
 * Game Module.
 */

/*----------------------------------------------------------------------
 * Included headers.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "levelmap.h"
#include "sccc.h"
#include "config.h"
#include "game.h"
#include "unit.h"
#include "cabinet.h"
#include "stack.h"
#include "item.h"
#include "skill.h"
#include "fatal.h"
#include "utils.h"
#include "ai.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var titles Mission Titles. */
static char *titles[] = {
    "Random",
    "Assassination",
    "Annihilation",
    "Hostage",
    "Rescue",
    "Retrieval",
    "Gathering"
};

/** @var briefing Mission briefing templates. */
static char *briefings[] = {

    /* random mission will never be shown */
    "",

    /* Assassination */
    "A double agent named %s is in possession of sensitive military"
    " data which they intend to give to our enemies. Having tracked"
    " them down to their hideout, your team must infiltrate the area,"
    " locate the target and assassinate them. You are scored on the"
    " health of your team and the speed with which you complete the"
    " mission.",
    
    /* Annihilation */
    "We have located the base of a group of pirates who have been"
    " plundering the local space lanes. Your team must enter the base,"
    " and take out as many of the pirates as you can. You are scored on"
    " the speed with which you complete the mission, the number of"
    " enemies you kill, and on the health of your team upon completion"
    " of the mission.",

    /* Hostage */
    "Space terrorists have kinapped an important individual named %s."
    " We have tracked down the terrorists and found their base. Your"
    " team must enter the base, locate the hostage, free them from"
    " captivity and leave the base safely. Your performance will be"
    " scored on the speed at which you complete the mission, and the"
    " health of your team.",

    /* Rescue */
    "Pirates have captured the crew of a starship and are trying to"
    " sell them into slavery. Your team must infiltrate the pirate"
    " base, and find and rescue as many of the hostages as possible."
    " You are scored upon the number of hostages rescued, the health of"
    " your team, and the speed with which you complete the mission.",

    /* Retrieval */
    "Rebels are preparing a new attack upon us. Your team must enter"
    " their headquarters, find the data card on which the battle plans"
    " are stored, and retrieve it. You are scored upon the speed with"
    " which you complete the mission, and the health of your team once"
    " the mission is over.",

    /* Gathering */
    "Our enemies are rumoured to have devised a new range of weapons"
    " which would have devastating effect upon our colonies if we are"
    " not ready for them. Your team must enter the enemy weapons"
    " facility, and gather as many data cards as possible. You are"
    " scored on the number of data cards retrieved, the health of your"
    " team, and the time taken."
};

/**
 * @var entry
 * The square inside the entrance. The mission generation will ensure
 * that no enemy can see this square, and that no hostage or retrieval
 * mission objective can be seen from here.
 */
static int entry;

/**
 * @var missionturns
 * The number of turns each mission type is expected to take.
 */
static int missionturns[] = {
    1, /* shouldn't be used */
    5, /* assasinate */
    10, /* annihilate */
    5, /* hostage */
    10, /* rescue */
    5, /* retrieve */
    10 /* gather */
};

/**
 * @var scoremultipliers
 * Multipliers for each mission type for each score.
 */
static int scoremultipliers[7][5] = {
    {0, 0, 0, 0, 0},
    {2, 0, 0, 0, 3}, /* assassinate */
    {1, 3, 0, 0, 1}, /* annihilate */
    {2, 0, 0, 0, 3}, /* hostage */
    {1, 0, 3, 0, 1}, /* rescue */
    {2, 0, 0, 0, 3}, /* retrieve */
    {1, 0, 0, 3, 1} /* gathering */
};

/*----------------------------------------------------------------------
 * Level 4 Function Definitions.
 */

/**
 * Drop an item in a random cabinet.
 * @param game The game object.
 * @param item The item to drop.
 * @return     1 if a cabinet found.
 */
static int dropincabinet (Game *game, int item)
{
    Cabinet *cabinet; /* random cabinet */
    int c; /* general counter */
    if (! game->cabinetcount)
	return 0;
    if (game->type == MISSION_RETRIEVE && item == ITEM_CARD) {
	cabinet = game->cabinets[0];
	c = cabinet->x + game->map->width * cabinet->y;
	if (game->sightlines[c])
	    return 0;
    } else
	cabinet = game->cabinets[rand () % game->cabinetcount];
    for (c = 0; c < 12; ++c)
	if (! cabinet->items[c]) {
	    cabinet->items[c] = item;
	    return 1;
	}
    return 0;
}

/**
 * Drop an item in someone's inventory.
 * @param game    The game object.
 * @param item    The item to drop.
 * @param hostage 1 to prefer hostage, 0 for enemy only.
 */
static int droponunit (Game *game, int item, int hostage)
{
    Unit *unit; /* unit on which to deposit item */
    int c; /* inventory slot counter */

    /* decide the unit to give the item to */
    if (game->type == MISSION_RETRIEVE && item == ITEM_CARD)
	unit = game->enemies[0];
    else if (game->hostagecount && hostage)
	unit = game->hostages[rand () % game->hostagecount];
    else
	unit = game->enemies[rand () % game->enemycount];

    /* deposit the item */
    for (c = 4; c < 10; ++c)
	if (! unit->inventory[c]) {
	    unit->inventory[c] = item;
	    return 1;
	}

    /* 0 for failure */
    return 0;
}

/*----------------------------------------------------------------------
 * Level 3 Function Definitions.
 */

/**
 * See what, if anything, is in the adjacent square.
 * @param map     The map to look at.
 * @param pos     Position to look next to.
 * @param xoffset X offset to position.
 * @param yoffset Y offset to position.
 * @return        The block in the next square, or -1 if off the map.
 */
static int adjacentblock
(LevelMap *map, int pos, int xoffset, int yoffset)
{
    int x, /* x position */
	y; /* y position */

    /* work out position */
    x = (pos % map->width) + xoffset;
    y = (pos / map->width) + yoffset;

    /* return -1 if x/y is off the map */
    if (x < 0 || x >= map->width || y < 0 || y >= map->height)
	return -1;

    /* return what's in the map square */
    return map->blocks[x + map->width * y];
}

/**
 * Find a suitable location for a piece of furniture.
 * @param map The map to scan.
 * @param type The type of furniture to place.
 * @return A random suitable location.
 */
static int getfurniturelocation (LevelMap *map, int type)
{
    int pos, /* block position */
	squares, /* number of open squares surrounding */
	attempts = 255; /* number of attempts */

    do {

	/* make sure we're not in an infinite loop */
	if (! attempts--)
	    return 0;

	/* choose a random position and make sure it's open */
	pos = rand () % (map->width * map->height);
	if ((map->blocks[pos] & 0xf0) != LEVELMAP_OPEN)
	    continue;

	/* ensure furniture isn't in front of a door */
	if ((map->blocks[pos - map->width] & 0x30) == LEVELMAP_DOOR)
	    continue;
	if ((map->blocks[pos + map->width] & 0x30) == LEVELMAP_DOOR)
	    continue;
	if ((map->blocks[pos - 1] & 0x30) == LEVELMAP_DOOR)
	    continue;
	if ((map->blocks[pos + 1] & 0x30) == LEVELMAP_DOOR)
	    continue;
 
	/* ensure furniture won't block movement */
	squares = 0;
	if ((map->blocks[pos - map->width] & 0xf0) == LEVELMAP_OPEN)
	    ++squares;
	if ((map->blocks[pos + map->width] & 0xf0) == LEVELMAP_OPEN)
	    ++squares;
	if ((map->blocks[pos - 1] & 0xf0) == LEVELMAP_OPEN)
	    ++squares;
	if ((map->blocks[pos + 1] & 0xf0) == LEVELMAP_OPEN)
	    ++squares;
	if (squares < 3)
	    continue;

	/* ensure furniture is not at a corridor junction */
	squares = 0;
	if ((map->blocks[pos - map->width - 1] & 0x30) & LEVELMAP_WALL)
	    ++squares;
	if ((map->blocks[pos - map->width + 1] & 0x30) & LEVELMAP_WALL)
	    ++squares;
	if ((map->blocks[pos + map->width - 1] & 0x30) & LEVELMAP_WALL)
	    ++squares;
	if ((map->blocks[pos + map->width + 1] & 0x30) & LEVELMAP_WALL)
	    ++squares;
	if (squares == 4)
	    continue;

	/* ensure a cabinet is next to a wall */
	if (type != FURNITURE_CABINET ||
	    (map->blocks[pos - map->width] & 0xf0) == LEVELMAP_WALL ||
	    (map->blocks[pos + map->width] & 0xf0) == LEVELMAP_WALL ||
	    (map->blocks[pos - 1] & 0xf0) == LEVELMAP_WALL ||
	    (map->blocks[pos + 1] & 0xf0) == LEVELMAP_WALL)
	    break;

    } while (1);

    /* return the position */
    return pos;
}

/**
 * Add a cabinet to the level map.
 * @param game The game object.
 * @param pos  The position of the cabinet.
 */
static void addcabinet (Game *game, int pos)
{
    Cabinet *cabinet; /* pointer to the new cabinet */

    /* reserve memory for the new cabinet */
    if (game->cabinetcount)
	game->cabinets = realloc
	    (game->cabinets,
	     (game->cabinetcount + 1) * sizeof (Cabinet *));
    else
	game->cabinets = malloc (sizeof (Cabinet *));
    if (! game->cabinets)
	fatal_error (FATAL_MEMORY);

    /* create the new cabinet */
    if (! (cabinet = new_Cabinet ()))
	fatal_error (FATAL_MEMORY);
    cabinet->x = pos % game->map->width;
    cabinet->y = pos / game->map->width;
    game->cabinets[game->cabinetcount++] = cabinet;
}

/**
 * Place an enemy or hostage unit on the map.
 * @param game The game object.
 * @param unit The unit to place.
 * @return     1 if successful, 0 if not.
 */
static int placeunit (Game *game, Unit *unit, int id)
{
    int pos, /* block position */
	attempts = 255; /* attempts to place the unit */

    /* keep looking for a suitable square */
    do {

	/* make sure we're not in an infinite loop */
	if (! attempts--)
	    return 0;

	/* choose a random position and make sure it's open */
	pos = rand () % (game->map->width * game->map->height);
	if ((game->map->blocks[pos] & 0xf0) != LEVELMAP_OPEN)
	    continue;

	/* ensure the location isn't visible from the entrance */
	if (game->sightlines[pos])
	    continue;

	/* accept the unit position */
	break;

    } while (1);

    /* place the unit */
    unit->x = pos % game->map->width;
    unit->y = pos / game->map->width;
    game->map->blocks[pos] |= MISSION_UNIT;
    game->unitcache[pos]
	= id
	+ ((unit->class == UNIT_ENEMY)
	   ? MISSION_CACHE_ENEMY
	   : MISSION_CACHE_HOSTAGE);
    return 1;
}

/**
 * Generate a medikit.
 * @param game The game object.
 * @return     1 if successful, 0 otherwise.
 */
static int generatemedikit (Game *game)
{
    /* put a medikit in a cabinet */
    if (game->cabinetcount) {
	dropincabinet (game, ITEM_MEDIKIT);
	return 1;
    }

    /* put the medikit in someone's inventory */
    return droponunit (game, ITEM_MEDIKIT, 1);
}

/**
 * Generate the data cards.
 * @param game The game object.
 * @return     1 if successful, 0 otherwise.
 */
static int generatecards (Game *game)
{
    int c; /* general counter */

    /* how many cards on the level? */
    if (game->type == MISSION_RETRIEVE)
	game->cardcount = 1;
    else if (game->type == MISSION_GATHER)
	game->cardcount = throwdice (2, 0, 0, 0);
    else
	return 1;

    /* place the cards */
    c = 0;
    while (c < game->cardcount)
	if (throwdice (1, 0, 0, 0) <= 4 &&
	    dropincabinet (game, ITEM_CARD))
	    ++c;
	else if (droponunit (game, ITEM_CARD, 0))
	    ++c;

    /* cards generated */
    return 1;
}

/**
 * Generate a blade.
 * @param game The game object.
 * @return     1 if successful, 0 otherwise.
 */
static int generateblade (Game *game)
{
    /* put a blade in a cabinet */
    if (game->cabinetcount) {
	dropincabinet (game, ITEM_BLADE);
	return 1;
    }

    /* put the blade in someone's inventory */
    return droponunit (game, ITEM_BLADE, 0);
}

/**
 * Find the furthest unit from the entrance.
 * @param game The game object.
 * @return     A pointer to the furthest unit.
 */
static Unit *furthestunit (Game *game)
{
    int maxdist, /* longest distance found */
	dist, /* unit distance from the entrance */
	c; /* counter */
    Unit *furthest = NULL; /* furthest unit found */

    /* initialise */
    maxdist = 0;

    /* look through the enemies */
    for (c = 0; c < game->enemycount; ++c) {
	dist = distance
	    (game->entrance % game->map->width,
	     game->entrance / game->map->width,
	     game->enemies[c]->x,
	     game->enemies[c]->y);
	if (dist > maxdist) {
	    maxdist = dist;
	    furthest = game->enemies[c];
	}
    }

    /* look through the hostages */
    for (c = 0; c < game->hostagecount; ++c) {
	dist = distance
	    (game->entrance % game->map->width,
	     game->entrance / game->map->width,
	     game->hostages[c]->x,
	     game->hostages[c]->y);
	if (dist > maxdist) {
	    maxdist = dist;
	    furthest = game->hostages[c];
	}
    }

    /* return the furthest unit found */
    return furthest;
}

/**
 * Swap the locations of two units.
 * @param game   The game object.
 * @param first  The first unit.
 * @param second The second unit.
 */
static void swapunitlocations (Game *game, Unit *first, Unit *second)
{
    int swap; /* variable to hold swap value */

    /* swap the locations in the unit objects */
    swap = first->x;
    first->x = second->x;
    second->x = swap;
    swap = first->y;
    first->y = second->y;
    second->y = swap;

    /* update the unit cache */
    game->unitcache[first->x + game->map->width * first->y]
	= first->id
	+ ((first->class == UNIT_ENEMY)
	   ? MISSION_CACHE_ENEMY
	   : MISSION_CACHE_HOSTAGE);
    game->unitcache[second->x + game->map->width * second->y]
	= second->id
	+ ((second->class == UNIT_ENEMY)
	   ? MISSION_CACHE_ENEMY
	   : MISSION_CACHE_HOSTAGE);
}

/**
 * Find the furthest cabinet from the entrance.
 * @param game The game object.
 * @return     A pointer to the furthest unit.
 */
static Cabinet *furthestcabinet (Game *game)
{
    int maxdist, /* longest distance found */
	dist, /* unit distance from the entrance */
	c; /* counter */
    Cabinet *furthest = NULL; /* furthest unit found */

    /* initialise */
    maxdist = 0;

    /* look through the cabinets */
    for (c = 0; c < game->cabinetcount; ++c) {
	dist = distance
	    (game->entrance % game->map->width,
	     game->entrance / game->map->width,
	     game->cabinets[c]->x,
	     game->cabinets[c]->y);
	if (dist > maxdist) {
	    maxdist = dist;
	    furthest = game->cabinets[c];
	}
    }

    /* return the furthest cabinet found */
    return furthest;
}

/**
 * Swap the locations of two cabinets.
 * @param first  The first cabinet.
 * @param second The second cabinet.
 */
static void swapcabinetlocations (Cabinet *first, Cabinet *second)
{
    int swap; /* variable to hold swap value */
    swap = first->x;
    first->x = second->x;
    second->x = swap;
    swap = first->y;
    first->y = second->y;
    second->y = swap;
}

/*----------------------------------------------------------------------
 * Level 2 Function Definitions.
 */

/**
 * Calculate sight of a particular location.
 * @param game    The game object.
 * @param loc     The location to check.
 * @param visible A pointer to the visibility flag.
 * @param checked A pointer to the checked flag.
 */
static void calcsightloc
(Game *game, int loc, int *visible, char *checked)
{
    int block; /* block at the location */
    block = game->map->blocks[loc] & 0x30;
    if (! *visible);
    else if (block == LEVELMAP_DOOR) {
	game->sightlines[loc] |= *visible;
	*visible = 0;
    } else if (block == LEVELMAP_WALL)
	*visible = 0;
    else
	game->sightlines[loc] |= *visible;
    *checked = 1;
}

/*----------------------------------------------------------------------
 * Private Level 1 Functions.
 */

/**
 * Clear all mission data ready for generation.
 * @param game The game object.
 */
static void clearmap (Game *game)
{
    LevelMap *map; /* pointer to the map */
    Unit *unit; /* pointer to a unit */
    AI *ai; /* pointer to a unit AI */
    Cabinet *cabinet; /* pointer to a cabinet */
    ItemStack *stack; /* pointer to an item stack */

    /* initialise mission data */
    game->score = 0;
    game->highest = 0;
    game->turns = 0;

    /* destroy the map and entrance data */
    if ((map = game->map)) {
	game->map->destroy (game->map);
	game->map = NULL;
    }
    game->entrance = 0;

    /* destroy the team */
    while (game->unitcount)
	if ((unit
	     = game->units[--game->unitcount]))
	    unit->destroy (unit);
    if (game->units) {
	free (game->units);
	game->units = NULL;
    }

    /* destroy the enemies */
    while (game->enemycount) {
	if ((unit = game->enemies[--game->enemycount]))
	    unit->destroy (unit);
	if ((ai = game->ai[game->enemycount]))
	    ai->destroy (ai);
    }
    if (game->enemies) {
	free (game->enemies);
	game->enemies = NULL;
    }
    if (game->ai) {
	free (game->ai);
	game->ai = NULL;
    }

    /* destroy the hostages */
    while (game->hostagecount)
	if ((unit
	     = game->hostages[--game->hostagecount]))
	    unit->destroy (unit);
    if (game->hostages) {
	free (game->hostages);
	game->hostages = NULL;
    }

    /* destroy the cabinets */
    while (game->cabinetcount)
	if ((cabinet = game->cabinets[--game->cabinetcount]))
	    cabinet->destroy (cabinet);
    if (game->cabinets) {
	free (game->cabinets);
	game->cabinets = NULL;
    }

    /* destroy the item stacks */
    while (game->stackcount)
	if ((stack = game->stacks[--game->stackcount]))
	    stack->destroy (stack);
    if (game->stacks) {
	free (game->stacks);
	game->stacks = NULL;
    }

    /* reset the card count */
    game->cardcount = 0;

    /* destroy any calculated sight lines */
    if (game->sightlines) {
	free (game->sightlines);
	game->sightlines = NULL;
    }

    /* destroy the caches */
    if (game->unitcache) {
	free (game->unitcache);
	free (game->stackcache);
	game->unitcache = game->stackcache = NULL;
    }
}

/**
 * Initialise the stack cache after loading a game.
 * @param game The game object.
 */
static void initstackcache (Game *game)
{
    int c, /* counter */
	loc; /* location of item stack */
    ItemStack *stack; /* unit pointer */

    /* clear unit cache */
    if (game->stackcache)
	free (game->stackcache);
    if (! (game->stackcache = calloc
	   (game->map->width * game->map->height, 1)))
	fatal_error (FATAL_MEMORY);

    /* add item stacks */
    for (c = 0; c < game->stackcount; ++c)
	if ((stack = game->stacks[c])) {
	    loc = stack->x + game->map->width * stack->y;
	    game->stackcache[loc] = stack->top (stack);
	}
}

/**
 * Generate the mission type.
 * @param game The game object.
 * @returns    1 if successful, 0 on failure.
 */
static int generatemissiontype (Game *game)
{
    /* reject invalid inputs */
    if (game->type < MISSION_RANDOM)
	return 0;
    else if (game->type >= MISSION_TYPES)
	return 0;

    /* choose a random mission if desired */
    if (game->type == MISSION_RANDOM)
	game->type = 1 + rand () % 6;

    /* return with success */
    return 1;
}

/**
 * Validate the mission level.
 * @param game The game object.
 * @returns    1 if successful, 0 on failure.
 */
static int validatemissionlevel (Game *game)
{
    if (game->level < MISSION_EASY)
	return 0;
    if (game->level > MISSION_HARD)
	return 0;
    return 1;
}

/**
 * Generate the team of four units.
 * @param game   The game object.
 * @param config The configuration.
 * @return       1 if successful, otherwise 0.
 */
static int generateteam (Game *game, Config *config)
{
    int u; /* unit counter */
    Unit *unit; /* newly-generated unit */

    /* reserve memory for units */
    game->units = malloc (4 * sizeof (Unit *));
    game->unitcount = 4;

    /* generate each unit */
    for (u = 0; u < game->unitcount; ++u) {
	if (! (unit = game->units[u] = new_Unit ()))
	    fatal_error (FATAL_MEMORY);
	unit->id = u;
	unit->level = UNIT_HARD;
	unit->class = config->classes[u];
	if (unit->class == UNIT_NONE)
	    unit->class = UNIT_MEDIC + (rand () % 4);
	if (! unit->generate (unit))
	    return 0;
    }

    /* set current unit and return */
    game->unit = game->units[0];
    return 1;
}

/**
 * Choose the mission location entry point.
 * @param map The mission map.
 * @return    1 on success.
 */
static int generateentrance (Game *game)
{
    int perimeter, /* size of parimeter */
	location, /* location on the perimeter */
	width, /* width of level map */
	height, /* height of level map */
	x, /* x coordinate */
	y, /* y coordinate */
	block; /* contents of map block */

    /* ascertain map geometry */
    width = game->map->width;
    height = game->map->height;
    perimeter = 2 * (width + height);

    /* choose random location */
    do {
	location = rand () % perimeter;
	if (location < width) {
	    x = location;
	    y = 0;
	} else if (location < width + height) {
	    x = width - 1;
	    y = location - width;
	} else if (location < 2 * width + height) {
	    x = location - width - height;
	    y = height - 1;
	} else {
	    x = 0;
	    y = location - 2 * width - height;
	}
	block = game->map->blocks[x + width * y];
    } while ((block & 0xf0) != LEVELMAP_WALL
	     || ((block & 0xf) != 0x5
		 && (block & 0xf) != 0xa));

    /* insert a door */
    game->entrance = x + width * y;
    block &= 0xf;
    block |= LEVELMAP_DOOR;
    game->map->blocks[game->entrance] = block;

    /* work out what square the units will enter by */
    if ((adjacentblock (game->map, game->entrance, 1, 0) & 0xf0)
	== LEVELMAP_OPEN)
	entry = game->entrance + 1;
    else if ((adjacentblock (game->map, game->entrance, -1, 0)
	      & 0xf0)
	== LEVELMAP_OPEN)
	entry = game->entrance - 1;
    else if ((adjacentblock (game->map, game->entrance, 0, 1)
	      & 0xf0)
	== LEVELMAP_OPEN)
	entry = game->entrance + game->map->width;
    else if ((adjacentblock (game->map, game->entrance, 0, -1)
	      & 0xf0)
	== LEVELMAP_OPEN)
	entry = game->entrance - game->map->width;

    /* work out lines of sight from the entry squares */
    if (game->sightlines) {
	free (game->sightlines);
	game->sightlines = NULL;
    }
    game->calcsightlines (game, game->entrance, 1);
    game->calcsightlines (game, entry, 2);

    /* success! */
    return 1;
}

/**
 * Add furniture to the mission location.
 * @param game The game object.
 * @return     1 on success, 0 on failure.
 */
static int generatefurniture (Game *game)
{
    int items, /* the number of furniture items */
	i, /* item counter */
	type, /* furniture type */
	pos; /* furniture position */

    /* how many items of furniture? */
    items = throwdice (3, 0, 0, 0);

    /* generate the items of furniture */
    for (i = 0; i < items; ++i) {
	type = rand () % FURNITURE_LAST;
	if (! (pos = getfurniturelocation (game->map, type)))
	    return 0;
	game->map->blocks[pos] = MISSION_FURNITURE + type;
	if (type == FURNITURE_CABINET)
	    addcabinet (game, pos);
    }
    return 1;
}

/**
 * Add enemies to the mission location.
 * @param game The game object.
 * @returns    1 on success, 0 on failure.
 */
static int generateenemies (Game *game)
{
    int qty, /* number of enemies */
	count; /* enemy counter */
    Unit *unit; /* generated unit */

    /* clear out any previous enemies */
    if (game->enemycount) {
	for (count = 0; count < game->enemycount; ++count) {
	    game->enemies[count]->destroy (game->enemies[count]);
	    game->ai[count]->destroy (game->ai[count]);
	}
	free (game->enemies);
    }

    /* how many enemies? */
    switch (game->level) {
    case MISSION_EASY:
	qty = 6 + lowestdice (2);
	break;
    case MISSION_FAIR:
	qty = throwdice (1, 6, 0, 0);
	break;
    case MISSION_HARD:
	qty = 6 + highestdice (2);
	break;
    default:
	break;
    }
    game->enemycount = qty;
    game->enemies = calloc (qty, sizeof (Unit *));
    if (! game->enemies)
	fatal_error (FATAL_MEMORY);
    game->ai = calloc (qty, sizeof (AI *));

    /* create the enemy cache */
    game->unitcache = calloc
	(game->map->width * game->map->height, 1);
    
    /* generate each enemy */
    for (count = 0; count < qty; ++count) {

	/* reserve memory for the unit and its AI */
	if (! (unit = new_Unit ()))
	    fatal_error (FATAL_MEMORY);
	if (! (game->ai[count] = new_AI ()))
	    fatal_error (FATAL_MEMORY);

	/* generate and place the unit */
	unit->id = count;
	unit->class = UNIT_ENEMY;
	unit->level = game->level;
	unit->generate (unit);
	if (! placeunit (game, unit, count))
	    return 0;
	game->enemies[count] = unit;

	/* initialise the AI with a random order */
	game->ai[count]->unit = unit;
	game->ai[count]->order = rand () % 3;
	if (game->ai[count]->order == AI_ORDER_PATROL)
	    game->ai[count]->newpatrollocation (game->ai[count]);
	else {
	    game->ai[count]->orderx = unit->x;
	    game->ai[count]->ordery = unit->y;
	}

    }

    /* return */
    return 1;
}

/**
 * Add hostages to the mission location.
 * @param game The game object.
 * @returns    1 on success, 0 on failure.
 */
static int generatehostages (Game *game)
{
    int count; /* hostage counter */
    Unit *unit; /* generated unit */

    /* clear out any previous hostages */
    if (game->hostagecount) {
	for (count = 0; count < game->hostagecount; ++count)
	    game->hostages[count]->destroy
		(game->hostages[count]);
	free (game->hostages);
	game->hostages = NULL;
    }

    /* how many hostages? */
    if (game->type == MISSION_HOSTAGE)
	game->hostagecount = 1;
    else if (game->type == MISSION_RESCUE)
	game->hostagecount = throwdice (2, 0, 0, 0);
    if (! game->hostagecount)
	return 1;
    else if (game->hostagecount + game->enemycount > 16)
	game->hostagecount = 16 - game->enemycount;

    /* reserve memory for hostages */
    game->hostages = calloc (game->hostagecount, sizeof (Unit *));
    if (! game->hostages)
	fatal_error (FATAL_MEMORY);

    /* generate each hostage */
    for (count = 0; count < game->hostagecount; ++count) {
	if (! (unit = new_Unit ()))
	    fatal_error (FATAL_MEMORY);
	unit->id = count;
	unit->class = UNIT_HOSTAGE;
	unit->level = MISSION_EASY;
	unit->hostage = 1;
	unit->generate (unit);
	if (! placeunit (game, unit, count))
	    return 0;
	game->hostages[count] = unit;
    }

    /* return */
    return 1;
}

/**
 * Add items to the mission location.
 * @param game The game object.
 * @returns    1 on success, 0 on failure.
 */
static int generateitems (Game *game)
{
    /* place the items */
    if (! generatemedikit (game))
	return 0;
    if (! generatecards (game))
	return 0;
    if (! generateblade (game))
	return 0;

    /* create the stack cache */
    game->stackcache = calloc
	(game->map->width * game->map->height, 1);

    /* success! */
    return 1;
}

/**
 * Ensure that target is as far from the entrance as possible.
 * @param game The game object.
 */
static void generatetargetposition (Game *game)
{
    Unit *unit; /* furthest unit */
    Cabinet *cabinet; /* furthest cabinet */
    if (game->type == MISSION_ASSASSINATE) {
	unit = furthestunit (game);
	if (unit != game->enemies[0])
	    swapunitlocations (game, game->enemies[0], unit);
    } else if (game->type == MISSION_HOSTAGE) {
	unit = furthestunit (game);
	if (unit != game->hostages[0])
	    swapunitlocations (game, game->hostages[0], unit);
    } else if (game->type == MISSION_RETRIEVE) {
	unit = furthestunit (game);
	if (unit != game->enemies[0])
	    swapunitlocations (game, unit, game->enemies[0]);
	cabinet = furthestcabinet (game);
	if (game->cabinets && cabinet != game->cabinets[0])
	    swapcabinetlocations (cabinet, game->cabinets[0]);
    }
}

/**
 * Calculate a single sightline.
 * @param game   The game object.
 * @param check  The array of locations checked.
 * @param origin The origin location.
 * @param dest   The destination location.
 * @param mask   The bitmask to apply.
 */
static void calcsightline
(Game *game, char *checked, int origin, int dest, int mask)
{
    int x0, y0, /* start coordinates */
	x1, y1, /* end coordinates */
	x, y, /* current coordinates */
	dx, dy, /* difference between start and end coordinates */
	sx, sy, /* step in X and Y directions */
	err, e2, /* error */
	loc, /* current location as an integer */
	visible; /* visibility mask */

    /* initialise */
    x0 = origin % game->map->width;
    y0 = origin / game->map->width;
    x = x1 = dest % game->map->width;
    y = y1 = dest / game->map->width;
    dx = abs (x1 - x0);
    dy = -abs (y1 - y0);
    visible = mask;

    /* check for destination to origin to eliminate squares already 
       checked */
    sx = x1 < x0 ? 1 : -1;
    sy = y1 < y0 ? 1 : -1;
    err = dx + dy;
    while (x != x0 || y != y0) {

	/* check the current location */
	loc = x + game->map->width * y;
	if (checked[loc]) {
	    visible = game->sightlines[loc] & mask;
	    break;
	}

	/* next square */
        e2 = 2 * err;
        if (e2 >= dy) {
            if (x == x0)
		break;
            err = err + dy;
            x = x + sx;
	}
        if (e2 <= dx) {
            if (y == y0)
		break;
            err = err + dx;
            y = y + sy;
	}
    }

    /* follow unchecked squares to destination */
    sx = x0 < x1 ? 1 : -1;
    sy = y0 < y1 ? 1 : -1;
    err = dx + dy;
    while (x != x1 || y != y1) {

	/* check the current location */
	if (x != x0 || y != y0) {
	    loc = x + game->map->width * y;
	    if (loc != origin)
		calcsightloc (game, loc, &visible, &checked[loc]);
	}

	/* next square */
        e2 = 2 * err;
        if (e2 >= dy) {
            if (x == x1)
		break;
            err = err + dy;
            x = x + sx;
	}
        if (e2 <= dx) {
            if (y == y1)
		break;
            err = err + dx;
            y = y + sy;
	}
    }

    /* mark final location */
    game->sightlines[dest] |= visible;
    checked[dest] = 1;
}

/**
 * Ascertain whether the assassination target is dead.
 * @param game The game object.
 * @return     1 if the target is dead, 0 if not.
 */
static int ascertainassassination (Game *game)
{
    return
	game->enemies[0] &&
	game->enemies[0]->health == 0;
}

/**
 * Ascertain whether there were any enemies killed.
 * @param game The game object.
 * @return     1 if there was a kill, 0 if not.
 */
static int ascertainannihilation (Game *game)
{
    int c; /* counter */
    for (c = 0; c < game->enemycount; ++c)
	if (game->enemies[c] &&
	    game->enemies[c]->health == 0)
	    return 1;
    return 0;
}

/**
 * Ascertain if any hostages have been freed, for checking victory on
 * both Hostage and Rescue missions.
 * @param game The game object.
 * @return     1 if a hostage has been freed, 0 if not.
 */
static int ascertainhostagefreed (Game *game)
{
    int c; /* counter */
    for (c = 4; c < game->unitcount; ++c)
	if (game->units[c] &&
	    game->units[c]->health &&
	    game->units[c]->x == 0 &&
	    game->units[c]->y == 0)
	    return 1;
    return 0;
}

/**
 * Ascertain if any cards have been collected, for checking victory on
 * both Retrieval and Gathering missions.
 * @param game The game object.
 * @return     1 if a hostage has been freed, 0 if not.
 */
static int ascertaincardretrieved (Game *game)
{
    int u, /* unit counter */
	i; /* item counter */
    for (u = 0; u < game->unitcount; ++u)
	if (game->units[u] &&
	    game->units[u]->health &&
	    game->units[u]->x == 0 &&
	    game->units[u]->y == 0)
	    for (i = 0; i < 10; ++i)
		if (game->units[u]->inventory[i] == ITEM_CARD)
		    return 1;
    return 0;
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Destroy a game when it is finished with.
 * @param game The game object.
 */
static void destroy (Game *game)
{
    if (game) {
	clearmap (game);
	free (game);
    }		    
}

/**
 * Save a game to a file.
 * @param game The game object.
 * @returns    1 on success, 0 on failure.
 */
static int save (Game *game)
{
    FILE *output; /* input file handle */
    Unit *unit; /* convenience pointer to unit */
    int c; /* general counter */

    /* attempt to open the file and write the header */
    if (! (output = fopen ("SCCC.GAM", "wb")))
	return 0;
    if (! fwrite ("SCC003G", 8, 1, output)) {
	fclose (output);
	return 0;
    }

    /* write basic mission information */
    if (! writebyte (&game->state, output)) {
	fclose (output);
	return 0;
    }
    if (! writebyte (&game->score, output)) {
	fclose (output);
	return 0;
    }
    if (! writebyte (&game->highest, output)) {
	fclose (output);
	return 0;
    }
    if (! writeint (&game->turns, output)) {
	fclose (output);
	return 0;
    }
    if (! writebyte (&game->type, output)) {
	fclose (output);
	return 0;
    }
    if (! writebyte (&game->level, output)) {
	fclose (output);
	return 0;
    }

    /* write map information */
    if (! game->map->write (game->map, output)) {
	fclose (output);
	return 0;
    }
    if (! writebyte (&game->entrance, output)) {
	fclose (output);
	return 0;
    }

    /* attempt to write the units */
    if (! writebyte (&game->unitcount, output)) {
	fclose (output);
	return 0;
    }
    for (c = 0; c < game->unitcount; ++c) {
	unit = game->units[c];
	if (! unit->write (unit, output)) {
	    fclose (output);
	    return 0;
	}
    }

    /* write enemy information */
    if (! writebyte (&game->enemycount, output)) {
	fclose (output);
	return 0;
    }
    if (game->enemycount)
	for (c = 0; c < game->enemycount; ++c) {
	    if (! game->enemies[c]->write (game->enemies[c], output)) {
		fclose (output);
		return 0;
	    }
	    if (! game->ai[c]->write (game->ai[c], output)) {
		fclose (output);
		return 0;
	    }
	}

    /* write hostage information */
    if (! writebyte (&game->hostagecount, output)) {
	fclose (output);
	return 0;
    }
    for (c = 0; c < game->hostagecount; ++c)
	if (game->hostages[c]) {
	    if (! writebyte (&c, output)) {
		fclose (output);
		return 0;
	    }
	    if (! game->hostages[c]->write
		(game->hostages[c], output))
	    {
		fclose (output);
		return 0;
	    }
	}
    if (! writebyte (&c, output)) {
	fclose (output);
	return 0;
    }

    /* write cabinet information */
    if (! writebyte (&game->cabinetcount, output)) {
	fclose (output);
	return 0;
    }
    if (game->cabinetcount)
	for (c = 0; c < game->cabinetcount; ++c)
	    if (! game->cabinets[c]->write
		(game->cabinets[c], output))
	    {
		fclose (output);
		return 0;
	    }

    /* write item stack information */
    if (! writebyte (&game->stackcount, output)) {
	fclose (output);
	return 0;
    }
    for (c = 0; c < game->stackcount; ++c)
	if (game->stacks[c]) {
	    if (! writebyte (&c, output)) {
		fclose (output);
		return 0;
	    }
	    if (! game->stacks[c]->write (game->stacks[c], output)) {
		fclose (output);
		return 0;
	    }
	}
    if (! writebyte (&c, output)) {
	fclose (output);
	return 0;
    }

    /* write card count */
    if (! writebyte (&game->cardcount, output)) {
	fclose (output);
	return 0;
    }

    /* attempt to write visibility & opportunity attack data */
    if (! writeint (&game->visibility, output)) {
	fclose (output);
	return 0;
    }
    if (! writeint (&game->attack, output)) {
	fclose (output);
	return 0;
    }

    /* attempt to write the unit cache */
    if (! fwrite
	(game->unitcache,
	 1,
	 game->map->width * game->map->height,
	 output))
    {
	fclose (output);
	return 0;
    }

    /* all successful */
    fclose (output);
    return 1;
}

/**
 * Load a game from a file.
 * @param game The game object.
 * @returns    1 on success, 0 on failure.
 */
static int load (Game *game)
{
    FILE *input; /* input file handle */
    char header[8]; /* the header to read and check */
    Unit *unit; /* convenience pointer to unit */
    int c; /* counter */

    /* clear the map and data */
    clearmap (game);

    /* attempt to open the file and read the header */
    if (! (input = fopen ("SCCC.GAM", "rb")))
	return 0;
    if (! fread (header, 8, 1, input)) {
	fclose (input);
	return 0;
    }
    if (strcmp (header, "SCC003G")) {
	fclose (input);
	return 0;
    }

    /* read basic mission information */
    if (! readbyte (&game->state, input)) {
	fclose (input);
	return 0;
    }
    if (! readbyte (&game->score, input)) {
	fclose (input);
	return 0;
    }
    if (! readbyte (&game->highest, input)) {
	fclose (input);
	return 0;
    }
    if (! readint (&game->turns, input)) {
	fclose (input);
	return 0;
    }
    if (! readbyte (&game->type, input)) {
	fclose (input);
	return 0;
    }
    if (! readbyte (&game->level, input)) {
	fclose (input);
	return 0;
    }

    /* read map information */
    if (! (game->map = new_LevelMap ()))
	fatal_error (FATAL_MEMORY);
    if (! game->map->read (game->map, input)) {
	fclose (input);
	return 0;
    }
    if (! readbyte (&game->entrance, input)) {
	fclose (input);
	return 0;
    }

    /* attempt to read the units */
    if (! readbyte (&game->unitcount, input)) {
	fclose (input);
	return 0;
    }
    if (! (game->units = calloc (game->unitcount, sizeof (Unit *))))
	fatal_error (FATAL_MEMORY);
    for (c = 0; c < game->unitcount; ++c) {
	unit = game->units[c] = new_Unit ();
	if (! unit->read (unit, input)) {
	    fclose (input);
	    return 0;
	}
    }

    /* read enemy information */
    if (! readbyte (&game->enemycount, input)) {
	fclose (input);
	return 0;
    }
    if (game->enemycount) {
	if (! (game->enemies
	       = malloc (game->enemycount * sizeof (Unit *))))
	    fatal_error (FATAL_MEMORY);
	if (! (game->ai
	       = malloc (game->enemycount * sizeof (AI *))))
	    fatal_error (FATAL_MEMORY);
	for (c = 0; c < game->enemycount; ++c) {
	    if (! (game->enemies[c] = new_Unit ()))
		fatal_error (FATAL_MEMORY);
	    if (! game->enemies[c]->read (game->enemies[c], input)) {
		fclose (input);
		return 0;
	    }
	    if (! (game->ai[c] = new_AI ()))
		fatal_error (FATAL_MEMORY);
	    if (! game->ai[c]->read (game->ai[c], input)) {
		fclose (input);
		return 0;
	    }
	    game->ai[c]->unit = game->enemies[c];
	}
    }

    /* read hostage information */
    if (! readbyte (&game->hostagecount, input)) {
	fclose (input);
	return 0;
    }
    if (game->hostagecount &&
	! (game->hostages
	   = calloc (game->hostagecount, sizeof (Unit *))))
	fatal_error (FATAL_MEMORY);
    do {
	if (! readbyte (&c, input)) {
	    fclose (input);
	    return 0;
	}
	if (c < game->hostagecount) {
	    if (! (game->hostages[c] = new_Unit ()))
		fatal_error (FATAL_MEMORY);
	    if (! game->hostages[c]->read
		(game->hostages[c], input))
	    {
		fclose (input);
		return 0;
	    }
	}
    } while (c < game->hostagecount);

    /* read cabinet information */
    if (! readbyte (&game->cabinetcount, input)) {
	fclose (input);
	return 0;
    }
    if (game->cabinetcount) {
	if (! (game->cabinets
	       = malloc (game->cabinetcount * sizeof (Unit *))))
	    fatal_error (FATAL_MEMORY);
	for (c = 0; c < game->cabinetcount; ++c) {
	    if (! (game->cabinets[c] = new_Cabinet ()))
		fatal_error (FATAL_MEMORY);
	    if (! game->cabinets[c]->read (game->cabinets[c], input)) {
		fclose (input);
		return 0;
	    }
	}
    }

    /* read item stack information */
    if (! readbyte (&game->stackcount, input)) {
	fclose (input);
	return 0;
    }
    if (game->stackcount &&
	! (game->stacks
	   = calloc (game->stackcount, sizeof (Unit *))))
	fatal_error (FATAL_MEMORY);
    do {
	if (! readbyte (&c, input)) {
	    fclose (input);
	    return 0;
	}
	if (c < game->stackcount) {
	    if (! (game->stacks[c] = new_ItemStack ()))
		fatal_error (FATAL_MEMORY);
	    if (! game->stacks[c]->read
		(game->stacks[c], input))
	    {
		fclose (input);
		return 0;
	    }
	}
    } while (c < game->stackcount);

    /* read card count */
    if (! readbyte (&game->cardcount, input)) {
	fclose (input);
	return 0;
    }

    /* attempt to write visibility & opportunity attack data */
    if (! readint (&game->visibility, input)) {
	fclose (input);
	return 0;
    }
    if (! readint (&game->attack, input)) {
	fclose (input);
	return 0;
    }

    /* attempt to read unit cache */
    if (! (game->unitcache
	   = malloc (game->map->width * game->map->height)))
	fatal_error (FATAL_MEMORY);
    if (! fread
	(game->unitcache,
	 1,
	 game->map->width * game->map->height,
	 input))
    {
	fclose (input);
	return 0;
    }

    /* calculate sight lines, and refresh cache */
    game->calcunitsightlines (game, generateprogress);
    initstackcache (game);

    /* all successful */
    fclose (input);
    return 1;
}

/**
 * Generate a new game at random.
 * @param game   The game object.
 * @param config The configuration.
 * @returns      1 on success, 0 on failure.
 */
static int generate (Game *game, Config *config)
{
    /* set the game parameters from the configuration */
    game->level = config->level;
    game->type = config->type;

    /* generate the mission type */
    if (! (generatemissiontype (game)))
	return 0;

    /* validate the level */
    if (! (validatemissionlevel (game)))
	return 0;

    /* generate the mission */
    do {

	/* clear out previous attempts at generation */
	clearmap (game);

	/* generate the team */
	if (! generateteam (game, config))
	    return 0;

	/* generate the standard map */
	if (! (game->map = new_LevelMap ()))
	    fatal_error (FATAL_MEMORY);
	if (! game->map->generate (game->map))
	    continue;
	if (! game->map->openvoids (game->map))
	    continue;
	if (! game->map->connecttextures (game->map))
	    continue;

	/* add the game-specific content to the map */
	if (! generateentrance (game))
	    continue;
	if (! generatefurniture (game))
	    continue;
	if (! generateenemies (game))
	    continue;
	if (! generatehostages (game))
	    continue;
	if (! generateitems (game))
	    continue;
	generatetargetposition (game);

	/* success: break out of the loop */
	break;

    } while (1);

    /* set the gaem state and return */
    game->state = STATE_MAP;
    return 1;
}

/**
 * Make the next available player unit the current one.
 * @param game The game object.
 * @return     1 if another unit found, 0 otherwise.
 */
static int nextunit (Game *game)
{
    Unit *curr = NULL, /* pointer to current unit */
	*next = NULL; /* pointer to potential next unit */
    int currid, /* id of current unit */
	c; /* counter */

    /* discern the current unit */
    for (c = 0; c < game->unitcount; ++c)
	if (game->units[c] == game->unit)
	    curr = game->units[currid = c];

    /* if there is no current unit, select the last one */
    if (! curr)
	curr = game->units[currid = 3];

    /* find the next unit */
    for (c = 1; c < game->unitcount; ++c)
	if ((next = game->units[(currid + c) % game->unitcount]) &&
	    ! next->unconscious &&
	    next->health &&
	    next->action)
	{
	    game->unit = next;
	    return 1;
	}

    /* no next unit found */
    return 0;
}

/**
 * Prepare a new turn.
 * @param game The game object.
 */
static void initturn (Game *game)
{
    int c, /* unit counter */
	unitcount; /* count of units */
    Unit **units; /* pointer to units */

    /* determine whose turn it is */
    if (game->state == STATE_MAP) {
	unitcount = game->unitcount;
	units = game->units;
    } else {
	unitcount = game->enemycount;
	units = game->enemies;
    }

    /* reset each unit */
    for (c = 0; c < unitcount; ++c)
	if (units[c])
	    units[c]->reset (units[c]);

    /* set the opportunity attack bits */
    game->attack = game->activeunits (game, 6);
    game->conscious = game->activeunits (game, 0);
}

/**
 * Check for line of sight between two points.
 * @param game   The game object.
 * @param origin The point of origin.
 * @param dest   The destination points.
 * @return       1 if origin can see destination, 0 otherwise.
 */
static int lineofsight (Game *game, int origin, int dest)
{
    int x0, y0, /* start coordinates */
	x1, y1, /* end coordinates */
	x, y, /* current coordinates */
	dx, dy, /* difference between start and end coordinates */
	sx, sy, /* step in X and Y directions */
	err, e2, /* error */
	loc, /* current location as an integer */
	block; /* block at current location */

    /* initialise */
    x = x0 = origin % game->map->width;
    y = y0 = origin / game->map->width;
    x1 = dest % game->map->width;
    y1 = dest / game->map->width;
    dx = abs (x1 - x0);
    sx = x0 < x1 ? 1 : -1;
    dy = -abs (y1 - y0);
    sy = y0 < y1 ? 1 : -1;
    err = dx + dy;

    /* main loop */
    while (x != x1 || y != y1) {

	/* check the current location */
	if (x != x0 || y != y0) {
	    loc = x + game->map->width * y;
	    block = game->map->blocks[loc] & 0xf0;
	    if ((block & MISSION_UNIT) ||
		(block == LEVELMAP_WALL) ||
		(block == LEVELMAP_DOOR))
		return 0;
	}

	/* next square */
        e2 = 2 * err;
        if (e2 >= dy) {
            if (x == x1)
		break;
            err = err + dy;
            x = x + sx;
	}
        if (e2 <= dx) {
            if (y == y1)
		break;
            err = err + dx;
            y = y + sy;
	}
    }

    /* no blockages encountered */
    return 1;
}

/**
 * Clear the sight line grid.
 * @param game The game object.
 */
static void clearsightlines (Game *game)
{
    if (game->sightlines)
	memset
	    (game->sightlines,
	     0,
	     game->map->width
	     * game->map->height
	     * sizeof (int));
    else if (! (game->sightlines = calloc
		(1, game->map->width
		 * game->map->height
		 * sizeof (int))))
	fatal_error (FATAL_MEMORY);
}

/**
 * Check the whole level for lines of sight.
 * @param game   The game object.
 * @param origin The point of origin.
 * @param mask   The bitmask.
 */
static void calcsightlines (Game *game, int origin, int mask)
{
    char *checked; /* an array to store whether each location has been
		    * checked yet */
    int x, /* x counter */
	y; /* y counter */

    /* reserve memory */
    if (! (checked = calloc
	   (1, game->map->width * game->map->height)))
	fatal_error (FATAL_MEMORY);
    if (! game->sightlines)
	if (! (game->sightlines = calloc
	       (1, game->map->width
		* game->map->height * sizeof (int))))
	    fatal_error (FATAL_MEMORY);

    /* initialise centre of search */
    game->sightlines[origin] |= mask;
    checked[origin] = 1;

    /* check towards top and bottom edges */
    for (x = 0; x < game->map->width; ++x) {
	calcsightline (game, checked, origin, x, mask);
	calcsightline
	    (game,
	     checked,
	     origin,
	     game->map->width * (game->map->height - 1) + x,
	     mask);
    }
    
    /* check towards left and right edges */
    for (y = 0; y < game->map->height; ++y) {
	calcsightline
	    (game,
	     checked,
	     origin,
	     y * game->map->width,
	     mask);
	calcsightline
	    (game,
	     checked,
	     origin,
	     (game->map->width - 1) + y * game->map->width,
	     mask);
    }

    /* check for any missed squares */
    for (x = 0; x < game->map->width; ++x)
	for (y = 0; y < game->map->height; ++y)
	    if (! checked[x + game->map->width * y])
		calcsightline
		    (game,
		     checked,
		     origin,
		     x + game->map->width * y,
		     mask);

    /* free temporary memory */
    free (checked);
}

/**
 * Check the sightlines to non-player units.
 * @param game The game object.
 */
static void calcunitsightlines (Game *game, DisplayHook hook)
{
    int unitcount, /* count of enemy units */
	c; /* counter */
    Unit **units; /* pointer to enemy unit array */

    /* initialise */
    game->clearsightlines (game);
    if (game->state == STATE_MAP) {
	unitcount = game->enemycount;
	units = game->enemies;
    } else {
	unitcount = game->unitcount;
	units = game->units;
    }

    /* calculate enemy sightlines */
    for (c = 0; c < unitcount; ++c) {
	if (hook)
	    hook (c);
	if (units[c])
	    game->calcsightlines
		(game,
		 units[c]->x + game->map->width * units[c]->y,
		 1 << c);
    }

    /* calculate hostage sightlines */
    if (game->state == STATE_MAP)
	for (c = 0; c < game->hostagecount; ++c) {
	    if (hook)
		hook (game->enemycount + c);
	    if (game->hostages[c])
		game->calcsightlines
		    (game,
		     game->hostages[c]->x
		     + game->map->width
		     * game->hostages[c]->y,
		     1 << (game->enemycount + c));
	}
}

/**
 * Check for change in unit visibility.
 * @param game       The game object.
 * @param origin     The origin square.
 * @param visibility The present visibility.
 * @return           The bits that were changed.
 */
static int changedvisibility
(Game *game, int origin, int visibility)
{
    if ((visibility | game->sightlines[origin]) == visibility)
	return 0;
    else
	return game->sightlines[origin] & ~visibility;
}

/**
 * Return the briefing text for this mission.
 * @param game   The game object.
 * @param buffer The buffer for the mission text.
 * @return       A pointer to the buffer.
 */
static char *briefing (Game *game, char *buffer)
{
    switch (game->type) {
    case MISSION_ASSASSINATE:
	sprintf
	    (buffer,
	     briefings[game->type],
	     game->enemies[0]->name);
	break;
    case MISSION_HOSTAGE:
	if (game->hostages[0])
	    sprintf
		(buffer,
		 briefings[game->type],
		 game->hostages[0]->name);
	else if (game->units[4])
	    sprintf
		(buffer,
		 briefings[game->type],
		 game->units[4]->name);
	else /* should never happen */
	    strcpy (buffer, briefings[game->type]);
	break;
    default:
	strcpy (buffer, briefings[game->type]);
    }
    return buffer;
}

/**
 * Return the unit at a particular location.
 * @param game The game object.
 * @param x    The X coordinate.
 * @param y    The Y coordinate.
 * @return     The unit at that location, or NULL.
 */
static Unit *getunit (Game *game, int x, int y)
{
    int loc; /* location as an integer */
    loc = x + game->map->width * y;
    switch (game->unitcache[loc] & 0xf0) {
    case MISSION_CACHE_UNIT:
	return getgame ()->units[game->unitcache[loc] & 0xf];
    case MISSION_CACHE_ENEMY:
	return game->enemies[game->unitcache[loc] & 0xf];
    case MISSION_CACHE_HOSTAGE:
	return game->hostages[game->unitcache[loc] & 0xf];
    default:
	return NULL;
    }
}

/**
 * Return the stack at the specified location.
 * @param game   The game object.
 * @param x      The X coordinate.
 * @param y      The Y coordinate.
 * @param create 1 to create a new stack if none exists.
 * @return       An item stack, or NULL if no room.
 */
static ItemStack *getstack (Game *game, int x, int y, int create)
{
    int loc; /* location as a single integer */
    ItemStack *stack; /* new stack */

    /* check for an existing stack */
    loc = x + game->map->width * y;
    if (game->stackcache[loc] & 0x80)
	return game->stacks[game->stackcache[loc] & 0x7f];
    if (! create)
	return NULL;

    /* create a new stack */
    if (! (game->stacks = realloc
	   (game->stacks,
	    ++game->stackcount * sizeof (ItemStack *))))
	fatal_error (FATAL_MEMORY);
    if (! (stack = new_ItemStack ()))
	fatal_error (FATAL_MEMORY);
    game->stacks[game->stackcount - 1] = stack;
    game->stackcache[loc] = 0x80 | (game->stackcount - 1);
    stack->x = x;
    stack->y = y;
    return stack;
}

/**
 * Remove a stack from the map.
 * @param game  The game object.
 * @param stack The stack to remove.
 */
static void removestack (Game *game, ItemStack *stack)
{
    int loc, /* stack location as an integer */
	id; /* the stack id */
    loc = stack->x + game->map->width * stack->y;
    id = game->stackcache[loc] & 0x7f;
    game->stackcache[loc] = 0;
    stack->destroy (stack);
    game->stacks[id] = NULL;
}

/**
 * Return a bitfield for units that can attack or can see. Used as a
 * quick reference for opportunity fire, and for sighting enemy units.
 * @param game   The game object.
 * @param action Minimum action points to qualify.
 * @return       A bitfield of units that can attack.
 */
static int activeunits (Game *game, int action)
{
    int unitcount, /* count of enemy units */
	attack, /* the bitmask to return */
	c; /* counter */
    Unit **units; /* pointer to enemy units */

    /* determine which side we're looking at */
    if (game->state == STATE_MAP) {
	unitcount = game->enemycount;
	units = game->enemies;
    } else {
	unitcount = game->unitcount;
	units = game->units;
    }

    /* check units */
    attack = 0;
    for (c = 0; c < unitcount; ++c)
	if (units[c] &&
	    units[c]->health > 0 &&
	    ! units[c]->unconscious &&
	    units[c]->action >= action)
	    attack |= 1 << c;

    /* return the attack bitfield */
    return attack;
}

/**
 * Check for opportunity attacks.
 * @param game The game object.
 * @return     Pointer to the unit that attacked, or NULL.
 */
static Unit *opportunity (Game *game)
{
    int unitcount, /* unit count */
	loc, /* location of the current unit */
	mask, /* visibility for current unit */
	c; /* counter */
    Unit **units, /* pointer to possible attackers */
	*attacker; /* the unit that attacked */

    /* check if there's a target to attack */
    if (! game->unit)
	return NULL;
    else if (! game->unit->health)
	return NULL;
    else if (game->unit->unconscious)
	return NULL;

    /* determine who the attacking side is */
    if (game->state == STATE_MAP) {
	unitcount = game->enemycount;
	units = game->enemies;
    } else {
	unitcount = game->unitcount;
	units = game->units;
    }

    /* see if anything can see the target */
    loc = game->unit->x + game->map->width * game->unit->y;
    if (! (mask = game->attack & game->sightlines[loc]))
	return NULL;

    /* try to attack with any units that can see the target */
    for (c = 0; c < unitcount; ++c)
	if (mask & (1 << c)) {
	    attacker = units[c];
	    if (attacker->surprise == UNIT_SURPRISED &&
		game->unit->creeping &&
		! attacker->creeping &&
		skillroll (game->unit->getskill
			   (game->unit, SKILL_STEALTH)))
		continue;
	    if (game->state == STATE_MAP) {
		game->ai[c]->investigating = 1;
		game->ai[c]->investx = game->unit->x;
		game->ai[c]->investy = game->unit->y;
	    }
	    if (attacker->attack (attacker, game->unit, game)) {
		if (attacker->action < 6)
		    game->attack ^= ~(1 << c) && 0xffff;
		attacker->surprise = UNIT_NOTSURPRISED;
		return attacker;
	    }
	}

    /* if we got here, no units could attack */
    return NULL;
}

/**
 * Detect the end of a game. A game is over if there are no living,
 * conscious player units on the map at the end of the turn.
 * @param game The game object.
 * @return     1 if the game is ended.
 */
static int detectend (Game *game)
{
    int c; /* unit counter */
    Unit *unit; /* unit being counted */
    for (c = 0; c < game->unitcount; ++c)
	if ((unit = game->units[c]) &&
	    unit->health &&
	    ! unit->unconscious &&
	    (unit->x || unit->y))
	    return 0;
    return 1;
}

/**
 * Ascertain if a mission is successful or not.
 * @param game The game object.
 * @return     1 on victory, 0 on defeat.
 */
static int ascertainvictory (Game *game)
{
    switch (game->type) {
    case MISSION_ASSASSINATE:
	return ascertainassassination (game);
    case MISSION_ANNIHILATE:
	return ascertainannihilation (game);
    case MISSION_HOSTAGE:
    case MISSION_RESCUE:
	return ascertainhostagefreed (game);
    case MISSION_RETRIEVE:
    case MISSION_GATHER:
	return ascertaincardretrieved (game);
    default:
	return 0;
    }
}

/**
 * Calculate mission score after victory.
 * @param game   The game object.
 * @param scores An array of integers to store the scores in.
 */
static void calcscore (Game *game, int *scores, int *performance)
{
    int c, /* general counter */
	u, /* unit counter */
	i, /* item counter */
	kills = 0, /* number of kills */
	maxhealth = 0, /* maximum unit health */
	health = 0, /* unit health */
	freed = 0, /* number of hostages freed */
	cards = 0, /* number of cards taken */
	elements = 0, /* elements of score counted */
	total = 0; /* total score */

    /* health check */
    for (c = 0; c < 4; ++c)
	if (game->units[c]) {
	    maxhealth
		+= game->units[c]->strength
		+ game->units[c]->endurance
		+ 7;
	    if (game->units[c]->x == 0 &&
		game->units[c]->y == 0)
		health += game->units[c]->health;
	}
    scores[SCORE_HEALTH] = 100 * ((float) health / maxhealth);
    performance[SCORE_HEALTH] = health;
    performance[5 + SCORE_HEALTH] = maxhealth;

    /* count kills */
    for (c = 0; c < game->enemycount; ++c)
	if (game->enemies[c] &&
	    game->enemies[c]->health == 0)
	    ++kills;
    scores[SCORE_KILLS] = 100 * ((float) kills / game->enemycount);
    performance[SCORE_KILLS] = kills;
    performance[5 + SCORE_KILLS] = game->enemycount;

    /* count freed hostages */
    for (c = 4; c < game->unitcount; ++c)
	if (game->units[c] &&
	    game->units[c]->health &&
	    game->units[c]->x == 0 &&
	    game->units[c]->y == 0)
	    ++freed;
    if (game->hostagecount)
	scores[SCORE_FREED]
	    = 100 * ((float) freed / game->hostagecount);
    else
	scores[SCORE_FREED] = 0;
    performance[SCORE_FREED] = freed;
    performance[5 + SCORE_FREED] = game->hostagecount;

    /* count cards taken */
    for (u = 0; u < game->unitcount; ++u)
	if (game->units[u] &&
	    game->units[u]->health &&
	    game->units[u]->x == 0 &&
	    game->units[u]->y == 0)
	    for (i = 0; i < 10; ++i)
		if (game->units[u]->inventory[i] == ITEM_CARD)
		    ++cards;
    if (game->cardcount)
	scores[SCORE_CARDS] = 100 * ((float) cards / game->cardcount);
    else
	scores[SCORE_CARDS] = 0;
    performance[SCORE_CARDS] = cards;
    performance[5 + SCORE_CARDS] = game->cardcount;

    /* score for speed */
    if (game->turns)
	scores[SCORE_SPEED]
	    = 100 * ((float) missionturns[game->type] / game->turns);
    else
	scores[SCORE_SPEED] = 100;
    if (scores[SCORE_SPEED] > 100)
	scores[SCORE_SPEED] = 100;
    performance[SCORE_SPEED] = game->turns;
    performance[5 + SCORE_SPEED] = missionturns[game->type];;

    /* total the scores */
    for (c = 0; c < 5; ++c) {
	elements += scoremultipliers[game->type][c];
	total += scoremultipliers[game->type][c] * scores[c];
    }
    game->score = scores[SCORE_TOTAL] = total / elements;
}

/**
 * Get score multiplier for the current mission.
 * @param game    The game object.
 * @param scoreid The mission score ID.
 * @return        The score multiplier.
 */
static int getmultiplier (Game *game, int scoreid)
{
    return scoremultipliers[game->type][scoreid];
}

/*----------------------------------------------------------------------
 * Constructors.
 */

/**
 * Create a new game.
 * @returns the newly initialised game.
 */
Game *new_Game (void)
{
    Game *game; /* the game object */

    /* reserve memory */
    if (! (game = malloc (sizeof (Game))))
	fatal_error (FATAL_MEMORY);

    /* initialise the attributes */
    game->state = STATE_NONE;
    game->score = 0;
    game->highest = 0;
    game->turns = 0;
    game->type = MISSION_RANDOM;
    game->level = MISSION_EASY;
    game->map = NULL;
    game->unitcount = 0;
    game->units = NULL;
    game->enemycount = 0;
    game->enemies = NULL;
    game->ai = NULL;
    game->hostagecount = 0;
    game->hostages = NULL;
    game->cabinetcount = 0;
    game->cabinets = NULL;
    game->stackcount = 0;
    game->stacks = NULL;
    game->unit = NULL;
    game->cardcount = 0;
    game->visibility = 0;
    game->sightlines = NULL;
    game->unitcache = NULL;
    game->stackcache = NULL;

    /* initialise the methods */
    game->destroy = destroy;
    game->save = save;
    game->load = load;
    game->generate = generate;
    game->nextunit = nextunit;
    game->initturn = initturn;
    game->lineofsight = lineofsight;
    game->clearsightlines = clearsightlines;
    game->calcsightlines = calcsightlines;
    game->calcunitsightlines = calcunitsightlines;
    game->changedvisibility = changedvisibility;
    game->briefing = briefing;
    game->getunit = getunit;
    game->getstack = getstack;
    game->removestack = removestack;
    game->activeunits = activeunits;
    game->opportunity = opportunity;
    game->detectend = detectend;
    game->ascertainvictory = ascertainvictory;
    game->calcscore = calcscore;
    game->getmultiplier = getmultiplier;

    /* return the new game */
    return game;
}

/**
 * Return the title of a particular mission type.
 * @param type The type ID of the mission.
 * @return     A pointer to the mission name.
 */
char *getmissiontype (int type)
{
    return titles[type];
}

