/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2020. Released under the GNU GPL.
 * Created: 06-Aug-2020.
 *
 * Main Program Module.
 */

/*----------------------------------------------------------------------
 * Headers
 */

/* Standard ANSI C */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Project Headers */
#include "cgalib.h"
#include "names.h"
#include "fatal.h"

/* this should have been included in cgalib.h but wasn't */
#include "effect.h"
#include "tune.h"

/*----------------------------------------------------------------------
 * File Level Variables.
 */

/** @var output is the output file handle */
static FILE *output;

/** @var scr is the screen */
static Screen *screen;

/*----------------------------------------------------------------------
 * Level 1 Functions.
 */

/**
 * Open a new DAT file.
 */
static void open_dat_file (void)
{
    if (! (output = fopen("sccc/SCCC.DAT", "wb")))
	fatal_error (FATAL_NODATA);
    fwrite ("SCM007D", 8, 1, output);
}

/**
 * Initialise the screen.
 */
static void initialise_screen (void)
{
    if (! (screen = new_Screen (CGALIB_MEDIUM_MONO, CGALIB_SHOWN)))
	fatal_error (FATAL_DISPLAY);
    fatal_screen (screen);
    screen->palette (screen, 5, 1);
}

/**
 * Load the assets from the PIC file.
 */
static void load_assets (void)
{
    FILE *input; /* screen input file */
    if (! (input = fopen ("assets/SCCC.PIC", "rb")))
	fatal_error (FATAL_NODATA);
    if (! screen->read (screen, CGALIB_SCREEN_FILE, input))
	fatal_error (FATAL_DISPLAY);
    fclose (input);
}

/**
 * Copy the title bitmaps to the output file.
 */
static void copy_title_bitmaps (void)
{
    /* local variables */
    Bitmap *bit; /* temporary reusable bitmap variable */

    /* copy Cyningstan logo */
    bit = screen->get (screen, 72, 8, 128, 16);
    bit->write (bit, output);
    bit->destroy (bit);

    /* copy title screen picture */
    bit = screen->get (screen, 64, 24, 144, 128);
    bit->write (bit, output);
    bit->destroy (bit);

    /* copy title screen text */
    bit = screen->get (screen, 64, 160, 240, 32);
    bit->write (bit, output);
    bit->destroy (bit);
}

/**
 * Copy the Future font into the output file.
 */
static void copy_font (void)
{
    /* local variables */
    FILE *input; /* input file handle */
    char header[8]; /* header read from input file */
    Font *fnt; /* font read from input file */
    Bitmap *bit; /* bitmap for fraction font */
    int c; /* counter */

    /* open input file, read and verify header */
    if (! (input = fopen ("cgalib2/cgalib/fnt/future.fnt", "rb")))
	fatal_error (FATAL_NODATA);
    if (! (fread (header, 8, 1, input)))
	fatal_error (FATAL_NODATA);
    if (strcmp (header, "CGA100F") && strcmp (header, "CGA200F"))
	fatal_error (FATAL_NODATA);

    /* read the font and copy it to the output file */
    if (! (fnt = read_Font (input, header[3] - '0')))
	fatal_error (FATAL_NODATA);
    fclose (input);
    fnt->write (fnt, output);
    fnt->destroy (fnt);

    /* extract the fraction font from the screen */
    if (! (fnt = new_Font ('0', ':', 4, 8)))
	fatal_error (FATAL_MEMORY);
    for (c = '0'; c <= ':'; ++c) {
	bit = screen->get (screen, 8 + 4 * (c - '0'), 28, 4, 8);
	fnt->put (fnt, bit, c);
	bit->destroy (bit);
    }
    fnt->write (fnt, output);
    fnt->destroy (fnt);
	
}

/**
 * Copy the tiles from the loaded assets to the output file.
 */
static void copy_tiles (void)
{
    Bitmap *bit; /* temporary reusable bitmap */
    int x, /* x coordinate counter */
	y; /* y coordinate counter */

    /* main block of tiles */
    for (x = 216; x < 312; x += 16)
	for (y = 24; y < 152; y += 16) {
	    bit = screen->get (screen, x, y, 16, 16);
	    bit->write (bit, output);
	    bit->destroy (bit);
	}

    /* tiles in backpack */
    for (x = 8; x < 56; x += 16)
	for (y = 160; y < 192; y += 16) {
	    bit = screen->get (screen, x, y, 16, 16);
	    bit->write (bit, output);
	    bit->destroy (bit);
	}

    /* unarmed combat tiles from all over the place */
    bit = screen->get (screen, 8, 40, 16, 16);
    bit->write (bit, output);
    bit->destroy (bit);
    bit = screen->get (screen, 8, 132, 16, 16);
    bit->write (bit, output);
    bit->destroy (bit);

    /* scroll arrow mini-tiles */
    for (y = 160; y <= 184; y += 12) {
	bit = screen->get (screen, 304, y, 8, 8);
	bit->write (bit, output);
	bit->destroy (bit);
    }

    /* skull tile and mask */
    for (x = 216; x <= 224; x += 8) {
	bit = screen->get (screen, x, 8, 8, 8);
	bit->write (bit, output);
	bit->destroy (bit);
    }
}

/**
 * Copy the backtround to the output file.
 */
static void copy_background (void)
{
    /* Local variables */
    Bitmap *bit; /* temporary bitmap for background */

    /* initialise the bitmap */
    if (! (bit = screen->get (screen, 0, 0, 320, 200)))
	fatal_error (FATAL_NODATA);

    /* blank out the tiles and other graphics */
    bit->box (bit, 64, 8, 144, 144, CGALIB_BOX_BLANK);
    bit->box (bit, 64, 160, 240, 32, CGALIB_BOX_BLANK);
    bit->box (bit, 216, 24, 96, 128, CGALIB_BOX_BLANK);
    bit->box (bit, 8, 28, 48, 8, CGALIB_BOX_BLANK);
    bit->box (bit, 8, 160, 48, 32, CGALIB_BOX_BLANK);
    bit->box (bit, 8, 132, 16, 16, CGALIB_BOX_BLANK);
    bit->box (bit, 216, 8, 96, 8, CGALIB_BOX_BLANK);
    bit->transfer (bit, bit, 304, 160, 304, 172, 8, 8, CGALIB_SET);
    bit->transfer (bit, bit, 304, 184, 304, 172, 8, 8, CGALIB_SET);

    /* write the bitmap and free the memory */
    bit->write (bit, output);
    bit->destroy (bit);
}

/**
 * Copy the name generator data.
 */
static void copy_names (void)
{
    FILE *input; /* input file handle */
    char header[8]; /* header read from input file */
    Names *names; /* the name generator */

    /* initialise the name generator */
    if (! (names = new_Names ()))
	fatal_error (FATAL_MEMORY);

    /* open input file, read and verify header */
    if (! (input = fopen ("namelib/bin/names.nam", "rb")))
	fatal_error (FATAL_NODATA);
    if (! (fread (header, 8, 1, input)))
	fatal_error (FATAL_NODATA);
    if (strcmp (header, "NAM100N"))
	fatal_error (FATAL_NODATA);

    /* read the name data and copy it to the output file */
    if (! names->read (names, input))
	fatal_error (FATAL_NODATA);
    fclose (input);
    names->write (names, output);
    names->destroy (names);
}

/**
 * Close the screen.
 */
static void close_screen (void)
{
    screen->destroy (screen);
    fatal_screen (NULL);
}

/**
 * Add music and sound effects to the asset files.
 */
static void add_sounds (void)
{
    Effect *effect; /* a sound effect object */
    Tune *tune; /* a tune object */
    FILE *input; /* input file handle for the tune */
    char header[8]; /* header of the tune input file */

    /* create the tune and sound effect objects */
    if (! (tune = new_Tune ()))
	fatal_error (FATAL_MEMORY);
    if (! (effect = new_Effect ()))
	fatal_error (FATAL_MEMORY);

    /* load the tune and save it to the asset file */
    if (! (input = fopen ("assets/BUXTEHU1.TUN", "rb")))
	fatal_error (FATAL_NODATA);
    if (! (fread (header, 8, 1, input)))
	fatal_error (FATAL_NODATA);
    if (strcmp (header, "SPK100T"))
	fatal_error (FATAL_NODATA);
    if (! tune->read (tune, input))
	fatal_error (FATAL_NODATA);
    fclose (input);
    tune->write (tune, output);

    /* generate and save the movement noise */
    effect->pattern = EFFECT_RISE;
    effect->repetitions = 1;
    effect->low = 24;
    effect->high = 24;
    effect->duration = 1;
    effect->pause = 0;
    effect->write (effect, output);

    /* generate and save the pew-pew noise */
    effect->pattern = EFFECT_FALL;
    effect->repetitions = 2;
    effect->low = 0;
    effect->high = 60;
    effect->duration = 3;
    effect->pause = 0;
    effect->write (effect, output);

    /* generate and save the stabbing noise */
    effect->pattern = EFFECT_RISE;
    effect->repetitions = 1;
    effect->low = 48;
    effect->high = 60;
    effect->duration = 2;
    effect->pause = 0;
    effect->write (effect, output);

    /* generate and save the hitting noise */
    effect->pattern = EFFECT_RISE;
    effect->repetitions = 1;
    effect->low = 24;
    effect->high = 36;
    effect->duration = 2;
    effect->pause = 0;
    effect->write (effect, output);

    /* generate and save the unconsciousness noise */
    effect->pattern = EFFECT_RISE;
    effect->repetitions = 1;
    effect->low = 12;
    effect->high = 24;
    effect->duration = 2;
    effect->pause = 0;
    effect->write (effect, output);

    /* generate and save the death noise */
    effect->pattern = EFFECT_FALL;
    effect->repetitions = 1;
    effect->low = 36;
    effect->high = 48;
    effect->duration = 7;
    effect->pause = 0;
    effect->write (effect, output);

    /* generate and save the comms noise */
    effect->pattern = EFFECT_RISE;
    effect->repetitions = 4;
    effect->low = 60;
    effect->high = 60;
    effect->duration = 1;
    effect->pause = 1;
    effect->write (effect, output);

    /* clean up */
    tune->destroy (tune);
    effect->destroy (effect);
}

/**
 * Close the DAT file.
 */
static void close_dat_file (void)
{
    fclose (output);
}

/*----------------------------------------------------------------------
 * Top Level Function.
 */

/**
 * Main Program.
 * @returns 0 on a successful run.
 */
int main (void)
{
    open_dat_file ();
    initialise_screen ();
    load_assets ();
    copy_title_bitmaps ();
    copy_font ();
    copy_tiles ();
    copy_background ();
    copy_names ();
    close_screen ();
    add_sounds ();
    close_dat_file ();
    printf ("SCMZ\\SCMZ.DAT generated successfully.\n");
    return 0;
}
