/*======================================================================
 * Star Cadre: Combat Class
 * A single-level tactical combat game.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 30-May-2024.
 *
 * Config Module.
 */

/*
 * Required Headers
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "config.h"
#include "game.h"
#include "unit.h"
#include "utils.h"

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Clear the configuration and insert default settings.
 * @param config The configuration object.
 */
static void clearconfig (Config *config)
{
    int c; /* counter */
    config->level = MISSION_EASY;
    config->type = MISSION_RANDOM;
    for (c = 0; c < 4; ++c)
	config->classes[c] = UNIT_MEDIC + c;
    config->tutorial = 0;
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Destroy the score table object.
 * @param config The score table object.
 */
static void destroy (Config *config)
{
    if (config)
	free (config);
}

/**
 * Save the high score table.
 * @param config The score table object.
 * @return       1 if successful, 0 if not.
 */
static int save (Config *config)
{
    FILE *output; /* the output file */
    int c; /* general counter */

    /* open the output file and write the header */
    if (! (output = fopen ("SCCC.INI", "wb")))
	return 0;
    if (! (fwrite ("SCC002I", 8, 1, output))) {
	fclose (output);
	return 0;
    }

    /* write the difficulty level and mission type */
    if (! writebyte (&config->level, output)) {
	fclose (output);
	return 0;
    }
    if (! writebyte (&config->type, output)) {
	fclose (output);
	return 0;
    }

    /* write the unit classes */
    for (c = 0; c < 4; ++c)
	if (! writebyte (&config->classes[c], output)) {
	    fclose (output);
	    return 0;
	}

    /* write the tutorial bitfield */
    if (! writeint (&config->tutorial, output)) {
	fclose (output);
	return 0;
    }

    /* success */
    fclose (output);
    return 1;
}

/**
 * Load the high score table. If there is no valid high score
 * table file, initialise all the config to zero.
 * @param config The score table object.
 * @return       1 if successfully loaded, 0 if not.
 */
static int load (Config *config)
{
    FILE *input; /* the output file */
    char header[8]; /* header text */
    int c; /* counter */

    /* open the input file and read and check the header */
    if (! (input = fopen ("SCCC.INI", "rb")))
	return 0;
    if (! fread (header, 8, 1, input)) {
	fclose (input);
	return 0;
    }
    if (strcmp ("SCC002I", header)) {
	clearconfig (config);
	fclose (input);
	return 0;
    }

    /* read the difficulty level and mission type */
    if (! readbyte (&config->level, input)) {
	fclose (input);
	return 0;
    }
    if (! readbyte (&config->type, input)) {
	fclose (input);
	return 0;
    }

    /* read the unit classes */
    for (c = 0; c < 4; ++c)
	if (! readbyte (&config->classes[c], input)) {
	    fclose (input);
	    return 0;
	}

    /* read the tutorial bitfield */
    if (! readint (&config->tutorial, input)) {
	fclose (input);
	return 0;
    }

    /* success */
    fclose (input);
    return 1;
}

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Create the score table object.
 * @return The new score table object.
 */
Config *new_Config (void)
{
    Config *config; /* the new config */

    /* allocate memory for the configuration */
    if (! (config = malloc (sizeof (Config))))
	return NULL;

    /* initialise attributes */
    clearconfig (config);

    /* initialise classes */
    config->destroy = destroy;
    config->save = save;
    config->load = load;

    /* return the new config */
    return config;
}

