/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 24-Aug-2024.
 *
 * Puzzle UI Module.
 */

/*----------------------------------------------------------------------
 * Included Headers.
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

/* project headers */
#include "pym.h"
#include "ui.h"
#include "display.h"
#include "controls.h"
#include "calendar.h"
#include "game.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var ui A pointer to the UI structure. */
static UI *ui;

/** @var display The display object. */
static Display *display;

/** @var controls The controls object. */
static Controls *controls;

/** @var calendar The puzzle calendar. */
static Calendar *calendar;

/** @var game The game object. */
static Game *game;

/** @var calmenu The main puzzle screen menu. */
static char *calmenu[] = {
    "Cancel",
    "Puzzle",
    "Today",
    "Quit game"
};

/** @enum CalmenuID Identifiers for menu options. */
typedef enum {
    CALMENU_CANCEL,
    CALMENU_PUZZLE,
    CALMENU_TODAY,
    CALMENU_QUITGAME,
    CALMENU_COUNT
} CalmenuID;

/** @var yday The year day. */
static int yday;

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Move the cursor around the calendar grid.
 */
static void movecursor (void)
{
    int col, /* column position */
	row; /* row position */

    /* loop until menu is requested */
    do {

	/* show the cursor and wait for input */
	display->showcalcursor (calendar, yday);
	display->update ();
	controls->release (250);
	controls->wait (0);
	controls->poll ();
	display->hidecalcursor ();
	display->update ();
	col = calendar->col[yday];
	row = calendar->row[yday];

	/* movement left */
	if (controls->left)
	    while (--col >= 0)
		if (calendar->yday[col][row] != -1) {
		    yday = calendar->yday[col][row];
		    break;
		}

	/* movement right */
	if (controls->right)
	    while (++col <= 20)
		if (calendar->yday[col][row] != -1) {
		    yday = calendar->yday[col][row];
		    break;
		}

	/* movement up */
	if (controls->up)
	    while (--row >= 0)
		if (calendar->yday[col][row] != -1) {
		    yday = calendar->yday[col][row];
		    break;
		}

	/* movement up */
	if (controls->down)
	    while (++row <= 19)
		if (calendar->yday[col][row] != -1) {
		    yday = calendar->yday[col][row];
		    break;
		}

    } while (! controls->fire);
}

/**
 * Return the default option.
 */
static int defaultoption (void)
{
    time_t t; /* time */
    struct tm tm; /* time structure */
    t = time (NULL);
    tm = *localtime (&t);
    return
	(calendar->year < tm.tm_year + 1900) ||
	(yday <= calendar->today (calendar) &&
	 calendar->year == tm.tm_year + 1900);
}

/**
 * Choose a puzzle.
 */
static int choosepuzzle (void)
{
    /* check that the puzzle is available */
    if (calendar->today (calendar) < 0)
	return 0;
    if (calendar->today (calendar) < calendar->days &&
	yday > calendar->today (calendar))
	return 0;

    /* initialise the new puzzle */
    if (game->yday != yday || game->year != calendar->year) {
	game->year = calendar->year;
	game->yday = yday;
	game->initialise (calendar);
    }

    /* return */
    return 1;
}

/**
 * Go to the today, if we're looking at this year's calendar.
 */
static void gototoday (void)
{
    yday = calendar->today (calendar);
    if (yday < 0)
	yday = 0;
    else if (yday >= calendar->days)
	yday = calendar->days - 1;
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Initialise the UI.
 * @return   1 on success, 0 on failure.
 */
static int init (void)
{
    /* initialise module-wide variables */
    calendar = getcalendar ();
    game = getgame ();
    display = getdisplay ();
    controls = getcontrols ();
    yday = game->yday;

    /* show the initial display */
    display->showpanel ();
    display->showcalendar (calendar, game);
    display->update ();

    /* return success */
    return 1;
}

/**
 * Operate the UI.
 * @return The next game state.
 */
static int operate (void)
{
    int option; /* menu option chosen */
    do {

	/* get cursor position */
	movecursor ();

	/* launch menu */
	if (controls->fire) {
	    option = display->menu
		(CALMENU_COUNT,
		 calmenu,
		 defaultoption ());
	    switch (option) {
	    case CALMENU_PUZZLE:
		if (choosepuzzle ())
		    return (game->state = GAME_PUZZLE);
		break;
	    case CALMENU_TODAY:
		gototoday ();
		break;
	    case CALMENU_QUITGAME:
		return GAME_QUIT;
	    }
	}

    } while (1);
}

/**
 * Construct Puzzle UI
 * @return The new UI.
 */
UI *new_CalendarUI (void)
{
    /* reserve memory for the UI */
    if (! (ui = new_UI ()))
	return NULL;

    /* initialise methods */
    ui->init = init;
    ui->operate = operate;

    /* return the UI */
    return ui;
}
