/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 17-Aug-2024.
 *
 * Game Module
 */

/*----------------------------------------------------------------------
 * Headers.
 */

/* standard C headers */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/* project headers */
#include "game.h"
#include "utils.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var game The game object. */
static Game *game;

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Destroy the game object.
 */
static void destroy (void)
{
    if (game->guesscount) {
	while (game->guesscount--)
	    free (game->guesses[game->guesscount]);
	free (game->guesses);
    }
    free (game);
}

/**
 * Save the game object to a file.
 * @return 1 if successful, 0 if not.
 */
static int save (void)
{
    FILE *output; /* the output file handle */
    int c, /* counter */
	ch; /* character to write */
    char word[5]; /* word to write to the game file */

    /* open the output file and write the header */
    if (! (output = fopen ("pym.gam", "wb")))
	return 0;
    fwrite ("PYM100G", 8, 1, output);

    /* write the current game state */
    if (! writebyte (&game->state, output)) {
	fclose (output);
	return 0;
    }

    /* write the current date */
    if (! writeint (&game->year, output)) {
	fclose (output);
	return 0;
    }
    if (! writeint (&game->yday, output)) {
	fclose (output);
	return 0;
    }

    /* write the puzzle grid */
    for (c = 0; c < 25; ++c) {
	ch = (game->letter[c] & 0x3f) |
	    (game->status[c] << 6);
	if (! writebyte (&ch, output)) {
	    fclose (output);
	    return 0;
	}
    }

    /* write the guesses */
    fputc (game->guesscount, output);
    for (c = 0; c < game->guesscount; ++c) {
	strncpy (word, game->guesses[c]->word, 5);
	if (game->guesses[c]->correct)
	    *word |= 0x80;
	fwrite (word, 5, 1, output);
    }

    /* close the output file */
    fclose (output);
    return 1;
}

/**
 * Load the game object from a file.
 * @return 1 if successful, 0 if not.
 */
static int load (void)
{
    FILE *input; /* the input file handle */
    int c; /* counter */
    char header[8], /* file header */
	word[5]; /* word to read to the game file */

    /* open the input file, and read and check the header */
    if (! (input = fopen ("pym.gam", "rb")))
	return 0;
    if (! fread (header, 8, 1, input))
	return 0;
    if (strcmp (header, "PYM100G"))
	return 0;

    /* read the current game state */
    if (! readbyte (&game->state, input)) {
	fclose (input);
	return 0;
    }

    /* read the game date */
    if (! readint (&game->year, input)) {
	fclose (input);
	return 0;
    }
    if (! readint (&game->yday, input)) {
	fclose (input);
	return 0;
    }

    /* read the puzzle grid */
    game->filled = 0;
    fread (game->letter, 25, 1, input);
    for (c = 0; c < 25; ++c) {
	game->status[c] = (game->letter[c] & 0xc0) >> 6;
	game->letter[c] &= 0x3f;
	game->letter[c] |= 0x40 * (game->status[c] != GAME_BLANK);
	game->filled += (game->status[c] != GAME_BLANK);
    }

    /* read the guesses */
    if (! readbyte (&game->guesscount, input)) {
	fclose (input);
	return 0;
    }
    if (game->guesscount) {
	game->guesses = calloc (game->guesscount, sizeof (Guess *));
	if (! game->guesses) {
	    game->guesscount = 0;
	    fclose (input);
	    return 0;
	}
    }
    for (c = 0; c < game->guesscount; ++c) {
	if (! fread (word, 5, 1, input)) {
	    fclose (input);
	    return 0;
	}
	if (! (game->guesses[c] = malloc (sizeof (Guess)))) {
	    fclose (input);
	    return 0;
	}
	game->guesses[c]->correct = *word & 0x80;
	*word &= 0x7f;
	sprintf (game->guesses[c]->word, "%-5.5s", word);
    }

    /* close the output file */
    fclose (input);
    return 1;
}

/**
 * Initialise a new puzzle.
 * game->year and game->yday must already be set to the desired date.
 * @param game     The game object.
 * @param calendar The calendar object.
 */
static void initialise (Calendar *calendar)
{
    int r, /* row counter */
	c; /* column counter */

    /* the player is looking at the puzzle */
    game->state = GAME_PUZZLE;

    /* initialise the puzzle grid */
    game->filled = 0;
    for (r = 0; r < 5; ++r)
	for (c = 0; c < 5; ++c)
	    if (r == c || r == 4 - c) {
		game->letter[c + 5 * r]
		    = calendar->puzzles[game->yday][c + 5 * r];
		game->status[c + 5 * r] = GAME_CLUE;
		++game->filled;
	    } else if (calendar->completed[game->yday]) {
		game->letter[c + 5 * r]
		    = calendar->puzzles[game->yday][c + 5 * r];
		game->status[c + 5 * r] = GAME_VERIFIED;
		++game->filled;
	    } else {
		game->letter[c + 5 * r] = 0;
		game->status[c + 5 * r] = GAME_BLANK;
	    }

    /* clear out the guesses */
    if (game->guesscount) {
	while (game->guesscount--)
	    free (game->guesses[game->guesscount]);
	free (game->guesses);
    }
    game->guesscount = 0;
    game->guesses = NULL;
}

/**
 * Add a word to the list of guesses.
 * @param word    The word to add.
 * @param correct 1 if correct, 0 if incorrect.
 * @return        1 if added, 0 if alread there.
 */
static int add (char *word, int correct)
{
    int w; /* word counter */

    /* check if the word is already there */
    for (w = 0; w < game->guesscount; ++w)
	if (! strncmp (word, game->guesses[w]->word, 5))
	    return 0;

    /* allocate memory for new guess */
    if (game->guesscount) {
	game->guesses = realloc
	    (game->guesses,
	     (game->guesscount + 1) * sizeof (Guess *));
    } else
	game->guesses = malloc (sizeof (Guess *));
    game->guesses[game->guesscount] = malloc (sizeof (Guess));

    /* copy word into guess */
    sprintf (game->guesses[game->guesscount]->word, "%-5.5s", word);
    game->guesses[game->guesscount]->correct = correct;

    /* increase guess counter */
    ++game->guesscount;
    return 1;
}

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Contruct a new game object, or return the existing one.
 * @return The new game object.
 */
Game *get_Game (void)
{
    int c; /* counter */

    /* return existing game object if it exists */
    if (game)
	return game;

    /* reserve memory for the game */
    game = malloc (sizeof (Game));
    if (! game)
	return NULL;

    /* initialise the attributes */
    game->state = GAME_PUZZLE;
    game->year = 0;
    game->yday = 0;
    memset (game->letter, 0, 25);
    for (c = 0; c < 25; ++c)
	game->status[c] = 0;
    game->filled = 0;
    game->guesscount = 0;
    game->guesses = NULL;

    /* initialise the methods */
    game->destroy = destroy;
    game->save = save;
    game->load = load;
    game->initialise = initialise;
    game->add = add;

    /* return the new game */
    return game;
}
