/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 22-Aug-2024.
 *
 * Puzzle Calendar Module.
 */

/*----------------------------------------------------------------------
 * Headers
 */

/* standard c headers */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

/* project headers */
#include "calendar.h"
#include "utils.h"

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Work out where on the calendar grid a date lies.
 * @param year The calendar object.
 * @param yday The day of the year.
 * @param mday Buffer for day of month, 1..31.
 * @param col  Buffer for column number, 0..20.
 * @param row  Buffer for row number, 0..19.
 */
static void getdateposition
(int year, int yday, int *mday, int *col, int *row)
{
    int wday, /* day of week */
	offset, /* offset (day of 1st of month) */
	mrow; /* row in month grid */
    struct tm tm; /* time structure */

    /* get a calendar date from the year day */
    tm.tm_sec = 0;
    tm.tm_min = 0;
    tm.tm_hour = 0;
    tm.tm_mday = yday + 1;
    tm.tm_mon = 0;
    tm.tm_year = year - 1900;
    tm.tm_wday = 0;
    tm.tm_yday = 0;
    tm.tm_isdst = -1;
    mktime (&tm);
    *mday = tm.tm_mday;
    wday = tm.tm_wday;
    tm.tm_mday = 1;
    mktime (&tm);
    offset = tm.tm_wday;

    /* work out the row in the month */
    mrow = (offset + *mday - 1) / 7;
    if (mrow >= 5) mrow -= 5;
    *col = 7 * (tm.tm_mon % 3) + wday;
    *row = 5 * (tm.tm_mon / 3) + mrow;
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Destroy the calendar object.
 * @param calendar The calendar to destroy.
 */
static void destroy (Calendar *calendar)
{
    free (calendar);
}

/**
 * Build the calendar grid.
 * @param calendar The calendar to build.
 */
static void build (Calendar *calendar)
{
    int i, j; /* counters */

    /* initialise the grid */
    for (i = 0; i < 21; ++i)
	for (j = 0; j < 20; ++j)
	    calendar->yday[i][j] = -1;

    /* position each of the days */
    for (i = 0; i < calendar->days; ++i) {
	getdateposition
	    (calendar->year, i,
	     &calendar->mday[i],
	     &calendar->col[i],
	     &calendar->row[i]);
	calendar->yday[calendar->col[i]][calendar->row[i]] = i;
    }
}

/**
 * Save the calendar file.
 * @param calendar The calendar to save.
 * @return         1 if successful, 0 if not.
 */
static int save (Calendar *calendar)
{
    char filename[128], /* output filename */
	ch; /* character to write to the calendar */
    FILE *output; /* output file handle */
    int c, /* character count */
	d; /* day count */

    /* determine output filename */
    sprintf (filename, "%04d-EN.PYM", calendar->year);

    /* open output file and write the header*/
    if (! (output = fopen (filename, "wb")))
	return 0;
    fwrite ("PYM001P", 8, 1, output);

    /* strobe the puzzles */
    for (c = 0; c < 25; ++c)
	for (d = 0; d < calendar->days; ++d) {
	    ch = calendar->puzzles[d][c]
		+ 0x80 * (calendar->completed[d]);
	    fputc (ch, output);
	}

    /* output the scores */
    for (d = 0; d < calendar->days; ++d) {
	ch = calendar->scores[d];
	fputc (ch, output);
    }

    /* close the puzzles */
    fclose (output);
    return 1;
}

/**
 * Load a calendar file.
 * @param calendar The calendar to load.
 * @return         1 if successful, 0 if not.
 */
static int load (Calendar *calendar)
{
    char filename[128], /* input filename */
	header[8]; /* header read from file */
    FILE *input; /* input file handle */
    int ch, /* character read from file */
	c, /* character count */
	d; /* day count */

    /* determine input filename */
    sprintf (filename, "%04d-EN.PYM", calendar->year);

    /* open input file and read the header*/
    if (! (input = fopen (filename, "rb")))
	return 0;
    else if (! (fread (header, 8, 1, input))) {
	fclose (input);
	return 0;
    } else if (strcmp (header, "PYM001P")) {
	fclose (input);
	return 0;
    }

    /* strobe the puzzles */
    for (c = 0; c < 25; ++c)
	for (d = 0; d < calendar->days; ++d) {
	    if (! readbyte (&ch, input)) {
		fclose (input);
		return 0;
	    }
	    if (c == 0)
		calendar->completed[d]
		    = (ch & 0x80)
		    ? 1 : 0;
	    calendar->puzzles[d][c] = ch & 0x7f;
	}

    /* read the scores */
    for (d = 0; d < calendar->days; ++d) {
	if (! readbyte (&ch, input)) {
	    fclose (input);
	    return 0;
	}
	calendar->scores[d] = ch;
    }

    /* close the puzzles */
    fclose (input);
    return 1;
}

/**
 * Submit a puzzle.
 * @param calendar The calendar object.
 * @param game     The present game state.
 * @return         1 if puzzle completed, 0 otherwise.
 */
static int submit (Calendar *calendar, Game *game)
{
    int r, /* row counter */
	c, /* column counter */
	correct[5], /* array of correct flags */
	count = 0; /* count of correct words */

    /* check if already submitted */
    if (calendar->completed[game->yday])
	return 1;

    /* check if square completed */
    for (r = 0; r < 5; ++r)
	for (c = 0; c < 5; ++c)
	    if (game->status[c + 5 * r] == GAME_BLANK)
		return 0;

    /* count the number of correct words */
    for (r = 0; r < 5; ++r) {
	correct[r] = 1;
	for (c = 0; c < 5; ++c) {
	    if (game->letter[c + 5 * r]
		!= calendar->puzzles[game->yday][c + 5 * r])
		correct[r] = 0;
	}
	game->add (&game->letter[5 * r], correct[r]);
	count += correct[r];
    }

    /* mark the puzzle grid */
    for (r = 0; r < 5; ++r)
	for (c = 0; c < 5; ++c)
	    if (r == c || r == 4 - c);
	    else if (correct[r] || correct[c])
		game->status[c + 5 * r] = GAME_VERIFIED;
	    else {
		game->letter[c + 5 * r] = 0;
		game->status[c + 5 * r] = GAME_BLANK;
		game->filled--;
	    }

    /* mark the puzzle correct if appropriate */
    if (count == 5) {
	calendar->completed[game->yday] = 1;
	calendar->scores[game->yday]
	    = game->guesscount > 40
	    ? 5
	    : 45 - game->guesscount;
    }

    /* return correct or not */
    return (count == 5);
}

/**
 * Return today's date in day-of-year form.
 * @param calendar The calendar object.
 * @return         The day number, 0..365.
 */
static int today (Calendar *calendar)
{
    time_t t; /* time */
    struct tm tm; /* time structure */
    t = time (NULL);
    tm = *localtime (&t);
    if (calendar->year < tm.tm_year + 1900)
	return calendar->days;
    else if (calendar->year > tm.tm_year + 1900)
	return -1;
    else /* if (calendar->year == tm.tm_year + 1900) */
	return tm.tm_yday;
}

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Contruct a new calendar.
 * @param year The calendar year.
 * @return     The new calendar object.
 */
Calendar *new_Calendar (int year)
{
    Calendar *calendar; /* calendar to return */
    int i, j; /* initialisation counters */

    /* reserve memory for the calendar */
    if (! (calendar = malloc (sizeof (Calendar))))
	return NULL;

    /* initialise methods */
    calendar->destroy = destroy;
    calendar->build = build;
    calendar->save = save;
    calendar->load = load;
    calendar->submit = submit;
    calendar->today = today;

    /* initialise year attributes */
    calendar->year = year;
    calendar->days
	= 365
	+ ((year % 4 == 0 && year % 100 != 0) || (year % 400 == 0));

    /* initialise puzzle attributes */
    for (i = 0; i < 366; ++i) {
	calendar->completed[i] = 0;
	calendar->scores[i] = 0;
	memset (calendar->puzzles[i], ' ', 25);
    }

    /* initialise calendar grid */
    for (i = 0; i < 366; ++i)
	calendar->col[i] = calendar->row[i] = -1;
    for (i = 0; i < 21; ++i)
	for (j = 0; j < 20; ++j)
	    calendar->yday[i][j] = -1;

    /* return the new calendar */
    return calendar;
}
