/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 18-Aug-2024.
 *
 * Graphical Display Handler Header.
 */

/* types defined here */
typedef struct display Display;

#ifndef __DISPLAY_H__
#define __DISPLAY_H__

/* required headers */
#include "game.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/**
 * @enum display_control is a control code.
 */
typedef enum display_control {
    DISPLAY_UP,
    DISPLAY_DOWN,
    DISPLAY_LEFT,
    DISPLAY_RIGHT,
    DISPLAY_FIRE
} DisplayControl;

/**
 * @struct display is the display handler.
 */
typedef struct display_data DisplayData;
struct display {

    /** @var data is the private display data */
    DisplayData *data;

    /**
     * Destroy the display when no longer needed.
     */
    void (*destroy) (void);

    /**
     * Update the screen from the buffer.
     */
    void (*update) (void);

    /**
     * Display a menu and get an option from it.
     * @param count   The number of options in the menu.
     * @param options The option names in an array.
     * @param initial The initial option selected.
     * @return        The number of the option selected.
     */
    int (*menu) (int count, char **options, int initial);

    /**
     * Start a delay timer.
     * @param msecs Number of milliseconds to wait.
     */
    void (*startdelay) (int msecs);

    /**
     * Wait until a delay timer has finished.
     */
    void (*enddelay) (void);

    /**
     * Show the title screen.
     * @param display is the display to affect.
     */
    void (*showtitlescreen) (void);

    /**
     * Await the fire key at the title screen.
     * @param display is the display to affect.
     */
    void (*titlekey) (void);

    /**
     * Make a sound.
     * @param id The ID of the sound.
     */
    void (*playsound) (int id);

    /**
     * Typeset a piece of text and display it.
     * @param text The text to typeset.
     */
    void (*typeset) (char *text);

    /**
     * Show the right-hand screen panel.
     */
    void (*showpanel) (void);

    /**
     * Show the puzzle.
     * @param calendar The calendar.
     * @param game     The current game.
     */
    void (*showpuzzle) (Calendar *calendar, Game *game);

    /**
     * Show the puzzle tiles.
     * @param game The current game.
     */
    void (*showpuzzletiles) (Game *game);

    /**
     * Show a single puzzle tile.
     * @param game The current game.
     * @param row  The row number.
     * @param col  The column number.
     */
    void (*showpuzzletile) (Game *game, int row, int col);

    /**
     * Show the puzzle guesses.
     * @param game The current game.
     */
    void (*showpuzzleguesses) (Game *game);

    /**
     * Show the final puzzle score.
     * @param calendar The calendar object.
     * @param game     The current game.
     */
    void (*showpuzzlescore) (Calendar *calendar, Game *game);

    /**
     * Show the puzzle cursor.
     * @param col Cursor column.
     * @param row Cursor row.
     */
    void (*showpuzzlecursor) (int col, int row);

    /**
     * Hide the puzzle cursor.
     * @param col Cursor column.
     * @param row Cursor row.
     */
    void (*hidepuzzlecursor) (int col, int row);

    /**
     * Show the calendar.
     * @param calendar The calendar to show.
     * @param game     The game object.
     */
    void (*showcalendar) (Calendar *calendar, Game *game);

    /**
     * Show the calendar cursor.
     * @param year The calendar year.
     * @param yday The day of the year.
     */
    void (*showcalcursor) (Calendar *calendar, int yday);

    /**
     * Hide the calendar cursor.
     */
    void (*hidecalcursor) (void);

    /**
     * Build the calendar bitmap.
     * @param calendar The calendar data.
     */
    void (*buildcalendar) (Calendar *calendar);

};

/*----------------------------------------------------------------------
 * Constructor Declarations.
 */

/**
 * Display constructor.
 * @param colourset = 0 for mono, 1 for colour, 2 for nice colour.
 * @param quiet = 0 for sound and music, 1 for silence.
 * @return the new display.
 */
Display *new_Display (int colourset, int quiet);

#endif
