/*======================================================================
 * CGALIB version 2
 * Bitmap handling header.
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 06-Dec-2022.
 */

/* types defined in this header */
typedef struct bitmap Bitmap;

#ifndef __BITMAP_H__
#define __BITMAP_H__

/*----------------------------------------------------------------------
 * Required Headers.
 */

/* standard headers */
#include <stdio.h>

/* project headers */
#include "font.h"

/*----------------------------------------------------------------------
 * Data Declarations.
 */

/**
 * @struct bitmap
 * Object for bitmap handling.
 */
struct bitmap {

    /*
     * Attributes
     */

    /** @var width The width of the bitmap in pixels. */
    int width;

    /** @var height The height of the bitmap in pixels. */
    int height;

    /** @var ink The ink colour for writing and drawing. */
    int ink;

    /** @var paper The paper colour for writing and drawing. */
    int paper;

    /** @var font The current font. */
    Font *font;

    /** @var pixels The pixel data for the bitmap. */
    char *pixels;

    /*
     * Methods
     */

    /**
     * Destroy the bitmap and return to text mode.
     * @param bitmap The bitmap to destroy.
     */
    void (*destroy) (Bitmap *bitmap);

    /**
     * Clone the bitmap.
     * @param  bitmap The bitmap to clone.
     * @return        The new bitmap.
     */
    Bitmap *(*clone) (Bitmap *bitmap);

    /**
     * Write the bitmap to an already open file.
     * @param  bitmap The bitmap to write.
     * @param  output The output file handle.
     * @return        1 on success, 0 on failure.
     */
    int (*write) (Bitmap *bitmap, FILE *output);

    /**
     * Get a bitmap from another bitmap.
     * @param  bitmap The bitmap to extract from.
     * @param  x      The x coordinate on the source bitmap.
     * @param  y      The y coordinate on the destination bitmap.
     * @param  w      The width of the bitmap to exstract.
     * @param  h      The height of the bitmap to extract.
     * @return        The new bitmap.
     */
    Bitmap *(*get) (Bitmap *bitmap, int x, int y, int w, int h);

    /**
     * Put another bitmap onto this one.
     * @param bitmap The destination bitmap.
     * @param source The source bitmap.
     * @param x      The x coordinate on the destination bitmap.
     * @param y      The y coordinate on the destination bitmap.
     * @param mode   The draw mode.
     */
    void (*put) (Bitmap *bitmap, Bitmap *source, int x, int y, int mode);

    /**
     * Put part of another bitmap onto this one.
     * @param bitmap The destination bitmap.
     * @param source The source bitmap.
     * @param xd     The x coordinate on the destination bitmap.
     * @param yd     The y coordinate on the destination bitmap.
     * @param xs     The x coordinate on the source bitmap.
     * @param ys     The y coordinate on the source bitmap.
     * @param w      The width of the area to copy.
     * @param h      The height of the area to copy.
     * @param mode   The draw mode.
     */
    void (*transfer) (Bitmap *bitmap, Bitmap *source, int xd, int yd,
		      int xs, int ys, int w, int h, int draw);

    /**
     * Draw a filled box onto the bitmap.
     * @param bitmap  The bitmap to draw onto.
     * @param x       The x coordinate of the left edge of the box.
     * @param y       The y coordinate of the top edit of the box.
     * @param width   The width of the box.
     * @param height  The height of the box.
     * @param pattern The 16-bit pattern for the box.
     */
    void (*box) (Bitmap *bitmap, int x, int y, int width, int height,
	unsigned int pattern);

    /**
     * Print a message onto the bitmap.
     * @param bitmap  The bitmap to print onto.
     * @param x       The x coordinate of the message.
     * @param y       The y coordinate of the message.
     * @param message The message to print.
     */
    void (*print) (Bitmap *bitmap, int x, int y, char *message);

};

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Construct a new blank bitmap.
 * @param  width  The width of the bitmap.
 * @param  height The height of the bitmap.
 * @return        The new bitmap.
 */
Bitmap *new_Bitmap (int width, int height);

/**
 * Construct a new bitmap read from an already open file.
 * @param  input The input file handle.
 * @return       The new bitmap.
 */
Bitmap *read_Bitmap (FILE *input);

#endif
