/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 17-Aug-2024.
 *
 * Main Program Module
 */

/*----------------------------------------------------------------------
 * Headers
 */

/* Standard ANSI C */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/* Project Headers */
#include "cgalib.h"
#include "pym.h"
#include "fatal.h"
#include "display.h"
#include "controls.h"
#include "calendar.h"
#include "game.h"
#include "ui.h"

/*----------------------------------------------------------------------
 * File Level Variables.
 */

/** @var display options */
static int displayoption = 2;

/** @var quiet Game is silent if 1, or has sound and music if 1. */
static int quiet = 0;

/** @var year The year desired. */
static int year = 0;

/** @var calendar The puzzle calendar. */
static Calendar *calendar = NULL;

/** @var controls The controls object. */
static Controls *controls = NULL;

/** @var display is the display object. */
static Display *display = NULL;

/** @var game The game. */
static Game *game;

/*----------------------------------------------------------------------
 * Private Level 2 Functions.
 */

/**
 * Work out desired program options from command line.
 * @param argc is the command line argument count.
 * @param argv is the array of command line argumets.
 */
static void initialiseargs (int argc, char **argv)
{
    /* local variables */
    int a; /* argument count */
    time_t t; /* current timestamp */
    struct tm tm; /* local time */

    /* get current year */
    t = time (NULL);
    tm = *localtime (&t);
    year = 1900 + tm.tm_year;

    /* look for recognised parameters */
    for (a = 1; a < argc; ++a)
	if (! strcmp (argv[a], "-p") || ! strcmp (argv[a], "-P"))
	    displayoption = 1;
	else if (! strcmp (argv[a], "-m") || ! strcmp (argv[a], "-M"))
	    displayoption = 0;
	else if (! strcmp (argv[a], "-q") || ! strcmp (argv[a], "-Q"))
	    quiet = 1;
	else if (atoi (argv[a]) >= 1900)
	    year = atoi (argv[a]);
	else
	    fatal_error (FATAL_COMMAND_LINE);
}

/**
 * Initialise the puzzle calendar.
 */
static void initialisecalendar (void)
{
    if (! (calendar = new_Calendar (year)))
	fatal_error (FATAL_MEMORY);
    if (! calendar->load (calendar))
	fatal_error (FATAL_CALENDAR);
    display->buildcalendar (calendar);
}

/**
 * Initialise the game.
 */
static void initialisegame (void)
{
    time_t t; /* current timestamp */
    struct tm tm; /* time structure */
    int valid = 0; /* 1 if a valid game was loaded */

    /* what's the date? */
    t = time (NULL);
    tm = *localtime (&t);

    /* reserve memory for game */
    if (! (game = get_Game ()))
	fatal_error (FATAL_MEMORY);

    /* attempt to load a game */
    valid = game->load ();

    /* see if we need to initialise a new game */
    if (valid && game->year != year)
	valid = 0;
    if (valid &&
	year == tm.tm_year + 1900 &&
	calendar->completed[game->yday])
	valid = 0;
    if (valid && year > tm.tm_year + 1900)
	valid = 0;

    /* if no valid game loaded, initialise a new game */
    if (valid)
	return;
    else if (year < tm.tm_year + 1900) {
	game->year = year;
	game->yday = calendar->days - 1;
	game->initialise (calendar);
	game->state = GAME_CALENDAR;
    } else if (year == tm.tm_year + 1900) {
	game->year = year;
	game->yday = tm.tm_yday;
	game->initialise (calendar);
    } else if (year > tm.tm_year + 1900) {
	game->year = year;
	game->yday = 0;
	game->initialise (calendar);
	game->state = GAME_CALENDAR;
    }
}

/*----------------------------------------------------------------------
 * Private Level 1 Functions.
 */

/**
 * Initialise the program.
 * @param argc is the command line argument count.
 * @param argv is the array of command line argumets.
 */
static void initialiseprogram (int argc, char **argv)
{
    /* check command line */
    initialiseargs (argc, argv);

    /* initialise controls */
    if (! (controls = new_Controls ()))
	fatal_error (FATAL_MEMORY);

    /* initialise the display and assets */
    if (! (display = new_Display (displayoption, quiet)))
	fatal_error (FATAL_DISPLAY);
    display->showtitlescreen ();

    /* initialise the puzzle calendar */
    initialisecalendar ();
    initialisegame ();

    /* await the fire key */
    display->titlekey ();
}

/**
 * The main game loop selects a screen to activate.
 */
static int playgame (void)
{
    UI *ui; /* the UI for the current game state */
    int retstate; /* state to return */

    /* construct the current game view */
    switch (game->state) {
    case GAME_PUZZLE:
	ui = new_PuzzleUI ();
	break;
    case GAME_CALENDAR:
	ui = new_CalendarUI ();
	break;
    default:
	return GAME_QUIT;
    }

    /* initialise the current game view */
    if (! ui->init ())
	fatal_error (FATAL_INITUI);

    /* operate the UI */
    retstate = ui->operate ();

    /* return the next game state or STATE_NONE for exit */
    ui->destroy ();
    return retstate;
}

/**
 * Clean up when the user quits normally.
 */
static void endprogram (void)
{
    display->destroy ();
    controls->destroy ();
    calendar->save (calendar) ;
    calendar->destroy (calendar);
    game->save () ;
    game->destroy ();
}

/*----------------------------------------------------------------------
 * Top Level Routine.
 */

/**
 * Share the control handler with other modules.
 * @return A pointer to the game control handler.
 */
Controls *getcontrols (void)
{
    return controls;
}

/**
 * Share the display handler with other modules.
 * @return A pointer to the display module.
 */
Display *getdisplay (void)
{
    return display;
}

/**
 * Share the calendar handler with other modules.
 * @return A pointer to the calendar module.
 */
Calendar *getcalendar (void)
{
    return calendar;
}

/**
 * Share the game handler with other modules.
 * @return A pointer to the game module.
 */
Game *getgame (void)
{
    return game;
}

/**
 * Main Program.
 * @param argc is the count of arguments from the OS.
 * @param argv is an array of arguments from the OS.
 * @returns 0 on a successful run.
 */
int main (int argc, char **argv)
{
    initialiseprogram (argc, argv);
    while (playgame ());
    endprogram ();
    return 0;
}
