/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 17-Aug-2024.
 *
 * Puzzle Calendar Header.
 */

/* type definitions */
typedef struct calendar Calendar;

#ifndef __CALENDAR_H__
#define __CALENDAR_H__

/* required headers */
#include "game.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/**
 * @struct calendar
 * A calendar year of puzzles.
 */
struct calendar {

    /*
     * Attributes
     */

    /** @var year The calendar year. */
    int year;

    /** @var days The number of days in the year. */
    int days;

    /** @var completed 1 if day's puzzle completed, 0 if not. */
    int completed[366];

    /** @var scores Scores for completed puzzles. */
    int scores[366];

    /** @var puzzles The puzzles for each day. */
    char puzzles[366][25];

    /** @var col Column position of each day */
    int col[366];

    /** @var row Row position of each day */
    int row[366];

    /** @var mday Day of month for each day. */
    int mday[366];

    /** @var yday Day for each column/row position */
    int yday[21][20];

    /*
     * Methods
     */

    /**
     * Destroy the calendar object.
     * @param calendar The calendar to destroy.
     */
    void (*destroy) (Calendar *calendar);

    /**
     * Build the calendar grid.
     * @param calendar The calendar to build.
     */
    void (*build) (Calendar *calendar);

    /**
     * Save the calendar file.
     * @param calendar The calendar to save.
     * @return         1 if successful, 0 if not.
     */
    int (*save) (Calendar *calendar);

    /**
     * Load a calendar file.
     * @param calendar The calendar to load.
     * @return         1 if successful, 0 if not.
     */
    int (*load) (Calendar *calendar);

    /**
     * Submit a puzzle.
     * @param calendar The calendar object.
     * @param game     The present game state.
     * @return         1 if puzzle completed, 0 otherwise.
     */
    int (*submit) (Calendar *calendar, Game *game);

    /**
     * Return today's date in day-of-year form.
     * @param calendar The calendar object.
     * @return         The day number, 0..365.
     */
    int (*today) (Calendar *calendar);

};

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Contruct a new calendar.
 * @param year The calendar year.
 * @return     The new calendar object.
 */
Calendar *new_Calendar (int year);

#endif
