/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 24-Aug-2024.
 *
 * Puzzle UI Module.
 */

/*----------------------------------------------------------------------
 * Included Headers.
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* project headers */
#include "pym.h"
#include "ui.h"
#include "display.h"
#include "controls.h"
#include "game.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var ui A pointer to the UI structure. */
static UI *ui;

/** @var display The display object. */
static Display *display;

/** @var controls The controls object. */
static Controls *controls;

/** @var calendar The puzzle calendar. */
static Calendar *calendar;

/** @var game The game object. */
static Game *game;

/** @var cursorx The X coordinate of the cursor. */
static int cursorx;

/** @var cursory The Y coordinate of the cursor. */
static int cursory;

/** @var puzzlemenu The main puzzle screen menu. */
static char *puzzlemenu[] = {
    "Cancel",
    "Reset",
    "Submit",
    "Calendar",
    "Quit game"
};

/** @enum PuzzleMenuID Identifiers for menu options. */
typedef enum {
    PUZZLEMENU_CANCEL,
    PUZZLEMENU_RESET,
    PUZZLEMENU_SUBMIT,
    PUZZLEMENU_CALENDAR,
    PUZZLEMENU_QUITGAME,
    PUZZLEMENU_COUNT
} PuzzleMenuID;

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Allow the user to move the cursor.
 */
static void movecursor (void)
{
    /* loop until menu is requested */
    do {

	/* show the cursor and wait for input */
	display->showpuzzlecursor (cursorx, cursory);
	display->update ();
	controls->release (250);
	controls->wait (0);
	controls->poll ();

	/* horizontal movement */
	if (controls->left && cursorx > 0)
	    display->hidepuzzlecursor (cursorx--, cursory);
	else if (controls->right && cursorx < 4)
	    display->hidepuzzlecursor (cursorx++, cursory);

	/* vertical movement */
	if (controls->up && cursory > 0)
	    display->hidepuzzlecursor (cursorx, cursory--);
	else if (controls->down && cursory < 4)
	    display->hidepuzzlecursor (cursorx, cursory++);

    } while (! controls->fire &&
	     ! isalpha (controls->ascii) &&
	     ! (controls->ascii == 8 && ! controls->left) &&
	     ! (controls->ascii == 127));
}

/**
 * Reset the letter grid.
 */
static void resetpuzzle (void)
{
    int r, /* row counter */
	c; /* column counter */
    for (r = 0; r < 5; ++r)
	for (c = 0; c < 5; ++c)
	    if (game->status[c + 5 * r] == GAME_FILLED) {
		game->letter[c + 5 * r] = 0;
		game->status[c + 5 * r] = GAME_BLANK;
		game->filled--;
		display->showpuzzletile (game, r, c);
	    }
    display->update ();
}

/**
 * Submit the puzzle.
 */
static void submitpuzzle (void)
{
    calendar->submit (calendar, game);
    display->showpuzzletiles (game);
    if (calendar->completed[game->yday])
	display->showpuzzlescore (calendar, game);
    else
	display->showpuzzleguesses (game);
    display->update ();
}

/**
 * Put a letter on the grid.
 * @param letter The letter code.
 */
static void putletter (char letter)
{
    if (game->status[cursorx + 5 * cursory] == GAME_BLANK ||
	game->status[cursorx + 5 * cursory] == GAME_FILLED)
    {
	if (game->status[cursorx + 5 * cursory] == GAME_BLANK)
	    game->filled += 2;
	game->letter[cursorx + 5 * cursory]
	    = game->letter[cursory + 5 * cursorx]
	    = letter;
	game->status[cursorx + 5 * cursory]
	    = game->status[cursory + 5 * cursorx]
	    = GAME_FILLED;
	display->showpuzzletile (game, cursorx, cursory);
	display->showpuzzletile (game, cursory, cursorx);
	display->update ();
    }
}

/**
 * Erase a letter from the grid.
 */
static void eraseletter (void)
{
    if (game->status[cursorx + 5 * cursory] == GAME_FILLED) {
	game->filled -= 2;
	game->letter[cursorx + 5 * cursory]
	    = game->letter[cursory + 5 * cursorx]
	    = 0;
	game->status[cursorx + 5 * cursory]
	    = game->status[cursory + 5 * cursorx]
	    = GAME_BLANK;
	display->showpuzzletile (game, cursorx, cursory);
	display->showpuzzletile (game, cursory, cursorx);
	display->update ();
    }
}

/**
 * Return the default option.
 */
static int defaultoption (void)
{
    if (game->filled == 25 && calendar->completed[game->yday])
	return PUZZLEMENU_CALENDAR;
    else if (game->filled == 25)
	return PUZZLEMENU_SUBMIT;
    else
	return PUZZLEMENU_CANCEL;
}

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Initialise the UI.
 * @return   1 on success, 0 on failure.
 */
static int init (void)
{
    /* initialise module-wide variables */
    calendar = getcalendar ();
    game = getgame ();
    display = getdisplay ();
    controls = getcontrols ();

    /* initialise cursor position */
    cursorx = 0;
    cursory = 0;

    /* show the initial display */
    display->showpanel ();
    display->showpuzzle (calendar, game);
    display->update ();

    /* return success */
    return 1;
}

/**
 * Operate the UI.
 * @return The next game state.
 */
static int operate (void)
{
    int option; /* menu option chosen */
    do {

	/* get cursor position */
	movecursor ();

	/* launch menu */
	if (controls->fire) {
	    option = display->menu
		(PUZZLEMENU_COUNT,
		 puzzlemenu,
		 defaultoption ());
	    switch (option) {
	    case PUZZLEMENU_RESET:
		resetpuzzle ();
		break;
	    case PUZZLEMENU_SUBMIT:
		submitpuzzle ();
		break;
	    case PUZZLEMENU_CALENDAR:
		return (game->state = GAME_CALENDAR);
		break;
	    case PUZZLEMENU_QUITGAME:
		return GAME_QUIT;
	    }
	}

	/* enter a letter */
	else if (isalpha (controls->ascii))
	    putletter (toupper (controls->ascii));

	/* remove a letter */
	else if ((controls->ascii == 8 && ! controls->left) ||
		 controls->ascii == 127)
	    eraseletter ();

    } while (1);
}

/**
 * Construct Puzzle UI
 * @return The new UI.
 */
UI *new_PuzzleUI (void)
{
    /* reserve memory for the UI */
    if (! (ui = new_UI ()))
	return NULL;

    /* initialise methods */
    ui->init = init;
    ui->operate = operate;

    /* return the UI */
    return ui;
}
