/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 18-Aug-2024.
 *
 * Game Controls Module
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/timeb.h>

/* project headers */
#include "keyboard.h"
#include "controls.h"
#include "fatal.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var controls A pointer to the one game controls handler. */
static Controls *this;

/** @var keys The keyboard handler. */
static KeyHandler *keys;

/*----------------------------------------------------------------------
 * Public Method Function Definitions.
 */

/**
 * Destroy the keyboard handler when it is no longer needed.
 */
static void destroy (void)
{
    if (this) {
	free (this);
	this = NULL;
	if (keys)
	    keys->destroy ();
	fatal_controls (NULL);
    }
}

/**
 * Expose the KeyHandler for other libraries that need it.
 * @return A pointer to the KeyHandler object.
 */
static KeyHandler *getkeyhandler (void)
{
    return keys;
}

/**
 * Wait for a key.
 * @param msecs Number of milliseconds to wait, 0 forever.
 */
static void wait (int msecs)
{
    int keydown, /* 1 if a key is down */
	k, /* key scancode counter */
	remaining; /* time remaining (negative) */
    struct timeb start, /* time we started waiting */
	end, /* time we will finish waiting */
	now; /* time now */

    /* initialise the timer */
    ftime (&start);
    end = start;
    end.millitm += msecs;
    while (end.millitm > 1000) {
	++end.time;
	end.millitm -= 1000;
    }
    
    /* main wait loop */
    do {

	/* check all the keys */
	keydown = 0;
	for (k = 0x01; k <= 0x54; ++k)
	    if (keys->key (k))
		keydown = 1;

	/* check the timer */
	ftime (&now);
	remaining = msecs ?
	    (now.time - end.time) * 1000 + now.millitm - end.millitm :
	    -1;

    } while (! keydown && remaining < 0);
}

/**
 * Wait for all keys to be released.
 * @param msecs Number of milliseconds to wait, 0 forever.
 */
static void release (int msecs)
{
    int keydown, /* 1 if a key is down */
	k, /* key scancode counter */
	remaining; /* time remaining (negative) */
    struct timeb start, /* time we started waiting */
	end, /* time we will finish waiting */
	now; /* time now */

    /* initialise the timer */
    ftime (&start);
    end = start;
    end.millitm += msecs;
    while (end.millitm > 1000) {
	++end.time;
	end.millitm -= 1000;
    }
    
    /* main wait loop */
    do {

	/* check all the keys */
	keydown = 0;
	for (k = 0x01; k <= 0x54; ++k)
	    if (keys->key (k))
		keydown = 1;

	/* check the timer */
	ftime (&now);
	remaining = msecs ?
	    (now.time - end.time) * 1000 + now.millitm - end.millitm :
	    -1;

    } while (keydown && remaining < 0);
}

/**
 * Poll the keyboard and update the attributes.
 */
static void poll (void)
{
    this->left = keys->key (KEY_KP4);
    this->right = keys->key (KEY_KP6);
    this->up = keys->key (KEY_KP8);
    this->down = keys->key (KEY_KP2);
    this->fire = keys->key (KEY_CTRL) ||
	keys->key (KEY_ENTER) ||
	keys->key (KEY_SPACE);
    this->ascii = keys->ascii ();
}

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Construct new game controls handler.
 * @return The new game controls handler.
 */
Controls *new_Controls (void)
{
    /* ensure only one control hander exists */
    if (this)
	return this;
    if (! (this = malloc (sizeof (Controls))))
	return NULL;

    /* initialise attributes */
    this->left = 0;
    this->right = 0;
    this->up = 0;
    this->down = 0;
    this->fire = 0;
    this->ascii = 0;

    /* initialise methods */
    this->destroy = destroy;
    this->getkeyhandler = getkeyhandler;
    this->wait = wait;
    this->release = release;
    this->poll = poll;

    /* initialise the keyhandler */
    keys = new_KeyHandler ();
    fatal_controls (this);

    /* return the new control handler */
    return this;
}
