/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 17-Aug-2024.
 *
 * Game Header
 */

/* type definitions */
typedef struct game Game;
typedef struct guess Guess;

#ifndef __GAME_H__
#define __GAME_H__

/* required headers */
#include "calendar.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/**
 * @enum GameState
 * State of the game.
 */
typedef enum {
    GAME_QUIT, /* player has selected quit */
    GAME_CALENDAR, /* player is viewing the calendar */
    GAME_PUZZLE /* player is viewing a puzzle */
} GameState;

/**
 * @enum GameTileStatus
 * State of a game tile.
 */
typedef enum {
    GAME_CLUE, /* the tile is a clue tile */
    GAME_BLANK, /* the tile is blank */
    GAME_FILLED, /* the tile has been filled in */
    GAME_VERIFIED /* the tile has been verified */
} GameTileStatus;

/**
 * @struct game
 * A game in progress.
 */
struct game {

    /*
     * Attributes
     */

    /** @var state The current game state. */
    int state;

    /** @var year The year of the puzzle. */
    int year;

    /** @var day Day of the current puzzle, 0..365. */
    int yday;

    /** @var grid The currently filled in letters. */
    char letter[25];

    /** @var status The status of each letter. */
    int status[25];

    /** @var blank A count of the number of squares filled. */
    int filled;

    /** @var guesscount The number of reject words. */
    int guesscount;

    /** @var guesses An array of guesses. */
    Guess **guesses;

    /*
     * Methods
     */

    /**
     * Destroy the game object.
     */
    void (*destroy) (void);

    /**
     * Save the game object to a file.
     * @return 1 if successful, 0 if not.
     */
    int (*save) (void);

    /**
     * Load the game object from a file.
     * @return 1 if successful, 0 if not.
     */
    int (*load) (void);

    /**
     * Initialise a new puzzle for a given day.
     * @param game     The game object.
     * @param calendar The calendar object.
     */
    void (*initialise) (Calendar *calendar);

    /**
     * Add a word to the list of guesses.
     * @param word    The word to add.
     * @param correct 1 if correct, 0 if incorrect.
     * @return        1 if added, 0 if already there.
     */
    int (*add) (char *word, int correct);

};

/**
 * @struct guess
 * A guessed word.
 */
struct guess {

    /** @var word The guessed word. */
    char word[6];

    /** @var correct 1 if correct, 0 if not. */
    int correct;

};

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Contruct a new game object, or return the existing one.
 * @return The new game object.
 */
Game *get_Game (void);


#endif
