/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 19-Aug-2024.
 *
 * Create all word squares in text format, from a dictionary fed to
 * standard input. Print the word squares to standard output.
 */

/*----------------------------------------------------------------------
 * Required headers
 */

/* standard C headers */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var dict The internal dictionary. */
static char **dict;

/** @var len_dict The number of words in the internal dictionary. */
static int len_dict;

/** @var square The currently constructed word square. */
static char *square[5];

/*----------------------------------------------------------------------
 * Level 2 Function Definitions.
 */

/**
 * Add a word to the internal dictionary, filtering out proper nouns,
 * words with punctuation, etc.
 * @param line The line read in from the dictionary source.
 */
void addword (char *line)
{
    /* check that this is a 5 letter word */
    if (line[5] != '\n')
	return;
    if (! islower (line[0]) ||
	! islower (line[1]) ||
	! islower (line[2]) ||
	! islower (line[3]) ||
	! islower (line[4]))
	return;

    /* add the word to the dictionary */
    line[5] = '\0';
    if (len_dict)
	dict = realloc (dict, (len_dict + 1) * sizeof (char *));
    else
	dict = malloc (sizeof (char *));
    dict[len_dict] = malloc (6);
    strcpy (dict[len_dict++], line);
}

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Load the internal dictionary from a plain text source,
 * filtering out anything that's not 5 characters long.
 */
static void loaddict (void)
{
    char line[1024]; /* line read in from file */
    while (fgets (line, 1023, stdin)) {
	if (strlen (line) == 6)
	    addword (line);
    }
}

/**
 * Find all words that fit the current row/column.
 * Recurse to find fits for lower rows in the same way.
 * Output the complete squares found along the way.
 * @param row The row number 0..4.
 */
static void findfit (int row)
{
    char partial[5];
    int c; /* counter */
    for (c = 0; c < row; ++c)
	partial[c] = square[c][row];
    partial[row] = '\0';
    for (c = 0; c < len_dict; ++c)
	if (! strncmp (dict[c], partial, row)) {
	    square[row] = dict[c];
	    if (row == 4)
		printf
		    ("%s/%s/%s/%s/%s\n",
		     square[0],
		     square[1],
		     square[2],
		     square[3],
		     square[4]);
	    else
		findfit (row + 1);
	}
}

/**
 * Clean up.
 */
void cleanup (void)
{
    int c; /* counter */
    if (len_dict) {
	for (c = 0; c < len_dict; ++c)
	    free (dict[c]);
	free (dict);
    }
}

/*----------------------------------------------------------------------
 * Top Level Function Definition
 */

/**
 * Main Function.
 */
int main (void)
{
    loaddict ();
    findfit (0);
    cleanup ();
    return 0;
}
