/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 20-Aug-2024.
 *
 * Puzzle generator module
 */

/*----------------------------------------------------------------------
 * Headers
 */

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* project headers */
#include "calendar.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var calendar The calendar for the year. */
static Calendar *calendar;

/*----------------------------------------------------------------------
 * Level 1 Routines.
 */

/**
 * Read a year's worth of puzzles.
 * @param year The year to read.
 */
static int readyear (void)
{
    char filename[128], /* name of input file */
	puzzle[32]; /* puzzle line from input file */
    int d, /* day counter */
	c; /* character counter */
    FILE *input; /* input file */

    /* ascertain filename */
    sprintf (filename, "assets/%d.txt", calendar->year);

    /* attempt to open file */
    if (! (input = fopen (filename, "r"))) {
	printf ("Cannot open file %s\n", filename);
	exit (2);
    }

    /* read the squares */
    for (d = 0; d < calendar->days; ++d) {
	if (! fgets (puzzle, 32, input)) {
	    fclose (input);
	    return 0;
	}
	for (c = 0; c < strlen (puzzle); ++c)
	    puzzle[c] = toupper (puzzle[c]);
	memcpy (calendar->puzzles[d], puzzle, 5);
	memcpy (&calendar->puzzles[d][5], &puzzle[6], 5);
	memcpy (&calendar->puzzles[d][10], &puzzle[12], 5);
	memcpy (&calendar->puzzles[d][15], &puzzle[18], 5);
	memcpy (&calendar->puzzles[d][20], &puzzle[24], 5);
    }

    /* close the file */
    fclose (input);
    return 1;
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Main Program.
 * @param argc Argument count.
 * @param argv Argument array.
 * @return     Result code.
 */
int main (int argc, char **argv)
{
    int year; /* calendar year */

    /* validate year */
    if (argc != 2 ||
	! (year = atoi (argv[1])))
    {
	printf ("Usage: %s YEAR\n", argv[0]);
	exit (1);
    }

    /* create calendar */
    if (! (calendar = new_Calendar (year))) {
	printf ("Cannot initialise calendar: insufficient memory\n");
	exit (2);
    }

    /* copy the puzzles */
    if (! readyear ()) {
	printf
	    ("Cannot load %d puzzles from input file\n",
	     calendar->days);
	exit (3);
    }
    if (! calendar->save (calendar)) {
	printf ("Cannot save output file\n");
	exit (4);
    }

    /* return */
    calendar->destroy (calendar);
    return 0;
}
