/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 20-Aug-2024.
 *
 * Graphical Display Handler Module.
 */

/*----------------------------------------------------------------------
 * Headers.
 */

/* standard C headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include <sys/timeb.h>

/* project-specific headers */
#include "cgalib.h"
#include "pym.h"
#include "fatal.h"
#include "display.h"
#include "controls.h"
#include "game.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var display The display object */
static Display *display = NULL;

/** @var screen is the CGALIB screen data. */
static Screen *screen;

/** @var controls A pointer to the game control handler. */
static Controls *controls;

/** @var titleimage is the title screen image bitmap */
static Bitmap *titleimage;

/** @var calendarbg Calendar background image. */
static Bitmap *calendarbg;

/** @var puzzlebg Puzzle background image. */
static Bitmap *puzzlebg;

/** @var panel Panel image. */
static Bitmap *panel;

/** @var menupull Menu ring pull image. */
static Bitmap *menupull;

/** @var menumask Menu ring pull mask. */
static Bitmap *menumask;

/** @var datecursor Date cursor image. */
static Bitmap *datecursor;

/** @var datemask Date cursor mask. */
static Bitmap *datemask;

/** @var tick Calendar tick. */
static Bitmap *tick;

/** @var tickmask Calendar tick mask. */
static Bitmap *tickmask;

/** @var lettercursor Letter cursor image. */
static Bitmap *lettercursor;

/** @var lettermask Letter cursor mask. */
static Bitmap *lettermask;

/** @var calfont The calendar font. */
static Font *calfont;

/** @var futurefont The font for future calendar dates. */
static Font *futurefont;

/** @var font Standard white-on-green font. */
static Font *font;

/** @var greenfont The green-on-white font. */
static Font *greenfont;

/** @var redfont The white-on-red font. */
static Font *redfont;

/** @var strikefont The strike-through font. */
static Font *strikefont;

/** @var cluefont The tile font for clue letters. */
static Font *cluefont;

/** @var guessfont The tile font for letter guesses. */
static Font *guessfont;

/** @var correctfont The tile font for correct letters. */
static Font *correctfont;

/** @var tune The intro tune. */
static Tune *tune;

/** @var click The click sound effect. */
/* static Effect *click; */

/** @var gamesound 1 if there is game sound. */
static int gamesound;

/** @var monthnames The calendar month names, padded */
static char *monthnames[] = {
    "JANUARY  ",
    "FEBRUARY ",
    "MARCH    ",
    "APRIL    ",
    "MAY      ",
    "JUNE     ",
    "JULY     ",
    "AUGUST   ",
    "SEPTEMBER",
    "OCTOBER  ",
    "NOVEMBER ",
    "DECEMBER "
};

/** @var daynames The day names, padded */
static char *daynames[] = {
    "   SUNDAY",
    "   MONDAY",
    "  TUESDAY",
    "WEDNESDAY",
    " THURSDAY",
    "   FRIDAY",
    " SATURDAY",
    "   SUNDAY"
};

/** @var instructions Instruction text on the puzzle page. */
static char *instructions[] = {
    "A word square is a 5x5",
    "grid of letters,",
    "containing five words;",
    "the same words are",
    "spelled across and",
    "down. Your task is to",
    "complete the word square",
    "puzzle set today. Arrow",
    "keys move; letters type;",
    "hold SPACE for menu."
};

/** @var scoretext The score text. */
static char *scoretext[] = {
    "Congratulations! You",
    "have solved this puzzle.",
    "",
    "       Score: %d",
    "",
    "Select Calendar from the",
    "menu to see if there are",
    "puzzles to complete for ",
    "other days.",
    ""
};

/** @var delaytime end of a delay period */
static struct timeb delaytime;

/**
 * @var behindcal
 * What was behind the calendar cursor before we showed it.
 */
static Bitmap *behindcal;

/** @var calx X coordindate of the calendar cursor. */
static int calx;

/** @var caly Y coordinate of the calendar cursor. */
static int caly;

/** @var calbitmap The pre-built calendar */
static Bitmap *calbitmap;

/**
 * @var completed 
 * Puzzles already marked as completed on the calendar.
 */
static int completed[366];

/** @var menubitmap Bitmap containing the menu. */
static Bitmap *menubitmap;

/*----------------------------------------------------------------------
 * Level 2 Private Function Definitions.
 */

/**
 * Show a frame of the menu animation.
 * @param count   The number of options in the menu.
 * @param options The option names in an array.
 * @param option  The option selected.
 * @param anim    Frame of animation.
 */
static void showmenuanimation
(int count,
 char **options,
 int option,
 int anim)
{
    /* frame should last about 1/20 second */
    display->startdelay (50);

    /* show the ring pull */
    screen->transfer
	(screen,
	 panel,
	 272, 176 - 8 * anim,
	 0, 176 - 8 * anim,
	 48, 16,
	 CGALIB_SET);
    screen->put (screen, menumask, 272, 176 - 8 * anim, CGALIB_AND);
    screen->put (screen, menupull, 272, 176 - 8 * anim, CGALIB_OR);

    /* show the options */
    screen->transfer
	(screen, menubitmap,
	 272, 192 - 8 * anim,
	 0, 0,
	 48, 8 * (1 + anim),
	 CGALIB_SET);

    /* update the display */
    display->update ();
    display->enddelay ();
}

/*----------------------------------------------------------------------
 * Level 1 Private Function Definitions.
 */

/**
 * Load the graphical assets.
 */
static void loadassets (void)
{
    FILE *input; /* input file handle */
    char header[8]; /* header buffer */
    Bitmap *logo; /* the Cyningstan logo */
    time_t start; /* time the Cyningstan logo was displayed */

    /* attempt to open the file and check the header */
    if (! (input = fopen ("PYM.DAT", "rb")))
	fatal_error (FATAL_NODATA);
    if (! fread (header, 8, 1, input))
	fatal_error (FATAL_INVALIDDATA);
    if (strcmp (header, "PYM001D"))
	fatal_error (FATAL_INVALIDDATA);

    /* load and display the Cyningstan logo */
    start = time (NULL);
    if (! (logo = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    screen->put (screen, logo, 96, 92, CGALIB_SET);
    logo->destroy (logo);
    screen->update (screen);

    /* load the title screen */
    if (! (titleimage = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);

    /* load the background screen components */
    if (! (calendarbg = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (puzzlebg = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (panel = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);

    /* load the miscellaneous bitmaps and masks */
    if (! (menupull = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (menumask = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (datecursor = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (datemask = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (tick = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (tickmask = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (lettercursor = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (lettermask = read_Bitmap (input)))
	fatal_error (FATAL_INVALIDDATA);

    /* load standard calendar font */
    if (! (calfont = read_Font (input, 2)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (futurefont = calfont->clone (calfont)))
	fatal_error (FATAL_MEMORY);
    futurefont->recolour (futurefont, 2, 3);
    calfont->recolour (calfont, 0, 3);

    /* load the normal font */
    if (! (font = read_Font (input, 2)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (greenfont = font->clone (font)))
	fatal_error (FATAL_MEMORY);
    greenfont->recolour (greenfont, 0, 3);
    if (! (redfont = font->clone (font)))
	fatal_error (FATAL_MEMORY);
    redfont->recolour (redfont, 3, 2);

    /* load the strikethrough and tile fonts */
    if (! (strikefont = read_Font (input, 2)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (cluefont = read_Font (input, 2)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (guessfont = read_Font (input, 2)))
	fatal_error (FATAL_INVALIDDATA);
    if (! (correctfont = read_Font (input, 2)))
	fatal_error (FATAL_INVALIDDATA);

    /* close the input file */
    fclose (input);

    /* clear the logo after at least three seconds */
    while (time (NULL) < start + 4);
    screen->box (screen, 96, 92, 128, 16, CGALIB_BOX_BLANK);
    screen->update (screen);
    while (time (NULL) < start + 5);
}

/**
 * Open the menu with animation.
 * @param count   The number of options in the menu.
 * @param options The option names in an array.
 * @param option  The option selected.
 */
static void openmenu (int count, char **options, int option)
{
    int a; /* animation counter */
    char str[13]; /* constructed string */
    menubitmap = new_Bitmap (48, 8 * count);
    menubitmap->font = greenfont;
    for (a = 0; a < count; ++a) {
	sprintf (str, " %-10.10s ", options[a]);
	menubitmap->print (menubitmap, 0, 8 * a, str);
    }
    for (a = 0; a < count; ++a)
	showmenuanimation (count, options, option, a);
}

/**
 * Highlight the current menu option
 * @param count   The number of options in the menu.
 * @param options The option names in an array.
 * @param option  The option selected.
 */
static void showmenuoption
(int count,
 char **options,
 int option,
 int highlight)
{
    char text[13]; /* menu option text */
    screen->font = highlight ? redfont : greenfont;
    sprintf (text, " %-10.10s ", options[option]);
    screen->print
	(screen, 272, 200 - 8 * count + 8 * option, text);
}

/**
 * Close the menu with animation.
 * @param count   The number of options in the menu.
 * @param options The option names in an array.
 * @param option  The option selected.
 */
static void closemenu (int count, char **options, int option)
{
    int a; /* animation counter */
    for (a = count - 1; a >= 0; --a) {
	showmenuanimation (count, options, option, a);
	screen->transfer
	    (screen,
	     panel,
	     272, 176 - 8 * a,
	     0, 176 - 8 * a,
	     48, 16,
	     CGALIB_SET);
	display->update ();
    }
    screen->transfer
	(screen, panel, 272, 184, 0, 184, 48, 16, CGALIB_SET);
    display->update ();
    menubitmap->destroy (menubitmap);
}

/**
 * Display a date on the puzzle screen.
 * @param year The date year.
 * @param yday The day of the year.
 */
static void showdate (int year, int yday)
{
    struct tm tm; /* time structure */
    char numstr[2]; /* digit as string */

    /* get a calendar date from the year day */
    tm.tm_sec = 0;
    tm.tm_min = 0;
    tm.tm_hour = 0;
    tm.tm_mday = yday + 1;
    tm.tm_mon = 0;
    tm.tm_year = year - 1900;
    tm.tm_wday = 0;
    tm.tm_yday = 0;
    tm.tm_isdst = -1;
    mktime (&tm);

    /* display the date in the calendar */
    screen->font = greenfont;
    screen->print (screen, 28, 16, daynames[tm.tm_wday]);
    sprintf (numstr, "%d", tm.tm_mday / 10);
    screen->print (screen, 72, 16, numstr);
    sprintf (numstr, "%d", tm.tm_mday % 10);
    screen->print (screen, 84, 16, numstr);
    screen->print (screen, 96, 16, monthnames[tm.tm_mon]);
}

/**
 * Work out screen location of a give date.
 * @param calendar The calendar object.
 * @param yday     The day of the year.
 * @param x        Pointer to the x coordinate variable.
 * @param y        Pointer to the y coordinate variable.
 */
static void getcalendarcoords
(Calendar *calendar, int yday, int *x, int *y)
{
    *x = 8 + 12 * calendar->col[yday]
	+ 4 * (calendar->col[yday] >= 7)
	+ 4 * (calendar->col[yday] >= 14);
    *y = 18 + 6 * calendar->row[yday]
	+ 10 * (calendar->row[yday] >= 5)
	+ 26 * (calendar->row[yday] >= 10)
	+ 10 * (calendar->row[yday] >= 15);
}

/*----------------------------------------------------------------------
 * Public Methods.
 */

/**
 * Destroy the display when no longer needed.
 */
static void destroy (void)
{
    /* clean up the display */
    if (display) {

	/* back to text mode */
	screen->destroy (screen);

	/* destroy any assets that were created */
	if (titleimage)
	    titleimage->destroy (titleimage);
	if (calendarbg)
	    calendarbg->destroy (calendarbg);
	if (puzzlebg)
	    puzzlebg->destroy (puzzlebg);
	if (panel)
	    panel->destroy (panel);
	if (menupull)
	    menupull->destroy (menupull);
	if (menumask)
	    menumask->destroy (menumask);
	if (datecursor)
	    datecursor->destroy (datecursor);
	if (datemask)
	    datemask->destroy (datemask);
	if (tick)
	    tick->destroy (tick);
	if (tickmask)
	    tickmask->destroy (tickmask);
	if (lettercursor)
	    lettercursor->destroy (lettercursor);
	if (lettermask)
	    lettermask->destroy (lettermask);
	if (calfont)
	    calfont->destroy (calfont);
	if (futurefont)
	    futurefont->destroy (futurefont);
	if (font)
	    font->destroy (font);
	if (greenfont)
	    greenfont->destroy (greenfont);
	if (redfont)
	    redfont->destroy (redfont);
	if (strikefont)
	    strikefont->destroy (strikefont);
	if (cluefont)
	    cluefont->destroy (cluefont);
	if (guessfont)
	    guessfont->destroy (guessfont);
	if (correctfont)
	    correctfont->destroy (correctfont);

	/* destroy other temporary data */
	if (behindcal)
	    behindcal->destroy (behindcal);
	if (calbitmap)
	    calbitmap->destroy (calbitmap);

	/* destroy the display */
    	free (display);
    }
}

/**
 * Update the screen from the buffer.
 */
static void update (void)
{
    screen->update (screen);
}

/**
 * Display a menu and get an option from it.
 * @param count   The number of options in the menu.
 * @param options The option names in an array.
 * @param initial The initial option selected.
 * @return        The number of the option selected.
 */
static int menu (int count, char **options, int initial)
{
    int option; /* option currently selected */

    /* initialise the menu */
    option = initial;
    openmenu (count, options, option);
    showmenuoption (count, options, option, 1);
    display->update ();

    /* allow option selection till fire pressed/released */
    controls->poll ();
    while (controls->fire) {

	/* up key pressed */
	if (controls->up && option > 0) {
	    showmenuoption (count, options, option, 0);
	    --option;
	    showmenuoption (count, options, option, 1);
	    display->update ();
	    do
		controls->poll ();
	    while (controls->up);
	}

	/* down key pressed */
	else if (controls->down && option < count - 1) {
	    showmenuoption (count, options, option, 0);
	    ++option;
	    showmenuoption (count, options, option, 1);
	    display->update ();
	    do
		controls->poll ();
	    while (controls->down);
	}

	/* check the keyboard again */
	controls->poll ();
    }

    /* clean up and return */
    closemenu (count, options, option);
    do
	controls->poll ();
    while (controls->fire);
    return option;
}

/**
 * Start a delay timer.
 */
static void startdelay (int msecs)
{
    struct timeb now; /* time now */
    ftime (&now);
    delaytime = now;
    delaytime.millitm += msecs;
    while (delaytime.millitm > 1000) {
	++delaytime.time;
	delaytime.millitm -= 1000;
    }
}

/**
 * Wait until a delay timer has finished.
 */
static void enddelay (void)
{
    struct timeb now; /* time now */
    do
	ftime (&now);
    while (now.time < delaytime.time ||
	     (now.time == delaytime.time &&
	      now.millitm <= delaytime.millitm));
}

/**
 * Show the title screen.
 * @param display is the display to affect.
 */
static void showtitlescreen (void)
{
    screen->put (screen, titleimage, 0, 0, CGALIB_SET);
    screen->update (screen);
}

/**
 * Await the fire key at the title screen.
 * @param display is the display to affect.
 */
static void titlekey (void)
{
    int key; /* value of key pressed */

    /* show the"Press FIRE" message */
    screen->font = font;
    screen->ink = 3;
    screen->print (screen, 128, 184, "  Press SPACE.  ");
    screen->update (screen);

    /* wait for a key until FIRE is pressed */
    do {
	if (gamesound) {
	    tune->play (tune, controls->getkeyhandler ());
	    while (! controls->getkeyhandler ()->anykey ());
	} else
	    while (! controls->getkeyhandler ()->anykey ());
	key = controls->getkeyhandler ()->scancode ();
    } while (key != KEY_CTRL && key != KEY_ENTER && key != KEY_SPACE);

    /* erase the "Press FIRE" message */
    screen->box (screen, 128, 184, 64, 8, CGALIB_BOX_BLANK);
    screen->update (screen);
    do
	controls->poll ();
    while (controls->fire);
}

/**
 * Make a sound.
 * @param id The ID of the sound.
 */
static void playsound (int id)
{
}

/**
 * Typeset a piece of text and display it.
 * @param text The text to typeset.
 */
static void typeset (char *text)
{
}

/**
 * Show the right-hand screen panel.
 */
static void showpanel (void)
{
    screen->put (screen, panel, 272, 0, CGALIB_SET);
}

/**
 * Show the puzzle.
 * @param calendar The calendar.
 * @param game     The current game.
 */
static void showpuzzle (Calendar *calendar, Game *game)
{
    int r; /* row counter */

    /* show background */
    screen->put (screen, puzzlebg, 0, 0, CGALIB_SET);

    /* show puzzle date */
    showdate (game->year, game->yday);

    /* show puzzle grid */
    display->showpuzzletiles (game);

    /* show notepad instructions */
    screen->font = greenfont;
    for (r = 0; r < 10; ++r)
	screen->print (screen, 168, 12 + 8 * r, instructions[r]);

    /* show existing guesses or final score */
    if (calendar->completed[game->yday])
	display->showpuzzlescore (calendar, game);
    else
	display->showpuzzleguesses (game);
}

/**
 * Show the puzzle tiles.
 * @param game The current game.
 */
static void showpuzzletiles (Game *game)
{
    int r, /* row counter */
	c; /* column counter */
    for (r = 0; r < 5; ++r)
	for (c = 0; c < 5; ++c)
	    display->showpuzzletile (game, r, c);
}

/**
 * Show a single puzzle tile.
 * @param game The current game.
 * @param row  The row number.
 * @param col  The column number.
 */
static void showpuzzletile (Game *game, int row, int col)
{
    char str[2]; /* character as string */
    /* decide the correct font (if any) */
    switch (game->status[col + 5 * row]) {
    case GAME_BLANK:
	screen->transfer
	    (screen, puzzlebg,
	     32 * col, 40 + 32 * row,
	     32 * col, 40 + 32 * row,
	     32, 32, CGALIB_SET);
	return;
    case GAME_CLUE:
	screen->font = cluefont;
	break;
    case GAME_FILLED:
	screen->font = guessfont;
	break;
    case GAME_VERIFIED:
	screen->font = correctfont;
	break;
    }

    /* show the letter */
    sprintf (str, "%c", game->letter[col + 5 * row]);
    screen->print (screen, 32 * col, 40 + 32 * row, str);
}

/**
 * Show the puzzle guesses.
 * @param game The current game.
 */
static void showpuzzleguesses (Game *game)
{
    int w; /* word counter */
    screen->box (screen, 168, 108, 96, 80, CGALIB_BOX_FILLED);
    for (w = 0; w < game->guesscount && w < 40; ++w) {
	screen->font
	    = game->guesses[w]->correct
	    ? greenfont
	    : strikefont;
	screen->print
	    (screen,
	     168 + 24 * (w / 10),
	     108 + 8 * (w % 10),
	     game->guesses[w]->word);
    }
}

/**
 * Show the final puzzle score.
 * @param calendar The calendar object.
 * @param game The current game.
 */
static void showpuzzlescore (Calendar *calendar, Game *game)
{
    int c; /* line counter */
    char text[25]; /* line of text */
    screen->box (screen, 168, 108, 96, 80, CGALIB_BOX_FILLED);
    screen->font = greenfont;
    for (c = 0; c < 10; ++c) {
	if (strchr (scoretext[c], '%'))
	    sprintf (text, scoretext[c], calendar->scores[game->yday]);
	else
	    strcpy (text, scoretext[c]);
	screen->print (screen, 168, 108 + 8 * c, text);
    }
}

/**
 * Show the puzzle cursor.
 * @param col Cursor column.
 * @param row Cursor row.
 */
static void showpuzzlecursor (int col, int row)
{
    screen->put (screen, lettermask, 32 * col, 40 + 32 * row, CGALIB_AND);
    screen->put (screen, lettercursor, 32 * col, 40 + 32 * row, CGALIB_OR);
}

/**
 * Hide the puzzle cursor.
 * @param col Cursor column.
 * @param row Cursor row.
 */
static void hidepuzzlecursor (int col, int row)
{
    screen->put (screen, lettermask, 32 * col, 40 + 32 * row, CGALIB_AND);
}

/**
 * Build the calendar bitmap.
 * @param calendar The calendar data.
 */
static void buildcalendar (Calendar *calendar)
{
    int m, /* month counter */
	today, /* today's date */
	d, /* day counter */
	x, /* x coordinate of day */
	y; /* y coordinate of day */
    char header[24], /* calendar month header */
	numstr[3]; /* day number as a string */

    /* create the calendar bitmap */
    if (! calbitmap &&
	! (calbitmap = new_Bitmap (272, 200)))
	fatal_error (FATAL_MEMORY);
    calbitmap->put (calbitmap, calendarbg, 0, 0, CGALIB_SET);

    /* display the month headers */
    for (m = 0; m < 12; ++m) {
	sprintf (header, "%-3.3s M  T  W  T  F  S", monthnames[m]);
	calbitmap->font = calfont;
	calbitmap->print
	    (calbitmap,
	     8 + 88 * (m % 3),
	     12 + 40 * (m / 3) + 16 * (m >= 6),
	     header);
    }

    /* display the day numbers */
    calendar->build (calendar);
    today = calendar->today (calendar);
    for (d = 0; d < calendar->days; ++d) {
	getcalendarcoords (calendar, d, &x, &y);
	sprintf (numstr, "%2d", calendar->mday[d]);
	if (d > today)
	    calbitmap->font = futurefont;
	calbitmap->print (calbitmap, x, y, numstr);
	if (calendar->completed[d]) {
	    calbitmap->put (calbitmap, tickmask, x, y, CGALIB_AND);
	    calbitmap->put (calbitmap, tick, x, y, CGALIB_OR);
	    completed[d] = 1;
	}
    }
}

/**
 * Show the calendar.
 * @param calendar The calendar to show.
 * @param game     The game object.
 */
static void showcalendar (Calendar *calendar, Game *game)
{
    int d, /* day counter */
	x, /* x coordinate of day */
	y; /* y coordinate of day */
    for (d = 0; d < calendar->days; ++d)
	if (calendar->completed[d] && ! completed[d]) {
	    getcalendarcoords (calendar, d, &x, &y);
	    calbitmap->put (calbitmap, tickmask, x, y, CGALIB_AND);
	    calbitmap->put (calbitmap, tick, x, y, CGALIB_OR);
	    completed[d] = calendar->completed[d];
	}
    screen->put (screen, calbitmap, 0, 0, CGALIB_SET);
}

/**
 * Show the calendar cursor.
 * @param calendar The calendar.
 * @param yday     The day of the year.
 */
static void showcalcursor (Calendar *calendar, int yday)
{
    /* where should the cursor go? */
    getcalendarcoords (calendar, yday, &calx, &caly);
    calx -= 4;
    caly -= 2;

    /* store what was behind the cursor */
    if (behindcal)
	behindcal->destroy (behindcal);
    if (! (behindcal = screen->get (screen, calx, caly, 16, 10)))
	fatal_error (FATAL_MEMORY);

    /* display the cursor */
    screen->put (screen, datemask, calx, caly, CGALIB_AND);
    screen->put (screen, datecursor, calx, caly, CGALIB_OR);
}

/**
 * Hide the calendar cursor.
 */
static void hidecalcursor (void)
{
    screen->put (screen, behindcal, calx, caly, CGALIB_SET);
}

/*----------------------------------------------------------------------
 * Constructor Functions.
 */

/**
 * Display constructor.
 * @param colourset = 0 for mono, 1 for colour, 2 for nice colour.
 * @param quiet = 0 for sound and music, 1 for silence.
 * @return the new display.
 */
Display *new_Display (int colourset, int quiet)
{
    /* local variables */
    int mode; /* the screen mode */

    /* allocate memory */
    if (display)
	return display;
    if (! (display = malloc (sizeof (Display))))
	return NULL;

    /* initialise methods */
    display->destroy = destroy;
    display->update = update;
    display->menu = menu;
    display->startdelay = startdelay;
    display->enddelay = enddelay;
    display->showtitlescreen = showtitlescreen;
    display->titlekey = titlekey;
    display->playsound = playsound;
    display->typeset = typeset;
    display->showpanel = showpanel;
    display->showpuzzle = showpuzzle;
    display->showpuzzletiles = showpuzzletiles;
    display->showpuzzletile = showpuzzletile;
    display->showpuzzleguesses = showpuzzleguesses;
    display->showpuzzlescore = showpuzzlescore;
    display->showpuzzlecursor = showpuzzlecursor;
    display->hidepuzzlecursor = hidepuzzlecursor;
    display->buildcalendar = buildcalendar;
    display->showcalendar = showcalendar;
    display->showcalcursor = showcalcursor;
    display->hidecalcursor = hidecalcursor;

    /* initialise class variables */
    controls = getcontrols ();

    /* prepare to initialise the CGALIB screen */
    if (colourset == 0)
	mode = 6;
    else if (colourset == 1)
	mode = 4;
    else if (colourset == 2)
	mode = 5;

    /* initialise the CGALIB screen */
    if (! (screen = new_Screen (mode, CGALIB_SHOWN))) {
	free (display);
	return NULL;
    }
    fatal_display (display);
    if (colourset == 2)
	screen->palette
	    (screen, CGALIB_LIGHT_CYAN_RED_WHITE, CGALIB_GREEN);
    screen->updates = 1;

    /* initialise the assets */
    loadassets ();

    /* set sound */
    gamesound = ! quiet;
    gamesound = 0; /* temporary */

    /* return the screen */
    return display;
}
