/*======================================================================
 * CGALIB version 2
 * Screen hardware handling module - CGA Emulation in X11.
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 18-Jan-2024.
 */

/*----------------------------------------------------------------------
 * Required headers.
 */

/* standard headers */
#include <stdlib.h>
#include <string.h>

/* project headers */
#include "cgalib.h"
#include "screen_u.h"
#include "cga_x11.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var super Screen superclass from data handler. */
static SuperScreen *super = NULL;

/*----------------------------------------------------------------------
 * Level 1 Private Definitions.
 */

/**
 * Draw a display list entry onto the CGA screen.
 * @param screen The screen to draw.
 * @param x      The x coordinate of the area.
 * @param y      The y coordinate of the area.
 * @param w      The width of the area.
 * @param h      The height of the area.
 */
void draw_display_entry
(Screen *screen, int x, int y, int w, int h)
{
    char *s, /* address to copy data from */
	p; /* pixel colour value */
    int r, /* row counter */
	c; /* column counter */

    /* copy the pixels from the back buffer */
    for (r = 0; r < h; ++r)
	for (c = 0; c < w; ++c) {
	    s = screen->buffer->pixels + ((x + c) / 4) + 80 * (y + r);
	    p = (*s >> (2 * (3 - ((x + c) % 4)))) & 3;
	    cgalib_x11_plot (x + c, y + r, p);
	}

    /* force a screen update */
    cgalib_x11_update ();
}

/*----------------------------------------------------------------------
 * Public Method Level Definitions.
 */

/**
 * Destroy the screen and return to text mode.
 * @param screen The screen to destroy.
 */
static void destroy (Screen *screen)
{
    if (super)
	super->destroy (screen);
    if (cgalib_x11_status ())
	cgalib_x11_close ();
    if (! --super->count) {
	free (super);
	super = NULL;
    }
}

/**
 * Show another screen.
 * @param screen The screen to show.
 */
static void show (Screen *screen)
{
    if (super)
	super->show (screen);
}

/**
 * Hide a screen.
 * @param screen The screen to hide.
 */
static void hide (Screen *screen)
{
    if (! screen->shown (screen))
	return;
    if (super)
	super->hide (screen);
}

/**
 * Copy updates from the buffer to the screen.
 * @param screen The screen to update.
 */
static void update (Screen *screen)
{
    DisplayList *entry; /* current display list entry */
    entry = cgalib_get_dl ();
    while (entry) {
	draw_display_entry
	    (screen, entry->x, entry->y,
	     entry->w, entry->h);
	entry = entry->next;
    }
    if (super)
	super->update (screen);
}

/**
 * Set the screen palette.
 * @param screen     The screen for which to set the palette.
 * @param foreground The foreground colour set to choose:
 *                   0: low intensity red, green, brown;
 *                   1: low intensity cyan, magenta, white;
 *                   2: low intensity cyan, red, white;
 *                   3: high intensity red, green, yellow;
 *                   4: high intensity cyan, magenta, white;
 *                   5: high intensity cyan, red, white.
 * @param background The background colour to choose.
 */
static void palette (Screen *screen, int foreground, int background)
{
    if (super)
	super->palette (screen, foreground, background);
    cgalib_x11_palette (foreground, background);
    cgalib_clear_dl ();
    draw_display_entry (screen, 0, 0, 320, 200);
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Construct a new screen.
 * @param  mode    The CGA screen mode: 4, 5, or 6.
 * @param  isshown 1 to show the screen on creation, 0 not to do so.
 * @return         The new screen.
 */
Screen *new_Screen (int mode, int isshown)
{
    Screen *screen; /* the screen to return */

    /* initialise the screen data */
    if (! (screen = cgalib_init (mode, isshown)))
	return NULL;

    /* preserve superclass methods */
    if (super)
	++super->count;
    else {
	if (! (super = malloc (sizeof (SuperScreen)))) {
	    free (screen);
	    return NULL;
	}
	super->count = 1;
	super->destroy = screen->destroy;
	super->show = screen->show;
	super->hide = screen->hide;
	super->update = screen->update;
	super->palette = screen->palette;
    }

    /* override methods */
    screen->destroy = destroy;
    screen->show = show;
    screen->hide = hide;
    screen->update = update;
    screen->palette = palette;

    /* initialise X and set up the screen */
    if (isshown)
	if (! cgalib_x11_status ())
	    cgalib_x11_init ();

    /* return the new screen */
    return screen;
}
