/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 19-Aug-2024.
 *
 * Extract a year's worth of puzzles from the word square source file.
 */

/*----------------------------------------------------------------------
 * Required headers
 */

/* standard C headers */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var year The year to extract. */
static int year;

/** @var indexcount Number of indexes for the word squares. */
static long int indexcount;

/** @var indexes Character indexes for the word squares. */
static long int *indexes;

/** @var days The number of days in the current year. */
static int days;

/** @var line The puzzle number of the first/current puzzle. */
static int puzzle;

/*----------------------------------------------------------------------
 * Level 2 Function Definitions.
 */

/**
 * Pseudorandom number generator.
 * @return The next pseudorandom number.
 */
static long int pseudorandom (void)
{
    static unsigned long int next = 1; /* seed */
    unsigned long int r; /* random number to return */
    int c; /* counter */
    r = 0;
    for (c = 0; c < 2; ++c) {
	next = next * 1103515245 + 12345;
	r = (r << 16) + ((unsigned int) (next / 65536) % 0x10000);
    }
    return r;
}

/*----------------------------------------------------------------------
 * Level 1 Function Definitions.
 */

/**
 * Check the arguments and extract program parameters.
 * @param argc argument count.
 * @param argv argument values.
 */
static void checkargs (int argc, char **argv)
{
    if (argc != 2) {
	printf ("Usage: %s YEAR\n", argv[0]);
	exit (1);
    } else if (! (year = atoi (argv[1]))) {
	printf ("Year must be a number\n");
	exit (2);
    } else if (year < 2024) {
	printf ("Year must be 2024 onwards\n");
	exit (3);
    }
}

/**
 * Load in the word square indexes.
 */
static void loadsquares (void)
{
    FILE *input; /* input file */
    char buffer[32]; /* input buffer */
    long int
	pos = 0, /* file position before reading */
	count = 0; /* line count */

    /* count the word squares */
    if (! (input = fopen ("assets/squares.txt", "r"))) {
	printf ("Cannot open assets/squares.txt\n");
	exit (4);
    }
    while (fgets (buffer, 31, input))
	++indexcount;
    fclose (input);

    /* prepare the index array */
    if (! (indexes = malloc (indexcount * sizeof (long)))) {
	printf ("Out of memory (%ld squares)\n", indexcount);
	exit (5);
    }

    /* read the indexes from the file */
    input = fopen ("assets/squares.txt", "r");
    while (fgets (buffer, 31, input)
	   && count < indexcount)
    {
	indexes[count++] = pos;
	pos = ftell (input);
    }
    fclose (input);
}

/**
 * Scramble the word square indexes.
 */
static void scramblesquares (void)
{
    long int
	count, /* index counter */
	alternate, /* alternate index to swap with */
	swap; /* temporary swap value */
    for (count = 0; count < indexcount; ++count) {
	alternate = pseudorandom () % indexcount;
	swap = indexes[count];
	indexes[count] = indexes[alternate];
	indexes[alternate] = swap;
    }
}

/**
 * Calculate the parameters for the current eyar.
 */
static void calcyear (void)
{
    int count; /* year counter */
    for (count = 2024; count <= year; ++count) {
	days
	    = 365
	    + ((count % 4 == 0 && count % 100 != 0)
	       || (count % 400 == 0));
	if (count < year)
	    puzzle += days;
    }
}

/**
 * Extract the word squares and output them.
 */
static void extractsquares (void)
{
    FILE *input; /* the input file */
    int d; /* day counter */
    char text[256]; /* text of the puzzle */
    input = fopen ("assets/squares.txt", "r");
    for (d = 0; d < days; ++d) {
	fseek (input, indexes[puzzle++], SEEK_SET);
	fgets (text, 255, input);
	printf ("%s", text);
    }
    fclose (input);
}

/*----------------------------------------------------------------------
 * Top Level Function Definition.
 */

/**
 * Main Function.
 * @param argc argument count.
 * @param argv argument values.
 * @return     0 if successful, !0 on failure.
 */
int main (int argc, char **argv)
{
    checkargs (argc, argv);
    loadsquares ();
    scramblesquares ();
    calcyear ();
    extractsquares ();
    return 0;
}
