/*======================================================================
 * Pym's Daily Word Square Puzzle.
 * A puzzle game for MS-DOS.
 *
 * Copyright (C) Damian Gareth Walker 2024. Released under the GNU GPL.
 * Created: 17-Aug-2024.
 *
 * Asset Generator Module.
 */

/*----------------------------------------------------------------------
 * Headers
 */

/* Standard ANSI C */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Project Headers */
#include "cgalib.h"
#include "fatal.h"

/*----------------------------------------------------------------------
 * File Level Variables.
 */

/** @var output is the output file handle */
static FILE *output;

/** @var scr is the screen */
static Screen *screens[5];

/*----------------------------------------------------------------------
 * Level 2 Functions.
 */

/*
 * Extract a bitmap from a sprite sheet and save it.
 * @param i The sprite sheet id.
 * @param x X position of sprite on sheet.
 * @param y Y position of sprite on sheet.
 * @param w Width of sprite.
 * @param h Height of sprite.
 */
static void save_bitmap (int i, int x, int y, int w, int h)
{
    Bitmap *bit; /* logo bitmap */
    if (! (bit = screens[i]->get (screens[i], x, y, w, h)))
	fatal_error (FATAL_MEMORY);
    if (! bit->write (bit, output))
	fatal_error (FATAL_INVALIDDATA);
    bit->destroy (bit);
}

/**
 * Extract a tile font from one of the sprite sheets, and save it.
 * @param i The sprite sheet id.
 * @param h Which half of the sprite sheet, 0=upper, 1=lower.
 */
static void copy_tile_font (int i, int h)
{
    Font *fnt; /* calendar font */
    Bitmap *bit; /* temporary character bitmap */
    int c; /* character counter */
    if (! (fnt = new_Font (65, 92, 32, 32)))
	fatal_error (FATAL_MEMORY);
    for (c = 0; c <= 25; ++c) {
	if (! (bit = screens[i]->get
	       (screens[i],
		32 * (c % 10),
		96 * h + 32 * (c / 10),
		32, 32)))
	    fatal_error (FATAL_MEMORY);
	fnt->put (fnt, bit, 65 + c);
	bit->destroy (bit);
    }
    fnt->write (fnt, output);
    fnt->destroy (fnt);
}

/*----------------------------------------------------------------------
 * Level 1 Functions.
 */

/**
 * Open a new DAT file.
 */
static void open_dat_file (void)
{
    if (! (output = fopen("pym/PYM.DAT", "wb")))
	fatal_error (FATAL_NODATA);
    fwrite ("PYM001D", 8, 1, output);
}

/**
 * Initialise the screen.
 */
static void initialise_screens (void)
{
    int i; /* screen index */
    for (i = 0; i < 5; ++i) {
	screens[i] = new_Screen (CGALIB_MEDIUM_MONO, CGALIB_HIDDEN);
	if (! screens[i])
	    fatal_error (FATAL_DISPLAY);
    }
}

/**
 * Load the assets from a PIC file.
 */
static void load_screens (void)
{
    FILE *input; /* screen input file */
    char filename[128]; /* filename */
    int i; /* screen index */
    for (i = 0; i < 5; ++i) {
	sprintf (filename, "assets/PYM-%d.PIC", i);
	if (! (input = fopen (filename, "rb")))
	    fatal_error (FATAL_NODATA);
	if (! screens[i]->read (screens[i], CGALIB_SCREEN_FILE, input))
	    fatal_error (FATAL_DISPLAY);
	fclose (input);
    }
}

/**
 * Copy the logo bitmap.
 */
static void copy_logo_bitmap (void)
{
    save_bitmap (2, 8, 8, 128, 16);
}

/**
 * Copy the title bitmap.
 */
static void copy_title_bitmap (void)
{
    save_bitmap (0, 0, 0, 320, 200);
}

/**
 * Copy the background bitmaps.
 */
static void copy_background_bitmaps (void)
{
    Bitmap *bit; /* calendar bitmap */
    if (! (bit = screens[2]->get (screens[2], 0, 0, 272, 200)))
	fatal_error (FATAL_MEMORY);
    bit->box (bit, 8, 8, 128, 36, CGALIB_BOX_FILLED);
    if (! bit->write (bit, output))
	fatal_error (FATAL_WRITE);
    bit->destroy (bit);
    save_bitmap (1, 0, 0, 272, 200);
    save_bitmap (1, 272, 0, 48, 200);
}

/**
 * Copy miscellaneous bitmaps.
 */
static void copy_misc_bitmaps (void)
{
    /* menu pull bitmap and mask */
    save_bitmap (1, 272, 184, 48, 16);
    save_bitmap (2, 272, 184, 48, 16);

    /* calendar cursor bitmap and mask */
    save_bitmap (2, 272, 164, 16, 10);
    save_bitmap (2, 272, 174, 16, 10);

    /* calendar tick bitmap and mask */
    save_bitmap (2, 272, 158, 8, 6);
    save_bitmap (2, 280, 158, 8, 6);

    /* puzzle cursor bitmap and mask */
    save_bitmap (2, 288, 120, 32, 32);
    save_bitmap (2, 288, 152, 32, 32);
}

/**
 * Extract the calendar font and save it.
 */
static void copy_calendar_font (void)
{
    Font *fnt; /* calendar font */
    Bitmap *bit; /* temporary character bitmap */
    int c; /* character counter */
    if (! (fnt = new_Font (32, 95, 4, 6)))
	fatal_error (FATAL_MEMORY);
    for (c = 32; c <= 95; ++c) {
	if (! (bit = screens[2]->get
	       (screens[2],
		8 + 4 * (c & 0x1f),
		32 + 6 * (((c & 0x20) ^ 0x20) / 0x20),
		4, 6)))
	    fatal_error (FATAL_MEMORY);
	fnt->put (fnt, bit, c);
	bit->destroy (bit);
    }
    fnt->write (fnt, output);
    fnt->destroy (fnt);
}

/**
 * Copy the standard "Present" font into the asset file.
 */
static void copy_standard_font (void)
{
    FILE *input; /* input file handle */
    char header[8]; /* header read from input file */
    Font *fnt; /* font read from input file */

    /* open input file, read and verify header */
    if (! (input = fopen ("cgalib2/cgalib/fnt/present.fnt", "rb")))
	fatal_error (FATAL_NODATA);
    if (! (fread (header, 8, 1, input)))
	fatal_error (FATAL_NODATA);
    if (strcmp (header, "CGA100F") && strcmp (header, "CGA200F"))
	fatal_error (FATAL_NODATA);

    /* read the font and copy it to the output file */
    if (! (fnt = read_Font (input, header[3] - '0')))
	fatal_error (FATAL_NODATA);
    fclose (input);
    fnt->write (fnt, output);
    fnt->destroy (fnt);
}

/**
 * Extract the strikethrough font and save it.
 */
static void copy_strikethrough_font (void)
{
    Font *fnt; /* calendar font */
    Bitmap *bit; /* temporary character bitmap */
    int c; /* character counter */
    if (! (fnt = new_Font (65, 92, 4, 8)))
	fatal_error (FATAL_MEMORY);
    for (c = 65; c <= 92; ++c) {
	if (! (bit = screens[2]->get
	       (screens[2], 8 + 4 * (c & 0x1f), 24, 4, 8)))
	    fatal_error (FATAL_MEMORY);
	fnt->put (fnt, bit, c);
	bit->destroy (bit);
    }
    fnt->write (fnt, output);
    fnt->destroy (fnt);
}

/**
 * Extract the tile fonts and write to the asset file.
 */
static void copy_tile_fonts (void)
{
    copy_tile_font (4, 0); /* clue letters */
    copy_tile_font (3, 1); /* guess letters */
    copy_tile_font (3, 0); /* correct letters */
}

/**
 * Close the screen.
 */
static void close_screens (void)
{
    int i; /* screen index */
    for (i = 0; i < 5; ++i)
	screens[i]->destroy (screens[i]);
    fatal_screen (NULL);
}

/**
 * Add music and sound effects to the asset files.
 */
static void add_sounds (void)
{
}

/**
 * Close the DAT file.
 */
static void close_dat_file (void)
{
    fclose (output);
}

/*----------------------------------------------------------------------
 * Top Level Function.
 */

/**
 * Main Program.
 * @returns 0 on a successful run.
 */
int main (void)
{
    open_dat_file ();
    initialise_screens ();
    load_screens ();
    copy_logo_bitmap ();
    copy_title_bitmap ();
    copy_background_bitmaps ();
    copy_misc_bitmaps ();
    copy_calendar_font ();
    copy_standard_font ();
    copy_strikethrough_font ();
    copy_tile_fonts ();
    close_screens ();
    add_sounds ();
    close_dat_file ();
    printf ("PYM\\PYM.DAT generated successfully.\n");
    return 0;
}
