/*======================================================================
 * CGALIB version 2
 * Screen hardware handling module - CGA and Hercules in DOS.
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 06-Dec-2022.
 */

/*----------------------------------------------------------------------
 * Required headers.
 */

/* standard headers */
#include <stdlib.h>
#include <string.h>

/* compiler headers */
#include <conio.h>
#include <dos.h>

/* project headers */
#include "cgalib.h"
#include "screen_u.h"

/*----------------------------------------------------------------------
 * Data Definitions.
 */

/** @var super Screen superclass from data handler. */
static SuperScreen *super = NULL;

/** @var base Base address of graphics output. */
static char far *base;

/** @var swidth Physical screen width in bytes. */
static int swidth;

/** @var interleave Number of rows interleaved. */
static int interleave;

/* @var ymult Y coordinate multiplier for calculating pixel address. */
static int ymult;

/* @var ydiv Y coordinate divisor for calculating pixel address. */
static int ydiv;

/*----------------------------------------------------------------------
 * Level 2 Functions.
 */

/**
 * Set the screen parameters according to the screen mode. For some
 * reason, if this is done within the set{,_hercules)_mode functions,
 * the values get reset on leaving. So they're here in a separate
 * function, to be called immediately before trying to update the
 * screen if not already set. That ensures they'll be available at the
 * point they are needed.
 * @param screen The screen data.
 */
void set_screen_parameters (Screen *screen)
{
    /* Hercules screen parameters */
    if (screen->mode == 7) {
	base = (char far *) 0xb0000221;
	swidth = 90;
	interleave = 4;
	ymult = 3;
	ydiv = 2;
    }

    /* CGA/EGA/VGA screen parameters */
    else {
	base = (char far *) 0xb8000000;
	swidth = 80;
	interleave = 2;
	ymult = 1;
	ydiv = 1;
    }
}

/**
 * Set hercules mode.
 */
static void set_hercules_mode (void)
{
    int i; /* index counter */
    char params[12] /* register values */
	= {0x35, 0x2d, 0x2e, 0x07, 0x5b, 0x02,
	   0x57, 0x57, 0x02, 0x03, 0x00, 0x00};

    /* set the mode and deactivate the screen */
    outp (0x3bf, 3);
    outp (0x3b8, 0xa /* 2 */);

    /* set the registers */
    for (i = 0; i < 12; ++i) {
	outp (0x3b4, i);
	outp (0x3b5, params[i]);
    }

    /* clear the screen */
    _fmemset ((char far *) 0xb0000000, 0, 0x7fff);

    /* activate the screen */
    outp (0x3b8, 0xa);
}

/*----------------------------------------------------------------------
 * Level 1 Functions.
 */

/**
 * Attempt to detect Hercules Graphics Card.
 * @return 1 if Hercules, 0 if not.
 */
static int is_hercules (void)
{
    union REGS regs;
    int86 (0x11, &regs, &regs);
    return ((regs.x.ax & 0x30) == 0x30);
}

/**
 * Set the display mode.
 * @params mode is the mode to select.
 */
static void set_mode (int mode)
{
    /* local variables */
    union REGS regs;

    /* set hercules mode */
    if (mode == 7)
	set_hercules_mode ();

    /* set bios-compatible modes */
    else {

	/* use the bios to select the mode */
	regs.h.ah = 0;
	regs.h.al = mode;
	int86 (0x10, &regs, &regs);

    }
}

/**
 * Sets the desired screen colours using the CGA palette register.
 * This works only on actual CGA cards. These palette registers are
 * ignored by EGA and VGA cards, which have independent control of the
 * two or four screen colours.
 * @param screen The screen to change.
 */
static void palette_cga (Screen *screen)
{
    /* registers */
    char mode_control;
    char colour_control;

    /* don't do any of this if we're in monochrome mode
       or showing another screen */
    if (! screen->shown (screen)
	|| screen->mode == CGALIB_HIGH
	|| screen->mode == CGALIB_HERCULES)
	return;

    /* set the mode according to the palette chosen */
    screen->mode = ((screen->foreground % 3) == 2)
	? CGALIB_MEDIUM_MONO
	: CGALIB_MEDIUM;

    /* initialise the registers */
    mode_control = 0;
    colour_control = 0;

    /* set the mode control register */
    mode_control |= 0x8; /* show display */
    mode_control |= (screen->mode == CGALIB_MEDIUM_MONO)
	? 0x4 /* greyscale */
	: 0; /* colour */
    mode_control |= 0x2; /* graphics mode */

    /* set the colour control register */
    colour_control |= (screen->foreground % 3) ? 0x20 : 0; /* palette */
    colour_control |= (screen->foreground / 3) ? 0x10 : 0; /* intensity */
    colour_control |= screen->background; /* colour choice */

    /* set the CGA registers */
    outp (0x3d8, mode_control);
    outp (0x3d9, colour_control);
}

/**
 * Sets the screen colours using the EGA palette registers.
 * Because EGA and VGA cards ignore the CGA palette registers, we need
 * to address these cards directly if we want to see the same colour
 * palette on these cards as we do on a CGA display. This function just
 * simulates the available CGA palettes, it does not support the actual
 * flexibility of a 4-colour EGA/VGA palette.
 * @param screen The screen to change.
 */
static void palette_ega (Screen *screen)
{
    union REGS regs;
    static int background[16] = { /* background colours */
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f
    };
    static int foreground[6][3] = { /* table of 320x200 mode colours */
        {0x02, 0x04, 0x06}, /* mode 4 palette 0 bright 0 */
        {0x03, 0x05, 0x07}, /* mode 4 palette 1 bright 0 */
        {0x03, 0x04, 0x07}, /* mode 5 palette 2 bright 0 */
        {0x3a, 0x3c, 0x3e}, /* mode 4 palette 0 bright 1 */
        {0x3b, 0x3d, 0x3f}, /* mode 4 palette 1 bright 1 */
        {0x3b, 0x3c, 0x3f}, /* mode 5 palette 2 bright 1 */
    };
    int fgcount; /* count of foreground colours */

    /* don't do any of this if we're in monochrome mode
       or showing another screen */
    if (! screen->shown (screen) || screen->mode == 6)
	return;

    /* background colour */
    regs.w.ax = 0x1000;
    regs.w.bx = 0x100 * background[screen->background];
    int86 (0x10, &regs, &regs);

    /* foreground colours */
    for (fgcount = 0; fgcount <= 2; ++fgcount) {
        regs.w.ax = 0x1000;
        regs.w.bx = 0x100 * foreground[screen->foreground][fgcount]
            + fgcount + 1;
        int86 (0x10, &regs, &regs);
    }
}

/**
 * Draw a display list entry onto the CGA screen.
 * @param screen The screen to draw.
 * @param x      The x coordinate of the area.
 * @param y      The y coordinate of the area.
 * @param w      The width of the area.
 * @param h      The height of the area.
 */
void draw_display_entry (Screen *screen, int x, int y,
				int w, int h)
{
    char far *d; /* address to copy data to */
    char *s; /* address to copy data from */
    int r, /* row counter */
	yd; /* destination y coordinate */
    if (! ydiv)
	set_screen_parameters (screen);
    for (r = 0; r < h; ++r) {
	yd = (y + r) * ymult / ydiv;
	d = x / 4
	    + 0x2000 * (yd % interleave)
	    + swidth * (yd / interleave)
	    + base;
	s = screen->buffer->pixels + (x / 4) + 80 * (y + r);
	_fmemcpy (d, s, w / 4);
    }
}

/*----------------------------------------------------------------------
 * Public Method Level Definitions.
 */

/**
 * Destroy the screen and return to text mode.
 * @param screen The screen to destroy.
 */
static void destroy (Screen *screen)
{
    if (super)
	super->destroy (screen);
    if (! --super->count) {
	free (super);
	super = NULL;
    }
    set_mode (3);
}

/**
 * Show another screen.
 * @param screen The screen to show.
 */
static void show (Screen *screen)
{
    if (super)
	super->show (screen);
    set_mode (screen->mode);
    screen->palette (screen, screen->foreground, screen->background);
    draw_display_entry (screen, 0, 0, 320, 200);
}

/**
 * Hide a screen.
 * @param screen The screen to hide.
 */
static void hide (Screen *screen)
{
    if (! screen->shown (screen))
	return;
    if (super)
	super->hide (screen);
    set_mode (3);
}

/**
 * Copy updates from the buffer to the screen.
 * @param screen The screen to update.
 */
static void update (Screen *screen)
{
    DisplayList *entry; /* current display list entry */
    entry = cgalib_get_dl ();
    while (entry) {
	draw_display_entry
	    (screen, entry->x, entry->y,
	     entry->w, entry->h);
	entry = entry->next;
    }
    if (super)
	super->update (screen);
}

/**
 * Set the screen palette.
 * @param screen     The screen for which to set the palette.
 * @param foreground The foreground colour set to choose:
 *                   0: low intensity red, green, brown;
 *                   1: low intensity cyan, magenta, white;
 *                   2: low intensity cyan, red, white;
 *                   3: high intensity red, green, yellow;
 *                   4: high intensity cyan, magenta, white;
 *                   5: high intensity cyan, red, white.
 * @param background The background colour to choose.
 */
static void palette (Screen *screen, int foreground, int background)
{
    if (super)
	super->palette (screen, foreground, background);
    palette_cga (screen);
    palette_ega (screen);
}

/*----------------------------------------------------------------------
 * Top Level Function Definitions.
 */

/**
 * Construct a new screen.
 * @param  mode    The CGA screen mode: 4, 5, or 6.
 * @param  isshown 1 to show the screen on creation, 0 not to do so.
 * @return         The new screen.
 */
Screen *new_Screen (int mode, int isshown)
{
    Screen *screen; /* the screen to return */

    /* initialise the screen data */
    if (! (screen = cgalib_init (mode, isshown)))
	return NULL;

    /* preserve superclass methods */
    if (super)
	++super->count;
    else {
	if (! (super = malloc (sizeof (SuperScreen)))) {
	    free (screen);
	    return NULL;
	}
	super->count = 1;
	super->destroy = screen->destroy;
	super->show = screen->show;
	super->hide = screen->hide;
	super->update = screen->update;
	super->palette = screen->palette;
    }

    /* override methods */
    screen->destroy = destroy;
    screen->show = show;
    screen->hide = hide;
    screen->update = update;
    screen->palette = palette;

    /* override attributes */
    screen->mode = is_hercules () ? 7 : mode;

    /* set the screen mode and palette */
    if (isshown) {
	set_mode (screen->mode);
	palette_cga (screen);
	palette_ega (screen);
    }

    /* return the new screen */
    return screen;
}
