/*======================================================================
 * CGALIB version 2
 * Screen handling header.
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 06-Dec-2022.
 */

/* types defined in this header */
typedef struct screen Screen;

#ifndef __SCREEN_H__
#define __SCREEN_H__

/*----------------------------------------------------------------------
 * Required Headers.
 */

/* standard headers */
#include <stdio.h>

/* project headers */
#include "bitmap.h"
#include "font.h"

/*----------------------------------------------------------------------
 * Data Declarations.
 */

/**
 * @struct screen
 * Object for screen handling.
 */
struct screen {

    /*
     * Attributes
     */

    /** @var mode The video mode */
    int mode;

    /** @var foreground The palette foreground colour set */
    int foreground;

    /** @var background The palette background colour */
    int background;

    /** @var ink The ink colour for writing and drawing */
    int ink;

    /** @var paper The paper colour for writing and drawing */
    int paper;

    /** @var font The current font */
    Font *font;

    /** @var buffer The off-screen buffer. */
    Bitmap *buffer;

    /**
     * @var updates 
     * The maximum number of display list updates to queue.
     * If the number of updates queued exceeds this, all updates
     * will be aggregated into one. If 0, there is no queue.
     */
    int updates;

    /*
     * Methods
     */

    /**
     * Destroy the screen and return to text mode.
     * @param screen The screen to destroy.
     */
    void (*destroy) (Screen *screen);

    /**
     * Write the screen buffer to an already open file.
     * @param  screen The screen to write.
     * @param  format The file format:
     *                CGALIB_BITMAP is in bitmap format.
     *                CGALIB_BSAVE is in BSAVE format.
     * @param  output The output file handle.
     * @return        1 on success, 0 on failure.
     */
    int (*write) (Screen *screen, int format, FILE *output);

    /**
     * Read the screen buffer from an already open file.
     * @param  screen The screen to read.
     * @param  format The file format:
     *                CGALIB_BITMAP is in bitmap format.
     *                CGALIB_BSAVE is in BSAVE format.
     * @param  input  The output file handle.
     * @return        1 on success, 0 on failure.
     */
    int (*read) (Screen *screen, int format, FILE *input);

    /**
     * Show a screen.
     * @param screen The screen to show.
     */
    void (*show) (Screen *screen);

    /**
     * Hide a screen.
     * @param screen The screen to hide.
     */
    void (*hide) (Screen *screen);

    /**
     * Determine if a screen is shown.
     * @param screen The screen to check.
     */
    int (*shown) (Screen *screen);

    /**
     * Copy updates from the buffer to the screen.
     * @param screen The screen to update.
     */
    void (*update) (Screen *screen);

    /**
     * Set the screen palette.
     * @param screen     The screen for which to set the palette.
     * @param foreground The foreground colour set to choose:
     *                   0: low intensity red, green, brown;
     *                   1: low intensity cyan, magenta, white;
     *                   2: low intensity cyan, red, white;
     *                   3: high intensity red, green, yellow;
     *                   4: high intensity cyan, magenta, white;
     *                   5: high intensity cyan, red, white.
     * @param background The background colour to choose.
     */
    void (*palette) (Screen *screen, int foreground, int background);

    /**
     * Get a bitmap from another bitmap.
     * @param  screen The screen to extract from.
     * @param  x      The x coordinate on the source bitmap.
     * @param  y      The y coordinate on the destination bitmap.
     * @param  w      The width of the bitmap to exstract.
     * @param  h      The height of the bitmap to extract.
     * @return        The new bitmap.
     */
    Bitmap *(*get) (Screen *screen, int x, int y, int w, int h);

    /**
     * Put a bitmap on the screen.
     * @param  screen The destination screen.
     * @param  bitmap The source bitmap.
     * @param  x      The x coordinate on the destination bitmap.
     * @param  y      The y coordinate on the destination bitmap.
     * @param  mode   The draw mode.
     */
    void (*put) (Screen *screen, Bitmap *source, int x, int y, int mode);

    /**
     * Put part of another bitmap onto this one.
     * @param screen The destination screen.
     * @param bitmap The source bitmap.
     * @param xd     The x coordinate on the destination bitmap.
     * @param yd     The y coordinate on the destination bitmap.
     * @param xs     The x coordinate on the source bitmap.
     * @param ys     The y coordinate on the source bitmap.
     * @param w      The width of the area to copy.
     * @param h      The height of the area to copy.
     * @param mode   The draw mode.
     */
    void (*transfer) (Screen *screen, Bitmap *bitmap, int xd, int yd,
		      int xs, int ys, int w, int h, int draw);

    /**
     * Draw a filled box onto the screen.
     * @param screen  The screen to draw onto.
     * @param x       The x coordinate of the left edge of the box.
     * @param y       The y coordinate of the top edit of the box.
     * @param width   The width of the box.
     * @param height  The height of the box.
     * @param pattern The 16-bit pattern for the box.
     */
    void (*box) (Screen *screen, int x, int y, int width, int height,
		 unsigned int pattern);

    /**
     * Print a message onto the screen.
     * @param screen  The screen to print onto.
     * @param x       The x coordinate of the message.
     * @param y       The y coordinate of the message.
     * @param message The message to print.
     */
    void (*print) (Screen *screen, int x, int y, char *message);

};

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Construct a new screen.
 * @param  mode The CGA screen mode: 4, 5, or 6.
 * @param  show 1 to show the screen on creation, 0 not to do so.
 * @return      The new screen.
 */
Screen *new_Screen (int mode, int show);

/**
 * Construct a new screen from an already open file.
 * @param  mode   The CGA screen mode: 4, 5, or 6.
 * @param  show   1 to show the screen on creation, 0 not to do so.
 * @param  format The file format:
 *                CGALIB_BITMAP is in bitmap format.
 *                CGALIB_BSAVE is in BSAVE format.
 * @param  input  The input file handle.
 */
Screen *read_Screen (int mode, int show, int format, FILE *input);

#endif
