/*======================================================================
 * CGALIB version 2
 * Font handling header.
 *
 * Released as Public Domain by Damian Gareth Walker, 2022.
 * Created 06-Dec-2022.
 */

/* types defined in this header */
typedef struct font Font;

#ifndef __FONT_H__
#define __FONT_H__

/*----------------------------------------------------------------------
 * Required Headers.
 */

/* standard headers */
#include <stdio.h>

/*----------------------------------------------------------------------
 * Data Declarations.
 */

/**
 * @struct font
 * Object for font handling.
 */
struct font {

    /*
     * Attributes
     */

    /** @var first The first character in a font. */
    int first;

    /** @var lst The last character in a font. */
    int last;

    /** @var width The font width. */
    int width;

    /** @var height The font height. */
    int height;

    /** @var pixels The pixel data for each character. */
    char *pixels;

    /*
     * Methods
     */

    /**
     * Destroy a font when it is no longer needed.
     * @param font The font to destroy.
     */
    void (*destroy) (Font *font);

    /**
     * Clone a font.
     * @param  font The font to clone.
     * @return      The new font.
     */
    Font *(*clone) (Font *font);

    /**
     * Write a font to an already open file.
     * @param  font   The font to write.
     * @param  output The output file handle.
     * @return        1 on success, 0 on failure.
     */
    int (*write) (Font *font, FILE *output);

    /**
     * Put a character into the font.
     * @param  font   The font to modify.
     * @param  bitmap The bitmap containing the character.
     * @param  code   The character code to modify.
     */
    void (*put) (Font *font, Bitmap *bitmap, int code);

    /**
     * Get a character from the font.
     * @param  font The font to extract from.
     * @param  code The character code to extract.
     * @return      A bitmap resembling the character.
     */
    Bitmap *(*get) (Font *font, int code);

    /**
     * Recolour a whole font permanently.  This is faster than
     * printing the standard font with alternative ink and paper
     * colours. The font must be in colours 3 ink and 0 paper before
     * the conversion.
     * @param  font  The font to recolour.
     * @param  ink   The new ink colour.
     * @param  paper The new paper colour.
     */
    void (*recolour) (Font *font, int ink, int paper);

};

/*----------------------------------------------------------------------
 * Top Level Function Declarations.
 */

/**
 * Cronstruct a new font.
 * @param  first  The first character in the font.
 * @param  last   The last character in the font.
 * @param  width  The width of the font.
 * @param  height The height of the font.
 * @return        The new font.
 */
Font *new_Font (int first, int last, int width, int height);

/**
 * Construct a new font by reading it from an already open file.
 * @param  input   The input file handle.
 * @param  version CGALIB version of the font file.
 * @return         The loaded font.
 */
Font *read_Font (FILE *input, int version);

#endif
